/*!
	Copyright 2010 British Broadcasting Corporation

	Licensed under the Apache License, Version 2.0 (the "License");
	you may not use this file except in compliance with the License.
	You may obtain a copy of the License at

	   http://www.apache.org/licenses/LICENSE-2.0

	Unless required by applicable law or agreed to in writing, software
	distributed under the License is distributed on an "AS IS" BASIS,
	WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
	See the License for the specific language governing permissions and
	limitations under the License.
*/
Glow.provide(function(glow) {
	var NodeList = glow.NodeList,
		NodeListProto = NodeList.prototype,
		undefined;
	
	/**
		@name glow.NodeList.focusable
		@function
		@extends glow.ui.Behaviour
		@description Manage a focusable element, or group of elements
			This method is a shortcut to {@link glow.ui.Focusable} and requires
			the 'ui' package to be loaded.
			
			The first item in the NodeList is treated as the focusable's container.
			An error is thrown if the first item in the NodeList is not an element.
		
			This can be used to create a single focus point for a set
			of focusable elements. Eg, a menu can have a single tab stop,
			and the arrow keys can be used to cycle through menu items.
			
			This means the user doesn't have to tab through every item in the
			menu to get to the next set of focusable items.
			
			The FocusManager can also be used to make a element 'modal', ensuring
			focus doesn't go to elements outside it.
		
		@param {object} [opts] Options
			The same options as the {@link glow.ui.Focusable} constructor
			
		@returns {glow.ui.Focusable}
	*/
	NodeListProto.focusable = function(opts) {
		/*!debug*/
			if (arguments.length > 1) {
				glow.debug.warn('[wrong count] glow.NodeList#focusable expects 0 or 1 argument, not ' + arguments.length + '.');
			}
			if (opts !== undefined && typeof opts !== 'object') {
				glow.debug.warn('[wrong type] glow.NodeList#focusable expects object as "opts" argument, not ' + typeof opts + '.');
			}
		/*gubed!*/
		return new glow.ui.Focusable(this, opts);
	};
});
// start-source: ui.js

/**
	@name glow.ui
	@namespace
 */
		 
Glow.provide(function(glow) {
	glow.ui = glow.ui || {};
});

// end-source: ui.js
Glow.provide(function(glow) {
	/**
		@name glow.ui.Behaviour
		@class
		@extends glow.events.Target
		@description Abstract behaviour class.
		@param {string} name The name of this widget.
		This is added to class names in the generated DOM nodes that wrap the widget interface.
		
	*/
	function Behaviour() {}
	glow.util.extend(Behaviour, glow.events.Target);
	
	/*!debug*/
		/**
			@name glow.ui.Behaviour#enabled
			@function
			@description Get/set the enabled state
				
			@param {boolean} [state=true] 
		*/
		Behaviour.prototype.enabled = function() {
			throw new Error('#enabled not implemented on behaviour');
		}
		
		/**
			@name glow.ui.Behaviour#destroy
			@function
			@description Removes the behaviour & event listeners
		*/
		Behaviour.prototype.destroy = function() {
			throw new Error('#destroy not implemented on behaviour');
		}
		
	/*gubed!*/
	
	// EXPORT
	glow.ui.Behaviour = Behaviour;
});
Glow.provide(function(glow) {
	var undefined, FocusableProto,
		// array of focusable instances
		focusables = [],
		// the focused element
		focused,
		// we use this to track the modal focusable, also to ensure there's only one
		modalFocusable,
		documentNodeList = glow(document),
		ignoreFocus = false;
	
	// keep track of what element has focus
	documentNodeList.on('blur', function(event) {
		focused = undefined;
		if (focusables.length) {
			// activate focusables on a timeout so we pick up a possible subsequent
			// focus event
			setTimeout(deactivateAllIfBlurred, 0);
		}
	}).on('focus', function(event) {
		if ( modalFocusable && !modalFocusable.container.contains(event.source) ) {
			// refocus either the active child or container
			( modalFocusable.activeChild[0] || modalFocusable.container[0] ).focus();
			return false;
		}
		
		focused = event.source;
		
		if (ignoreFocus) {
			return;
		}
		
		ignoreFocus = true;
		
		activateFocusables();
		
		setTimeout(stopIgnoringFocus, 0);
	});
	
	/**
		@private
		@function
		@description Wot it sez on da tin.
			(used to cater for browsers that fire multiple focuses per click)
	*/
	function stopIgnoringFocus() {
		ignoreFocus = false;
	}
	
	/**
		@private
		@function
		@description Deactivate all our focusables if nothing has focus
	*/
	function deactivateAllIfBlurred() {
		// if nothing has focus, deactivate our focusables
		!focused &&	activateFocusables();
	}
	
	/**
		@private
		@function
		@description React to a change in focus
	*/
	function activateFocusables() {
		// taking a copy of the array in case any destroy
		var instances = focusables.slice(0),
			i = instances.length;
		
		while (i--) {
			// activate / deactivate the focusable depending on where focus is.
			// This calls active(), passing in either the element focused (within the Focusable container) or false.
			// The 2 mentions of 'focused' is deliberate.
			instances[i].active( (focused && instances[i].container.contains(focused) && focused) || false );
		}
	}
	
	/**
		@private
		@function
		@description Update the children property for a focusable
	*/
	function updateChildren(focusable) {
		focusable.children = focusable.container.get( focusable._opts.children );
		
		// remove focusable items from the tab flow, we're going to conrol this with tab keys
		glow(focusable.children).push(focusable.container).prop('tabIndex', -1);
	}
	
	/**
		@private
		@function
		@description Create the default key handler functions
	*/
	function createKeyHandler(useLeftRight, useUpDown) {
		return function(event) {
			// listen for keypresses, react, and return false if the key was used
			switch (event.key) {
				case 'up':
					return !( useUpDown    && this.prev() );
				case 'left':
					return !( useLeftRight && this.prev() );
				case 'down':
					return !( useUpDown    && this.next() );
				case 'right':
					return !( useLeftRight && this.next() );
			}
		}
	}
	
	/**
		@private
		@description The default key handler functions
	*/
	var keyHandlers = {
		'arrows'  : createKeyHandler(1, 1),
		'arrows-x': createKeyHandler(1, 0),
		'arrows-y': createKeyHandler(0, 1)
	}
	
	/**
		@private
		@function
		@description Hover listener
			Used to focus items on hover.
			'this' is the Focusable.
	*/
	function hoverListener(event) {
		// set the _activeMethod so this can be passed onto the event
		this._activeMethod = 'hover';
		this._activeEvent = event;
		this.active(event.source);
		this._activeEvent = this._activeMethod = undefined;
	}
	
	/**
		@private
		@function
		@description Set _activeMethod to a value and call another function.
			This allows the _activeMethod to be passed to the event.
	*/
	function activeMethodWrap(focusable, methodName, func) {
		return function(event) {
			var returnVal;
			
			focusable._activeMethod = methodName;
			focusable._activeEvent = event;
			returnVal = func.apply(this, arguments);
			focusable._activeEvent = focusable._activeMethod = undefined;
			return returnVal;
		}
	}
	
	/**
		@name glow.ui.Focusable
		@class
		@extends glow.ui.Behaviour
		@description Manage a focusable element, or group of elements
			This can be used to create a single focus point for a set
			of focusable elements. Eg, a menu can have a single tab stop,
			and the arrow keys can be used to cycle through menu items.
			
			This means the user doesn't have to tab through every item in the
			menu to get to the next set of focusable items.
			
			The FocusManager can also be used to make a element 'modal', ensuring
			focus doesn't go to elements outside it.
			
			The aim of this behaviour is to make it easier to conform to
			<a href="http://www.w3.org/TR/2009/WD-wai-aria-practices-20091215/#keyboard">
				ARIA best practices for keyboard navigation
			</a>
			
		@param {glow.NodeList|string} container Parent focusable element of the group
			If tabindex isn't set on this element, it will be given tabindex="0",
			allowing the element to be focused using the tab key.
		@param {object} [opts] Options
			@param {string} [opts.children] Selector for child items that can be active
				These can be cycled through using the arrow keys when the Focusable
				or one of its children is active (usually when it has focus).
			@param {function|string} [opts.keyboardNav='arrows'] Alter the default keyboard behaviour.
				If 'arrows-x', the left & right arrow keys trigger {@link glow.ui.Focusable#next Focusable#next}
				and {@link glow.ui.Focusable#prev Focusable#prev} respectively. If 'arrows-y', the up & down
				arrow keys trigger {@link glow.ui.Focusable#next Focusable#next}
				and {@link glow.ui.Focusable#prev Focusable#prev} respectively. 'arrows' is
				a combination of the two.
				
				If a function is provided, it will be passed a {@link glow.events.KeyboardEvent} object.
				Use {@link glow.ui.Focusable#next Focusable#next},
				{@link glow.ui.Focusable#prev Focusable#prev} or
				{@link glow.ui.Focusable#activate Focusable#activate} to react to the
				key event.
				
				'this' inside this function refers to the Focusable.
			@param {boolean} [opts.setFocus=true] Sets whether focus is given to the active element.
				You need to set this to false if you want focus to remain in another
				element.
			@param {string} [opts.activeChildClass='active'] Class name to give the active child element.
			@param {boolean} [opts.activateOnHover=false] Activate items on hover?
			@param {boolean} [opts.loop=false] Loop from the last child item to the first (and vice-versa)?
				When this is true, calling {@link glow.ui.Focusable#next Focusable#next} when
				the last focusable item is active will activate the first.
				
		@example
			// A collection of buttons
			glow('#toolbar').focusable({
				children: '> li.button'
			});
			
			// The #toolbar now appears in tab order.
			// Once focused, the left & right arrow keys navigate between
			// buttons.
			
		@example
			// A modal dialog
			var dialog = glow('#dialog').hide();
			var focusable = dialog.focusable();
			
			glow('#openDialog').on('click', function() {
				dialog.show();
				focusable.modal(true);
			});
			
			glow('#closeDialog').on('click', function() {
				dialog.hide();
				focusable.modal(false);
			});
	*/
	function Focusable(container, opts) {
		/*!debug*/
			if (arguments.length > 2) {
				glow.debug.warn('[wrong count] glow.ui.Focusable expects 1 or 2 arguments, not ' + arguments.length + '.');
			}
			if (opts !== undefined && typeof opts !== 'object') {
				glow.debug.warn('[wrong type] glow.ui.Focusable expects object for "opts" argument, not ' + typeof opts + '.');
			}
		/*gubed!*/
		
		var keyboardNav;
		
		opts = this._opts = glow.util.apply({
			children: '',
			keyboardNav: 'arrows',
			setFocus: true,
			activeChildClass: 'active'
			// commented as undefined is falsey enough
			//activateOnHover: false,
			//loop: false
		}, opts || {});
		
		this.container = glow(container);
		keyboardNav = opts.keyboardNav;
		
		// build the keyhander, using presets or provided function
		this._keyHandler = activeMethodWrap(this, 'key',
			(typeof keyboardNav === 'string' ? keyHandlers[keyboardNav] : keyboardNav)
		);
		
		/*!debug*/
			if ( !this.container[0] ) {
				glow.debug.warn('[wrong value] glow.ui.Focusable - No container found');
			}
			if (typeof this._keyHandler != 'function') {
				glow.debug.warn('[wrong value] glow.ui.Focusable - unexpected value for opts.keyboardNav');
			}
			if (typeof opts.children != 'string') {
				glow.debug.warn('[wrong type] glow.ui.Focusable expects CSS string for "opts.children" argument, not ' + typeof opts.children + '.');
			}
		/*gubed!*/
		
		// populate #children
		updateChildren(this);
		
		// create initial focal point
		this.container[0].tabIndex = 0;
		
		// Add listener for activateOnHover
		if (opts.activateOnHover) {
			this.container.on('mouseover', hoverListener, this);
		}
		
		// listen for clicks
		this.container.on('click', clickSelectListener, this);
		
		// add to our array of focusables
		focusables.push(this);
	};
	glow.util.extend(Focusable, glow.ui.Behaviour);
	FocusableProto = Focusable.prototype;
	
	/**
		@name glow.ui.Focusable#_opts
		@type boolean
		@description Option object used in construction
	*/
	/**
		@name glow.ui.Focusable#_active
		@type boolean
		@description True/false to indicate if the Focusable is active
	*/
	FocusableProto._active = false;
	/**
		@name glow.ui.Focusable#_modal
		@type boolean
		@description True/false to indicate if the Focusable is modal
	*/
	FocusableProto._modal = false;
	/**
		@name glow.ui.Focusable#_disabled
		@type boolean
		@description True/false to indicate if the Focusable is enabled
	*/
	FocusableProto._disabled = false;
	/**
		@name glow.ui.Focusable#_lastActiveChild
		@type HTMLElement
		@description Stores the last value of #activeChild while the focusable is inactive
	*/
	/**
		@name glow.ui.Focusable#_keyHandler
		@type function
		@description Key handler function
	*/
	/**
		@name glow.ui.Focusable#_activeMethod
		@type string
		@description The last method used to activate a child element
	*/
	/**
		@name glow.ui.Focusable#_activeEvent
		@type string
		@description The event object accociated with _activeMethod
	*/
	/**
		@name glow.ui.Focusable#activeChild
		@type glow.NodeList
		@description The active child item.
			This will be an empty NodeList if no child is active
	*/
	FocusableProto.activeChild = glow();
	/**
		@name glow.ui.Focusable#activeIndex
		@type number
		@description The index of the active child item in {@link glow.ui.Focusable#children}.
			This will be undefined if no child is active.
	*/
	/**
		@name glow.ui.Focusable#container
		@type glow.NodeList
		@description Focusable container
	*/
	/**
		@name glow.ui.Focusable#children
		@type glow.NodeList
		@description NodeList of child items that are managed by this Focusable.
			This will be an empty nodelist if the focusable has no children
	*/
	FocusableProto.children = glow();
	
	/**
		@name glow.ui.Focusable#modal
		@function
		@description Get/set modality
			When a Focusable is modal it cannot be deactivated, focus cannot
			be given to elements outside of it until modal set to false.
			
		@param {boolean} setModal New modal value
		
		@returns this when setting, true/false when getting
	*/
	FocusableProto.modal = function(setModal) {
		/*!debug*/
			if (arguments.length > 1) {
				glow.debug.warn('[wrong count] glow.ui.Focusable#modal expects 0 or 1 argument, not ' + arguments.length + '.');
			}
		/*gubed!*/
		
		if (setModal === undefined) {
			return this._modal;
		}
		
		if (!this._disabled) {
			// Activate the modal if it isn't modal already
			if (setModal && !this._modal) {
				// Ensure we're not going to get a deadlock with another focusable
				if (modalFocusable) {
					modalFocusable.modal(false);
				}
				modalFocusable = this;
				this.active(true);
			}
			// switch modal off, if this focusable is modal
			else if (!setModal && this._modal) {
				modalFocusable = undefined;
			}
			
			this._modal = !!setModal;
		}
		return this;
	};
	
	/**
		@private
		@function
		@description Update activeChild and activeIndex according to an index.
	*/
	function activateChildIndex(focusable, index) {
		var prevActiveChild = focusable.activeChild[0],
			activeChildClass = focusable._opts.activeChildClass,
			activeChild = focusable.activeChild = glow( focusable.children[index] ),
			eventData = {
				item: activeChild,
				itemIndex: index,
				method: focusable._activeMethod || 'api',
				methodEvent: focusable._activeEvent
			};
		
		focusable.activeIndex = index;
		
		// have we changed child focus?
		if ( prevActiveChild === activeChild || focusable.fire('childActivate', eventData).defaultPrevented() ) {
			return;
		}
		
		// take the current active item out of the tab order
		if (prevActiveChild) {
			prevActiveChild.tabIndex = -1;
			glow(prevActiveChild).removeClass(activeChildClass);
		}
		
		// put the current active item into the tab order
		focusable.activeChild[0].tabIndex = 0;
		focusable.activeChild.addClass(activeChildClass);
		
		// give physical focus to the new item
		focusable._opts.setFocus && focusable.activeChild[0].focus();
	}
	
	/**
		@private
		@function
		@description Get the focusable child index of an element.
			The element may also be an element within the focusable's child items.
		@param {glow.ui.Focusable} focusable
		@param {glow.NodeList} child Element to get the index from.
		
		@returns {number} Index or -1 if element is not (and is not within) any of the focusable's child items.
	*/
	function getIndexFromElement(focusable, child) {
		var i,
			children = focusable.children,
			firstChild = children[0];
		
		// just exit if there are no child items
		if ( !firstChild ) {
			return -1;
		}
		
		child = glow(child).item(0);
		
		// do we have an active child to re-enable?
		if ( child[0] ) {
			i = children.length;
			
			// see if it's in the current child set
			while (i--) {
				if ( glow( children[i] ).contains(child) ) {
					return i;
				}
			}
		}
		return -1;
	}
	
	/**
		@private
		@function
		@description Ensure an index is within the range of indexes for this focusable.
		@param {glow.ui.Focusable} focusable
		@param {number} index Index to keep within range
		
		@returns {number} The index within range.
			If the focusable can loop, the index will be looped. Otherwise
			the index will be limited to its maximum & minimum
	*/
	function assertIndexRange(focusable, index) {
		var childrenLen = focusable.children.length;
		
		// ensure the index is within children range
		if (focusable._opts.loop) {
			index = index % childrenLen;
			if (index < 0) {
				index = childrenLen + index;
			}
		}
		else {
			index = Math.max( Math.min(index, childrenLen - 1), 0);
		}
		
		return index;
	}
	
	/**
		@private
		@function
		@description Deactivate the focusable
	*/
	function deactivate(focusable) {
		if ( focusable.fire('deactivate').defaultPrevented() ) {
			return;
		}
		
		// remove active class
		focusable.activeChild.removeClass(focusable._opts.activeChildClass);
		
		// store focusable so we can reactivate it later
		focusable._lastActiveChild = focusable.activeChild[0];
		
		// blur the active element
		( focusable.activeChild[0] || focusable.container[0] ).blur();
		
		focusable.activeIndex = undefined;
		
		// reset to empty nodelist
		focusable.activeChild = FocusableProto.activeChild;
		focusable._active = false;
		
		// remove listeners
		documentNodeList.detach('keypress', focusable._keyHandler).detach('keydown', keySelectListener);
		
		// allow the container to receive focus in case the child elements change
		focusable.container.prop('tabIndex', 0);
	}
	
	/**
		@private
		@function
		@description Activate the focusable
	*/
	function activate(focusable, toActivate) {
		var _active = focusable._active,
			focusContainerIfChildNotFound,
			indexToActivate = -1;
		
		// if currently inactive...
		if (!_active) {
			if ( focusable.fire('activate').defaultPrevented() ) {
				return;
			}
			
			updateChildren(focusable);
			focusable._active = true;
			// start listening to the keyboard
			documentNodeList.on('keypress', focusable._keyHandler, focusable).on('keydown', keySelectListener, focusable);
			// give focus to the container - a child element may steal focus in activateChildIndex
			focusContainerIfChildNotFound = true;
		}
		
		// Work out what child item to focus.
		// We avoid doing this if we were 
		if ( focusable.children[0] ) {
			// activating by index
			if (typeof toActivate === 'number') {
				indexToActivate = assertIndexRange(focusable, toActivate);
			}
			// activating by element
			else if (typeof toActivate !== 'boolean') {
				indexToActivate = getIndexFromElement(focusable, toActivate);
			}
			
			// still no index to activate? If we were previously inactive, try the last active item
			if (indexToActivate === -1 && !_active) {
				indexToActivate = getIndexFromElement(focusable, focusable._lastActiveChild);
				indexToActivate = indexToActivate !== -1 ? indexToActivate : 0;
			}
		}
		
		// If we have an item to activate, let's go for it
		if (indexToActivate !== -1 && indexToActivate !== focusable.activeIndex) {
			activateChildIndex(focusable, indexToActivate);
		}
		else if (focusContainerIfChildNotFound) {
			focusable._opts.setFocus && focusable.container[0].focus();
		}
	}
	
	/**
		@name glow.ui.Focusable#active
		@function
		@description Get/set the active state of the Focusable
			Call without arguments to get the active state. Call with
			arguments to set the active element.
			
			A Focusable will be activated automatically when it receieves focus.
		
		@param {number|glow.NodeList|boolean} [toActivate] Item to activate.
			Numbers will be treated as an index of {@link glow.ui.FocusManager#children children}.
			
			'true' will activate the container, but none of the children.
			
			'false' will deactivate the container and any active child
		
		@returns {glow.ui.Focusable|boolean}
			Returns boolean when getting, Focusable when setting
	*/
	FocusableProto.active = function(toActivate) {
		/*!debug*/
			if (arguments.length > 1) {
				glow.debug.warn('[wrong count] glow.ui.Focusable#active expects 0 or 1 argument, not ' + arguments.length + '.');
			}
		/*gubed!*/
		
		var _active = this._active;
		
		// getting
		if (toActivate === undefined) {
			return _active;
		}
		
		// setting
		if (!this._disabled) {
			// deactivating
			if (toActivate === false) {
				if (!this._modal && _active) {
					deactivate(this);
				}
			}
			// activating
			else {
				activate(this, toActivate)
			}
		}
		return this;
	};
	
	/**
		@private
		@function
		@description Generates #next and #prev
	*/
	function nextPrev(amount) {
		return function() {
			/*!debug*/
				if (arguments.length > 1) {
					glow.debug.warn('[wrong count] glow.ui.Focusable#' + (amount > 0 ? 'next' : 'prev') + ' expects 0 arguments, not ' + arguments.length + '.');
				}
			/*gubed!*/
			
			if (this._active) {
				this.active( this.activeIndex + amount );
			}
			return this;
		}
	}
	
	/**
		@name glow.ui.Focusable#next
		@function
		@description Activate next child item.
			Has no effect on an inactive Focusable.
		@returns this
	*/
	FocusableProto.next = nextPrev(1);
	
	/**
		@name glow.ui.Focusable#prev
		@function
		@description Activate previous child item
			Has no effect on an inactive Focusable.
		@returns this
	*/
	FocusableProto.prev = nextPrev(-1);
	
	/**
		@name glow.ui.Focusable#disabled
		@function
		@description Enable/disable the Focusable, or get the disabled state
			When the Focusable is disabled, it (and its child items) cannot
			be activated or receive focus.
			
		@param {boolean} [newState] Disable the focusable?
			'false' will enable a disabled focusable.
		
		@returns {glow.ui.Focusable|boolean}
			Returns boolean when getting, Focusable when setting
	*/
	FocusableProto.disabled = function(newState) {
		/*!debug*/
			if (arguments.length > 1) {
				glow.debug.warn('[wrong count] glow.ui.Focusable#disabled expects 0 or 1 argument, not ' + arguments.length + '.');
			}
		/*gubed!*/
		
		// geting
		if (newState === undefined) {
			return this._disabled;
		}
		
		// setting
		if (newState) {
			this.active(false);
			this._disabled = !!newState;
		}
		else {
			this._disabled = !!newState;
			
			// reactivate it if it were modal
			if (this._modal) {
				this.active(true);
			}
		}
		return this;
	}
	
	/**
		@name glow.ui.Focusable#destroy
		@function
		@description Destroy the Focusable
			This removes all focusable behaviour from the continer
			and child items.
			
			The elements themselves will not be destroyed.
		@returns this
	*/
	FocusableProto.destroy = function() {
		var i = focusables.length;
		
		glow.events.removeAllListeners( [this] );
		
		this.modal(false).active(false).container
			// remove listeners
			.detach('mouseover', hoverListener)
			.detach('click', clickSelectListener)
			// remove from tab order
			.prop('tabIndex', -1);
			
		this.container = undefined;
		
		// remove this focusable from the static array
		while (i--) {
			if (focusables[i] === this) {
				focusables.splice(i, 1);
				break;
			}
		}
	}
	
	/**
		@name glow.ui.Focusable#event:select
		@event
		@description Fires when a child of the Focusable is selected.
			Items are selected by clicking, or pressing enter when a child is active.
		
			Cancelling this event prevents the default click/key action.
		
		@param {glow.events.Event} event Event Object
		@param {glow.NodeList} event.item Item selected.
		@param {number} event.itemIndex The index of the selected item in {@link glow.ui.Focusable#children}.
	*/
	
	/**
		@private
		@function
		@description
			Listens for click selections on the Focusable
			'this' is the Focusable.
	*/
	function clickSelectListener() {
		if ( this.activeChild[0] ) {
			return !this.fire('select', {
				item: this.activeChild,
				itemIndex: this.activeIndex
			}).defaultPrevented();
		}
	}
	
	/**
		@private
		@function
		@description
			Same as above, but for keys
			'this' is the Focusable.
	*/
	function keySelectListener(event) {
		if (event.key === 'return') {
			return clickSelectListener.call(this);
		}
	}
	
	/**
		@name glow.ui.Focusable#event:activate
		@event
		@description Fires when the Focusable becomes active
			Cancelling this event prevents the Focusable being actived
		
		@param {glow.events.Event} event Event Object
	*/
	
	/**
		@name glow.ui.Focusable#event:childActivate
		@event
		@description Fires when a child item of the Focusable becomes active
			Cancelling this event prevents the child item being actived
		
		@param {glow.events.Event} event Event Object
		@param {glow.NodeList} event.item Item activated.
		@param {number} event.itemIndex The index of the activated item in {@link glow.ui.Focusable#children}.
		@param {string} event.method Either 'key', 'hover' or 'api' depending on how the item was activated.
			This allows you to react to certain kinds of activation.
		@param {glow.events.DomEvent} [event.methodEvent] An event object for the 'key' or 'hover' event.
			For 'key' methods this will be a more specific {@link glow.events.KeyboardEvent}.
			
			If the method was neither 'key' or 'hover', methodEvent will be undefined.
	*/
	
	/**
		@name glow.ui.Focusable#event:deactivate
		@event
		@description Fires when the Focusable becomes deactive
			Cancelling this event prevents the Focusable being deactived
		
		@param {glow.events.Event} event Event Object
	*/
	
	// EXPORT
	glow.ui.Focusable = Focusable;
});
Glow.provide(function(glow) {
	var undefined,
		WidgetProto;

	/**
		@name glow.ui.Widget
		@constructor
		@extends glow.events.Target
		
		@description An abstract Widget class
			The Widget class serves as a base class that provides a shared framework on which other,
			more specific, widgets can be implemented. While it is possible to create an instance
			of this generic widget, it is more likely that your widget class will extend this class.
			
			Your widget constructor should call the base constructor, and should end in a call to _init.
			
		@param {string} name The name of this widget.
			This is added to class names in the generated DOM nodes that wrap the widget interface.
			
		@param {object} [opts]
			@param {string} [opts.className] Class name to add to the container.
			@param {string} [opts.id]  Id to add to the container.
			
		@example
			function MyWidget(opts) {
				// set up your widget...
				// call the base constructor, passing in the name and the options
				glow.ui.Widget.call(this, 'MyWidget', opts);
				
				// start init
				this._init();
			}
			glow.util.extend(MyWidget, glow.ui.Widget);
	*/
	
	function Widget(name, opts) {
		/*!debug*/
			if (arguments.length < 1 || arguments.length > 2) {
				glow.debug.warn('[wrong count] glow.ui.Widget expects 1 or 2 arguments, not '+arguments.length+'.');
			}
			if (typeof arguments[0] !== 'string') {
				glow.debug.warn('[wrong type] glow.ui.Widget expects argument 1 to be of type string, not '+typeof arguments[0]+'.');
			}
		/*gubed!*/
		
		this._name = name;
		this._locale = 'en'; // todo: default should come from i18n module
		this.phase = 'constructed';
		this._observers = [];
		this._opts = opts || {};
	}
	glow.util.extend(Widget, glow.events.Target); // Widget is a Target
	WidgetProto = Widget.prototype;
	/**
		@name glow.ui.Widget#_locale
		@protected
		@type string
		@description The locale of the widget
	*/
	/**
		@name glow.ui.Widget#_name
		@protected
		@type string
		@description The name of the widget.
			This is the first argument passed into the constructor.
	*/
	/**
		@name glow.ui.Widget#_stateElm
		@protected
		@type glow.NodeList
		@description The wrapper element that contains the state class
	*/
	/**
		@name glow.ui.Widget#_themeElm
		@protected
		@type glow.NodeList
		@description The wrapper element that contains the theme class
	*/
	/**
		@name glow.ui.Widget#_opts
		@protected
		@type Object
		@description The option object passed into the constructor
	*/
	/**
		@name glow.ui.Widget#_disabled
		@protected
		@type boolean
		@description Is the widget disabled?
			This is read-only
	*/
	WidgetProto._disabled = false;
	/**
		@name glow.ui.Widget#_observers
		@protected
		@type object[]
		@description Objects (usually widgets & dehaviours) observing this Widget
	*/
	
	/**
		@name glow.ui.Widget#phase
		@type string
		@description The phase within the lifecycle of the widget.
			Will be one of the following:
			
			<dl>
				<dt>constructed</dt>
				<dd>The widget has been constructed but not yet initialised</dd>
				<dt>initialised</dt>
				<dd>The widget has been initialised but not yet build</dd>
				<dt>built</dt>
				<dd>The widget has been built but not yet bound</dd>
				<dt>ready</dt>
				<dd>The widget is in a fully usable state</dd>
				<dt>destroyed</dt>
				<dd>The widget has been destroyed</dd>
			</dl>
			
			Usually, init, build & bind are done in the constructor, so
			you may only interact with a widget that is either 'ready' or 'destroyed'.
	*/
	/**
		@name glow.ui.Widget#container
		@type glow.NodeList
		@description The outermost wrapper element of the widget.
	*/
	/**
		@name glow.ui.Widget#content
		@type glow.NodeList
		@description The content element of the widget
			This is inside various wrapper elements used to track the state of
			the widget.
	*/
	
	function syncListener(e) {
		// handle notifications about changes to the disabled state
		if (e.disabled !== undefined) {
			this.disabled(e.disabled);
		}
		else if (e.destroy) {
			this.destroy();
		}
	}
	
	/**
		@name glow.ui.Widget#_tie
		@protected
		@function
		@description Specify a group of widgets that should stay in _sync with this one.
			These synced widgets can listen for a '_sync' event on themselves, defining their own handler for the provided event.
			The disabled and destroy methods automatically synchronize with their synced child widgets.
		
		@param {glow.ui.Widget} ... Child widgets to synchronize with.
		
		@example
			function MyWidget() {
				this.value = 0; // initially
				
				// this widget handles notifications of new values
				// from other widgets it is syced to
				var that = this;
				this.on('_sync', function(e) {
					if (typeof e.newValue !== undefined) {
						that.value = e.newValue;
					}
				});	
			}
			glow.util.extend(MyWidget, glow.ui.Widget); // MyWidget extends the Base Widget
			
			MyWidget.prototype.setValue = function(n) {
				this.value = n;
				
				// this widget notifies about changes to its value
				this._sync({newValue: this.value});
			}
			
			// widgets b and c will all be notified when a's value changes
			var a = new MyWidget('A');
			var b = new MyWidget('B');
			var c = new MyWidget('C');
			
			a._tie(b, c);
			
			a.setValue(1); // a, b, and c all have a value of 1 now
	 */
	WidgetProto._tie = function(/*...*/) {
		/*!debug*/
			if (arguments.length === 0) {
				glow.debug.warn('[wrong count] glow.ui.Widget#_tie expects at least 1 argument, not '+arguments.length+'.');
			}
		/*gubed!*/
		
		var newObservers = Array.prototype.slice.call(arguments),
			i = newObservers.length;
		
		// add a default _sync listener to react to disabled and destroy
		while (i--) {
			newObservers[i].on('_sync', syncListener);
		}
		
		this._observers = this._observers.concat(newObservers);
		
		return this;
	}

	/**
		@developer
		@name glow.ui.Widget#_sync
		@method
		@param {object} [changes] Key/value collection of new information. Will be added to the listeners' event.
		@description Tell all widgets synced with this widget about any changes.
	 */
	WidgetProto._sync = function(changes) { // public shortcut to fire _notify
		/*!debug*/
			if (arguments.length > 1) {
				glow.debug.warn('[wrong count] glow.ui.Widget#_sync expects 1 or fewer arguments, not '+arguments.length+'.');
			}
			
			if (arguments.length && typeof changes !== 'object') {
				glow.debug.warn('[wrong type] glow.ui.Widget#_sync expects argument 1 to be of type object, not '+(typeof changes)+'.');
			}
		/*gubed!*/
		
		glow.events.fire( this._observers, '_sync', changes || {} );
		
		return this;
	}

	/**
		@name glow.ui.Widget#disabled
		@function
		@description Enable/disable the Widget, or get the disabled state
			If other widgets are synced with this one, they will become disabled too.
			
		@param {boolean} [newState] Disable the focusable?
			'false' will enable a disabled focusable.
		
		@example
			var a = new MyWidget();
			var b = new MyWidget();
			var c = new MyWidget();
			
			c._tie(a, b);
			
			c.disabled(true); // a, b, and c are now disabled
	 */
	WidgetProto.disabled = function(newState) {
		/*!debug*/
			if (arguments.length > 1) {
				glow.debug.warn('[wrong count] glow.ui.Widget#disabled expects 0 or 1 argument, not ' + arguments.length + '.');
			}
		/*gubed!*/
		
		// geting
		if (newState === undefined) {
			return this._disabled;
		}
		
		// setting
		newState = !!newState;
		if ( newState !== this._disabled && !this.fire('disabled', {disabled:newState}).defaultPrevented() ) {
			this._sync({
				disabled: newState
			});
			this._stateElm[newState ? 'addClass' : 'removeClass']('disabled');
			this._disabled = !!newState;
		}
		return this;
	}

	/**
		@name glow.ui.Widget#_init
		@protected
		@function
		@description Initialise the widget.
			This is similar to the constructor, but for code that you may need to run
			again.
			
			You init function should call the base _init, and end in a call to _build on your widget.
		
		@example
			MyWidget.prototype._init = function() {
				// set properties here
				
				// call base _init
				glow.ui.Widget.prototype._init.call(this);
				// call _build
				this._build();
			}
		
	 */
	WidgetProto._init = function() {
		this.phase = 'initialised';
	}

	/**
		@name glow.ui.Widget#_build
		@protected
		@function
		@description Build the html structure for this widget.
			All actions relating to wrapping, creating & moving elements should be
			done in this method. The base method creates the container, theme & state elements.
			
			Adding behaviour to these elements should be handed in {@link glow.ui.Widget#_bind}.
			
			You Widget's _build method should call the base build method and end in a call to _bind.
		
		@param {selector|HTMLElement|NodeList} [content] Content element for the widget.
			This will be wrapped in container, theme & state elements. By default this is
			an empty div.
			
		@example
			MyWidget.prototype._build = function() {
				// create some content
				var content = glow('<p>Hello!</p>');
				// call the base build
				glow.ui.Widget.prototype._build.call(this, content);
				// call _bind
				this._bind();
			}
	 */
	WidgetProto._build = function(content) {
		/*!debug*/
			if (arguments.length > 1) {
				glow.debug.warn('[wrong count] glow.ui.Widget#_build expects 0-1 argument, not '+arguments.length+'.');
			}
		/*gubed!*/
		
		var container,
			name = this._name,
			opts = this._opts;
		
		content = this.content = glow(content || '<div></div>');
		
		/*!debug*/
			if (content.length < 1) {
				glow.debug.warn('[error] glow.ui.Widget#_build expects a content node to attach to. The given "content" argument was empty or not found.');
			}
		/*gubed!*/
		
		container = this.container = glow('' +
			'<div class="glow200b1-' + name + '">' +
				'<div class="' + name + '-theme">' +
					'<div class="' + name + '-state"></div>' +
				'</div>' +
			'</div>' +
		'');
		
		content.addClass(name + '-content').wrap(container);
		this._stateElm = content.parent();
		this._themeElm = this._stateElm.parent();
		
		if (opts.className) {
			container.addClass(opts.className);
		}
		if (opts.id) {
			container[0].id = opts.id;
		}
		
		this.phase = 'built';
	}
	
	/**
		@developer
		@name glow.ui.Widget#_bind
		@function
		@description Add behaviour to elements created in {@link glow.ui.Widget#_build _build}.
			Your _bind method should call the base _bind and may end in a call
			to _updateUi for initial positioning etc.
			
		@example
			MyWidget.prototype._bind = function() {
				// add some behaviour
				this.content.on('click', function() {
					alert('Hello!');
				});
				// call base _bind
				glow.ui.Widget.prototype._bind.call(this);
			}
	 */
	WidgetProto._bind = function() {
		this.phase = 'ready';
	}

	/**
		@name glow.ui.Widget#_updateUi
		@function
		@description Cause any functionality that deals with visual layout or UI display to update.
			This function should be overwritten by Widgets that need to update or redraw. For example,
			you may use this method to reposition or reorder elements.
			
			This is a convention only, the base method does nothing.
		
		@example
			MyWidget.prototype.updateUi = function() {
				// update the UI
			}
			
	 */
	WidgetProto._updateUi = function() {}

	/**
		@developer
		@name glow.ui.Widget#destroy
		@function
		@description Cause any functionality that deals with removing and deleting this widget to run.
			By default the container and all it's contents are removed.
		@fires glow.ui.Widget#event:destroy
	 */
	WidgetProto.destroy = function() {
		/*!debug*/
			if (arguments.length !== 0) {
				glow.debug.warn('[wrong count] glow.ui.Widget#destroy expects 0 arguments, not '+arguments.length+'.');
			}
		/*gubed!*/
		if ( !this.fire('destroy').defaultPrevented() ) {
			this._sync({
				destroy: 1
			});
			glow.events.removeAllListeners( [this] );
			this.container.destroy();
			this.phase = 'destroyed';
		}
		return this;
	}

	/**
		@developer
		@name glow.ui.Widget#event:disable
		@event
		@description Fired after the disabled property is changed via the {@link glow.ui.Widget#disable} or {@link glow.ui.Widget#enable} method.
		This includes widgets that are changed as a result of being synced to this one.
	 */
	
	/**
		@developer
		@name glow.ui.Widget#event:destroy
		@event
		@description Fired when destroy is called on this widget.
		@see glow.ui.Widget#destroy
	 */

	// export
	glow.ui.Widget = Widget;
});
Glow.provide(function(glow) {
	var OverlayProto,
		WidgetProto = glow.ui.Widget.prototype,
		idCounter = 0,
		undefined,
		instances = {}; // like {uid: overlayInstance}
	
	
	var vis = {
		SHOWING: 2,
		SHOWN: 1,
		HIDING: -1,
		HIDDEN: -2
	};
	
	
	/**
		@name glow.ui.Overlay
		@class
		@augments glow.ui
		@description A container element displayed on top of the other page content
		@param {selector|NodeList|String|boolean} content
			the element that contains the contents of the overlay. If not in the document, you must append it to the document before calling show().

		@param {object} [opts]
			@param {function|selector|NodeList|boolean} [opts.hideWhenShown] Select which things to hide whenevr the overlay is in a shown state.
			By default all `object` and `embed` elements will be hidden, in browsers that cannot properly layer those elements, whenever any overlay is shown.
			Set this option to a false value to cause the overlay to never hide any elements, or set it to a bespoke selector, NodeList
			or a function that returns a NodeList which will be used instead.
		@example
			var myOverlay = new glow.ui.Overlay(
				glow(
					'<div>' +
					'  <p>Your Story has been saved.</p>' +
					'</div>'
				).appendTo(document.body)
			);
			
			glow('#save-story-button').on('click', function() {
				myOverlay.show();
			});
	 */
	
	function Overlay(content, opts) {
		/*!debug*/
			if (arguments.length < 1 || content === undefined) {
				glow.debug.warn('[wrong type] glow.ui.Overlay expects "content" argument to be defined, not ' + typeof content + '.');
			}
			if (opts !== undefined && typeof opts !== 'object') {
				glow.debug.warn('[wrong type] glow.ui.Overlay expects object as "opts" argument, not ' + typeof opts + '.');
			}
		/*gubed!*/
		var that = this,
			ua;
		
		opts = glow.util.apply({ }, opts);
		
		//call the base class's constructor
		Overlay.base.call(this, 'overlay', opts);
		
		this.uid = 'overlayId_' + glow.UID + '_' + (++idCounter);
		instances[this.uid] = this; // useful for modal overlays?
		
		this._init(opts);
		this._build(content);
		this._bind();
	}
	glow.util.extend(Overlay, glow.ui.Widget);
	
	OverlayProto = Overlay.prototype;
	
	OverlayProto._init = function() {
		WidgetProto._init.call(this);
		
		/**
			@name glow.ui.Overlay#shown
			@description True if the overlay is shown.
				This is a read-only property to check the state of the overlay.
			@type boolean
		*/
		this.shown = false;
		
		return this;
	}
	
	OverlayProto.destroy = function() {
		WidgetProto.destroy.call(this);
		
		delete instances[this.uid];
	}
	
	OverlayProto._build = function(content) {
		var that = this;
		
		WidgetProto._build.call(this, content);
		
		/*!debug*/
			if (this.content.length < 1) {
				glow.debug.warn('[ivalid argument] glow.ui.Overlay expects "content" argument to refer to an element that exists, no elements found for the content argument provided.');
			}
		/*gubed!*/
		
		// some browsers need to hide Flash when the overlay is shown (like non-mac opera and gecko 1.9 or less)
		if (this._opts.hideWhenShown === undefined) { // need to make our own flash handler
			ua = navigator.userAgent; // like ﻿"... rv:1.9.0.5) gecko ..."
			/**
				A function that returns a NodeList containing all elements that need to be hidden.
				@name glow.ui.Overlay#_whatToHide
				@private
				@returns {glow.NodeList} Elements that need to be hidden when the overlay is shown.
			 */
			this._whatToHide = (
				glow.env.opera && !/macintosh/i.test(ua)
				|| /rv:1\.9\.0.*\bgecko\//i.test(ua)
				|| glow.env.webkit && !/macintosh/i.test(ua)
			)?
				function() {
					return glow('object, embed')/*.filter(function() {
						return !that.container.contains(this); // don't hide elements that are inside the overlay
					});*/
				}
				: function() { return glow(); }
		}
		else { // user provides their own info about what to hide
			if (!this._opts.hideWhenShown) { // a value that is false
				this._whatToHide = function() { return glow(); }
			}
			else if (typeof this._opts.hideWhenShown === 'function') { // a function
				this._whatToHide = this._opts.hideWhenShown;
			}
			else if (this._opts.hideWhenShown.length !== undefined) { // nodelist or string?
				this._whatToHide = function() { return glow('*').filter(this._opts.hideWhenShown); }
			}
		}
		
		//add IE iframe hack if needed, wrap content in an iFrame to prevent certain elements below from showing through
		if (glow.env.ie) {
			this._iframe = glow('<iframe src="javascript:\'\'" style="display:block;width:100%;height:1000px;margin:0;padding:0;border:none;position:absolute;top:0;left:0;filter:alpha(opacity=0);"></iframe>')
			this._iframe.css('z-index', 0);
			
			this._iframe.insertBefore(this.content);
		}
		
		this.content
			.css('position', 'relative')
			.css('z-index', 1)
			.css('top', 0)
			.css('left', 0);

		return this;
	}
	
	/**
		@name glow.ui.Overlay#hideFlash
		@method
		@description Hides all Flash elements on the page, outside of the overlay.
		This should only be neccessary on older browsers that cannot properly display
		overlay content on top of Flash elements. On those browsers Glow will automatically
		call this method for you in the onshow event, and will automatically call
		showFlash for you in the afterhide event.
	 */
	OverlayProto.hideFlash = function() { /*debug*///console.log('hideFlash');
		var toHide,
			that = this,
			hidBy = '';
			
		toHide = this._whatToHide();
		
		// multiple overlays may hide the same element
		// flash elements keep track of which overlays have hidden them
		// trying to hide a flash element more than once does nothing
		for (var i = 0, leni = toHide.length; i < leni; i++) {
			hidBy = (toHide.item(i).data('overlayHidBy') || '');
			
			if (hidBy === '') {
				toHide.item(i).data('overlayOrigVisibility', toHide.item(i).css('visibility'));
				toHide.item(i).css('visibility', 'hidden');
			}
			if (hidBy.indexOf('['+this.uid+']') === -1) {
				toHide.item(i).data('overlayHidBy', hidBy + '['+this.uid+']');
			}
		}
		
		// things were hidden, make sure they get shown again
		if (toHide.length && !that._doShowFlash) { // do this only once
			that._doShowFlash = true;
			that.on('afterHide', function() { /*debug*///console.log('callback');
				that.showFlash();
			});
		}
		
		this._hiddenElements = toHide;
	}
	
	/**
		@name glow.ui.Overlay#showFlash
		@method
		@description Hides all Flash elements on the page, outside of the overlay.
		If a Flash element has been hidden by more than one overlay, you must call
		showFlash once for each time it was hidden before the Flash will finally appear.
	 */
	OverlayProto.showFlash = function() { /*debug*///console.log('showFlash');
		var hidBy = '';
		
		if (!this._hiddenElements || this._hiddenElements.length === 0) { // this overlay has not hidden anything?
			return;
		}
		
		var toShow = this._hiddenElements;
		
		for (var i = 0, leni = toShow.length; i < leni; i++) {
			hidBy = (toShow.item(i).data('overlayHidBy') || '');
			
			if (hidBy.indexOf('['+this.uid+']') > -1) { // I hid this
				hidBy = hidBy.replace('['+this.uid+']', ''); // remove me from the list of hiders
				toShow.item(i).data('overlayHidBy', hidBy);
			}
			
			if (hidBy == '') { // no hiders lefts
				toShow.item(i).css( 'visibility', toShow.item(i).data('overlayOrigVisibility') );
			}
		}
	}
	
	/**
		@name glow.ui.Overlay#event:show
		@event
		@description Fired when the overlay is about to appear on the screen, before any animation.

			At this	point you can access the content of the overlay and make changes 
			before it is shown to the user. If you prevent the default action of this
			event (by returning false or calling event.preventDefault) the overlay 
			will not show.
			
		@param {glow.events.Event} event Event Object
	*/
	
	/**
		@name glow.ui.Overlay#event:afterShow
		@event
		@description Fired when the overlay is showing to the user and any delay or 'show' animation is complete.

			This event is ideal to assign focus to a particular part of	the overlay.
			If you want to change content of the overlay before it appears, see the 
			'show' event.
			
		@param {glow.events.Event} event Event Object
	*/
	
	/**
		@name glow.ui.Overlay#event:hide
		@event
		@description Fired when the overlay is about to hide.

			If you prevent the default action of this event (by returning false or 
			calling event.preventDefault) the overlay will not hide.
			
		@param {glow.events.Event} event Event Object
	*/
	
	/**
		@name glow.ui.Overlay#event:afterHide
		@event
		@description Fired when the overlay has fully hidden, after any delay or hiding animation has completed.
		@param {glow.events.Event} event Event Object
	*/
	
	// animations that can be referred to in setAnim by string.
	// Each is an array of 2 item, one function to put the Overlay in an initial state
	// for this animation, and one for the animation itself
	var anims = {
		slide: [
			function(overlay) {
				overlay.container.height(0);
			},
			function(isShow, callback) {
				var anim,
					container = this.container;
					
				if (isShow) {
					anim = container.slideOpen(0.5).data('glow_slideOpen');
				}
				else {
					anim = container.slideShut(0.5).data('glow_slideShut');
				}
				
				anim.on('complete', callback);
			}
		],
		fade: [
			function(overlay) {
				overlay.container.css('opacity', 0);
			},
			function(isShow, callback) {
				var anim,
					container = this.container;
					
				if (isShow) {
					anim = container.fadeIn(0.5).data('glow_fadeIn');
				}
				else {
					anim = container.fadeOut(0.5).data('glow_fadeOut');
				}
				
				anim.on('complete', callback);
			}
		]
	}
	
	/**
		@name glow.ui.Overlay#setAnim
		@function
		@description Set the animation to use when showing and hiding this overlay.
		@param {string|Array|Function|null} anim Anim to use.
			At its simplest, this can be the string 'slide' or 'fade', to give
			the overlay a fading/sliding animation.
		
			If this value is an animation definition, in the form of an array of
			arguments to pass to the {@link glow.Anim} constructor, those values
			will be used to create the show animation. The hide animation will
			then be the reverse of the show. This is the easiest option if you
			intend your show and hide animations to simply reverse one another.
			
			Alternatively, if you need more control over your show and hide
			animations, you can provide a function.	This function will be called
			whenever the overlay has its show or hide method invoked, and will
			be provided a boolean (true meaning it's being shown, false meaning
			it's being hidden), and a callback. You can then manage the animations
			yourself within that function, and then invoke the callback when
			either animation is complete. In your function, 'this' refers to the
			overlay.
			
			Passing null will delete any previously set animation.
		
		@returns this
	*/
	OverlayProto.setAnim = function(anim) {
		if (anim === null) {
			delete this._animDef;
			delete this._animator;
		}
		else if (typeof anim === 'string') {
			anims[anim][0](this);
			this._animator = anims[anim][1];
		}
		else if (typeof anim === 'function') {
			this._animator = anim;
		}
		else {
			this._animDef = anim;
			this._animDef[2] = this._animDef[2] || {};
			this._animDef[2].destroyOnComplete = false;
		}
		
		return this;
	}
	
	/**
		@name glow.ui.Overlay#show
		@function
		@param {number} [delay=0] The delay before the overlay is shown.
			By default, the overlay will show immediately. Specify a number of seconds to delay showing.
			The event "afterShow" will be called after any delay and animation.
		@description Displays the overlay after an optional delay period and animation.

		@returns this
	*/
	OverlayProto.show = function(delay) {
		//if (this.shown) { /*debug*///console.log('show ignored');
		//	return this;
		//}
		var that = this;
		
		if ( !this.fire('show').defaultPrevented() ) {
			if (this._timer) {
				clearTimeout(this._timer);
			}
			
			if (delay) {
				this._timer = setTimeout(function() {
					show.call(that);
				}, delay * 1000);
			}
			else {
				show.call(that);
			}
		}
		
		return this;
	}
	
	function show() { /*debug*///console.log('show() curently '+this.state);
		var that = this;
		
		// already being shown?
		if (this.state === vis.SHOWING || this.state === vis.SHOWN) {
			return;
		}
		
		setShown(that, true);
		
		if (this._whatToHide) { this.hideFlash(); }
		
		if (this._animator) {
			that.state = vis.SHOWING;
			this._animator.call(this, true, function() { afterShow.call(that); });
		}
		else if (this._animDef) {
			if (this._anim) { // is hiding?
				this.state = vis.SHOWING;
				this._anim.reverse();
			}
			else { // is hidden?
				this.state = vis.SHOWING;
				
				// this same anim is reused (by reversing it) for showing and hiding
				this._anim = this.container.anim(this._animDef[0], this._animDef[1], this._animDef[2]);
				this._anim.on('complete', function() {
					
					if (that.state === vis.SHOWING) {
						setShown(that, true);
						afterShow.call(that);
					} 
					else if (that.state === vis.HIDING) {
						setShown(that, false);
						afterHide.call(that);
					}
				});
			}
			
			this._anim.start();
		}
		else {
			afterShow.call(this);
		}
	}
	
	function afterShow() { /*debug*///console.log('after show');
		this.state = vis.SHOWN;
		this.fire('afterShow');
	}
	
	/**
		@private
		@function
		@description Set the shown state & add/remove a class from the state element
	*/
	function setShown(overlay, shownState) {
		var stateElm = overlay._stateElm;
		
		overlay.shown = shownState;
		
		if (shownState) {
			stateElm.removeClass('hidden');
			stateElm.addClass('shown');
		}
		else {
			stateElm.removeClass('shown');
			stateElm.addClass('hidden');
		}
	}
	
	function hide() { /*debug*///console.log('hide() curently '+this.state);
		var that = this;
		
		if (this.state === vis.HIDING || this.state === vis.HIDDEN) {
			return;
		}
		
		if (this._animator) { // provided by user
			this._animator.call(this, false, function() {
				setShown(that, false);
				afterHide.call(that);
			});
		}
		else if (this._anim) { // generated by overlay
			this.state = vis.HIDING;
			this._anim.reverse();
			this._anim.start();
		}
		else { // no animation
			setShown(that, false);
			afterHide.call(this);
		}
	}
	
	function afterHide() { /*debug*///console.log('after hide');
		this.state = vis.HIDDEN;
		this.fire('afterHide');
	}
	
	/**
		@name glow.ui.Overlay#hide
		@function
		@param {number} [delay=0] The delay before the overlay is shown.
			By default, the overlay will show immediately. Specify a number of seconds to delay showing.
			The event "afterShow" will be called after any delay and animation.
		@description Hides the overlay after an optional delay period and animation

		@returns this
	*/
	OverlayProto.hide = function(delay) {
		//if (!this.shown) { /*debug*///console.log('hide ignored');
		//	return this;
		//}
		
		var that = this;
		
		if ( !this.fire('hide').defaultPrevented() ) {
			if (this._timer) {
				clearTimeout(this._timer);
			}
			
			if (delay) {
				this._timer = setTimeout(function() {
					hide.call(that);
				}, delay * 1000);
			}
			else {
				hide.call(that);
			}
		}
		
		return this;
	}

	// export
	glow.ui = glow.ui || {};
	glow.ui.Overlay = Overlay;
});
Glow.provide(function(glow) {
	var undefined, AutoSuggestProto,
		Widget = glow.ui.Widget,
		WidgetProto = Widget.prototype,
		// this is used for HTML escaping in _format
		tmpDiv = glow('<div></div>');
	
	/**
		@name glow.ui.AutoSuggest
		@extends glow.ui.Widget
		@constructor
		@description Create a menu that displays results filtered by a search term.
			This widget can be easily linked to a text input via {@link glow.ui.AutoSuggest#linkToInput}
			so results will be filtered by text entered by the user. This appears as a list of selectable
			items below the input element (optional) which dynamically updates based on what
			has been typed so far.
			
			By default, items where the search term matches the start of the item
			(or its 'name' property) will be returned. You can change the
			filtering behaviour via {@link glow.ui.AutoSuggest#setFilter setFilter}.
			
			The matched item (or its 'name' property) will be displayed with the matching
			portion underlined. You can change the output via {@link glow.ui.AutoSuggest#setFormat setFormat}
		
		@param {Object} [opts] Options				
			@param {number} [opts.width] Apply a width to the results list.
				By default, the AutoSuggest is the full width of its containing element,
				or the width of the input it's linked to if autoPositioning.
			@param {number} [opts.maxResults] Limit the number of results to display.
			@param {number} [opts.minLength=3] Minimum number of chars before search is executed
				This prevents searching being performed until a specified amount of chars
				have been entered.
			@param {boolean} [opts.caseSensitive=false] Whether case is important when matching suggestions.
				If false, the value passed to the filter will be made lowercase, a custom filter
				must also lowercase the property it checks.
			@param {boolean} [opts.activateFirst=true] Activate the first item when results appear?
				If false, results with be shown with no active item.
			@param {function|string} [opts.keyboardNav='arrow-y'] Alter the default keyboard behaviour.
				This is the same as keyboardNav in {@link glow.ui.Focusable}.
		
		@example
			// Make an input auto-complete from an array of tags for a recipe database
			glow.ui.AutoSuggest()
				.data(['Vegetarian', 'Soup', 'Sandwich', 'Wheat-free', 'Organic', 'etc etc'])
				.linkToInput('#recipeTags');
			
		@example
			// An AutoSuggest embedded in the page, rather than in an overlay
			var myAutoSuggest = glow.ui.AutoSuggest()
				.data('recipe.php?ingredients={val}')
				.linkToInput('#username', {
					// don't use an overlay, we'll add the autosuggest to the document outselves
					useOverlay: false
				});
			
			// add the results into the document
			myAutoSuggest.container.appendTo('#results');
			
		@example
			// Make an input suggest from an array of program names, where the
			// whole string is searched rather than just the start
			// When the item is clicked, we go to a url
			new glow.ui.AutoSuggest().setFilter(function(item, val) {
				return item.name.indexOf(val) !== -1;
			}).data([
				{name: 'Doctor Who', url: '...'},
				{name: 'Eastenders', url: '...'},
				{name: 'The Thick Of It', url: '...'},
				// ...
			]).linkToInput('#programSearch').on('select', function(event) {
				location.href = event.selected.url;
			});
	*/
	function AutoSuggest(opts) {
		/*!debug*/
			if (opts !== undefined && typeof opts !== 'object') {
				glow.debug.warn('[wrong type] glow.ui.AutoSuggest expects object as "opts" argument, not ' + typeof opts + '.');
			}
		/*gubed!*/
		opts = glow.util.apply({
			minLength: 3,
			keyboardNav: 'arrows-y',
			activateFirst: true
		}, opts);

		Widget.call(this, 'AutoSuggest', opts);
		this._init();
	};
	
	glow.util.extend(AutoSuggest, Widget);
	AutoSuggestProto = AutoSuggest.prototype;
	
	/**
		@name glow.ui.AutoSuggest#_loading
		@type boolean
		@description True if the autosuggest is waiting for data.
			This happens when getting data is async, has been requested but not returned.
	*/
	
	/**
		@name glow.ui.AutoSuggest#_pendingFind
		@type string
		@description Pending search string.
			This is populated if find is called while the autoSuggest is _loading.
	*/
	
	/**
		@name glow.ui.AutoSuggest#_data
		@type Object[]
		@description Array of objects, the current datasource for this AutoSuggest.
	*/
	AutoSuggestProto._data = [];
	
	/**
		@name glow.ui.AutoSuggest#_dataFunc
		@type function
		@description Function used for fetching data (potentially) async.
	*/
	
	/**
		@name glow.ui.AutoSuggest#_filter
		@type function
		@description The current filter function.
	*/
	AutoSuggestProto._filter = function(val, caseSensitive) {
		var nameStart = this.name.slice(0, val.length);
		nameStart = caseSensitive ? nameStart : nameStart.toLowerCase();
		
		return nameStart === val;
	};
	
	/**
		@name glow.ui.AutoSuggest#_format
		@type function
		@description The current format function.
	*/
	AutoSuggestProto._format = function(result, val) {
		var text = tmpDiv.text(result.name).html(),
			valStart = text.toLowerCase().indexOf( val.toLowerCase() ),
			valEnd = valStart + val.length;
		
		// wrap the selected portion in <strong>
		// This would be so much easier if it weren't for case sensitivity
		if (valStart !== -1) {
			text = text.slice(0, valStart) + '<em class="AutoSuggest-match">' + text.slice(valStart, valEnd) + '</em>' + text.slice(valEnd)
		}
		
		return text;
	};
	
	/**
		@name glow.ui.AutoSuggest#focusable
		@type glow.ui.Focusable
		@description The focusable linked to this autosuggest.
	*/
	
	// Widget lifecycle phases
	AutoSuggestProto._init = function() {
		WidgetProto._init.call(this);
		// call _build
		this._build();
	}
	
	AutoSuggestProto._build = function() {
		WidgetProto._build.call(this, '<ol></ol>');
		
		var opts = this._opts,
			width = opts.width
			content = this.content;
		
		this.focusable = content.focusable({
			children: '> li',
			keyboardNav: this._opts.keyboardNav,
			setFocus: false,
			activateOnHover: true
		});
		
		width && this.container.width(width);
		
		// call _build
		this._bind();
	}
	
	/**
		@private
		@function
		@description Select listener for the focusable.
			'this' is the AutoSuggest
	*/
	function focusableSelectListener(e) {
		return !this.fire('select', {
			li: e.item,
			item: e.item.data('as_data')
		}).defaultPrevented();
	}
	
	/**
		@private
		@function
		@description Listens for focus moving in the focusable.
			'this' is the autoSuggest
	*/
	function focusableChildActivate(e) {
		var item = e.item,
			focusable = this.focusable;
	}
	
	function returnFalse() { return false; }
	
	AutoSuggestProto._bind = function() {
		var focusable = this.focusable.on('select', focusableSelectListener, this)
			.on('childActivate', focusableChildActivate, this);
		
		this._tie(focusable);
		
		// prevent focus moving on mouse down
		this.container.on('mousedown', returnFalse);
		
		WidgetProto._bind.call(this);
	}
	
	/**
		@name glow.ui.AutoSuggest#setFilter
		@function
		@description Set the function used to filter the dataset for results.
			Overwrite this to change the filtering behaviour.
		
		@param {function} filter Filter function.
			Your function will be passed 2 arguments, the term entered by the user,
			and if the search should be case sensitive. Return true to confirm a match.
			
			'this' will be the item in the dataset to check.
			
			If the search is case-insensitive, the term entered by the user is automatically
			lowercased.
		  
			The default filter will return items where the search term matches the start of their 'name'
			property. If the dataset is simply an array of strings, that string will be used instead of the 'name' property.
		
		@example
			// Search the name property for strings that contain val
			myAutoSuggest.setFilter(function(val, caseSensitive) {
				var name = caseSensitive ? this.name : this.name.toLowerCase();
				return name.indexOf(val) !== -1;
			});
			
		@example
			// Search the tags property for strings that contain val surrounded by pipe chars
			// this.tags is like: |hello|world|foo|bar|
			myAutoSuggest.setFilter(function(val, caseSensitive) {
				var tags = caseSensitive ? this.tags : this.tags.toLowerCase();
				return tags.indexOf('|' + val + '|') !== -1;
			});
			
		@return this
	*/
	AutoSuggestProto.setFilter = function(filter) {
		/*!debug*/
			if (arguments.length !== 1) {
				glow.debug.warn('[wrong count] glow.ui.Autosuggest#setFilter expects 1 argument, not ' + arguments.length + '.');
			}
		/*gubed!*/
		this._filter = filter;
		return this;
	};
	
	/**
		@name glow.ui.AutoSuggest#setFormat
		@function
		@description Control how matches are output.
			
		@param {function} formatter Function to generate output.
			The first param to your function will be the matched item from your data list.
			The second param is the search value.
			
			Return an HTML string or glow.NodeList to display this item in the results
			list. Ensure you escape any content you don't want treated as HTML.
			
		@returns this
		
		@example
			// A username auto-complete
			
			// The data url returns a JSON object like [{name='JaffaTheCake', fullName:'Jake Archibald', photo:'JaffaTheCake.jpg'}, ...]
			glow.ui.AutoSuggest().setFormat(function() {
				// Format the results like <img src="JaffaTheCake.jpg" alt=""> Jake Archibald (JaffaTheCake)
				return '<img src="' + data.photo + '" alt=""> ' + data.fullName + ' (' + data.name + ')';
			}).data('userSearch.php?usernamePartial={val}').linkToInput('#username');
	*/
	AutoSuggestProto.setFormat = function(formatter) {
		/*!debug*/
			if (arguments.length !== 1) {
				glow.debug.warn('[wrong count] glow.ui.Autosuggest#setFormat expects 1 argument, not ' + arguments.length + '.');
			}
		/*gubed!*/
		this._format = formatter;
		return this;
	};
	
	/**
		@private
		@function
		@description Process the data into an acceptable format for #_data.
		@param {glow.ui.AutoSuggest} autoSuggest
		@param {Object[]|string[]|glow.net.XhrResponse} data
			Array of strings will be converted into an array of objects like {name: val}
			
			glow.net.XhrResponse will be converted into Object[]|string[] via .json
	*/
	function populateData(autoSuggest, data) {
		var i,
			tmpData,
			event = autoSuggest.fire('data', {data:data});

		if ( !event.defaultPrevented() ) {
			// a listener may have altered the data
			data = event.data;

			// if it's an XHR response, convert it to json
			if (data instanceof glow.net.XhrResponse) {
				data = data.json();
			}
			
			if (typeof data[0] === 'string') {
				tmpData = [];
				i = data.length;
				while (i--) {
					tmpData[i] = { name: data[i] };
				}
				data = tmpData;
			}
			
			/*!debug*/
				if ( !data.push ) {
					glow.debug.warn('[wrong type] glow.ui.Autosuggest data expected to be array, not ' + typeof data + '.');
				}
				else if (data.length && typeof data[0] !== 'object') {
					glow.debug.warn('[wrong type] glow.ui.Autosuggest data expected to be array of objects, not array of ' + typeof data[0] + '.');
				}
			/*gubed!*/
			
			autoSuggest._data = data;
		}
	}
	
	
	
	/**
		@private
		@function
		@description Create _dataFunc based on a custom function.
		@param {glow.ui.AutoSuggest} autoSuggest Instance
		@param {function} func Data fetching function provided by the user via #data
	*/
	function setDataFunction(autoSuggest, func) {
		// create a new function for fetching data
		autoSuggest._dataFunc = function(val) {
			var input = autoSuggest.input,
				bindOpts = autoSuggest._bindOpts,
				loadingClass = (bindOpts && bindOpts.loadingClass) || '';
			
			// put us in the loading state and call the user's function
			autoSuggest._loading = true;
			input.addClass(loadingClass);
			
			// call the user's function, providing a callback
			func.call(this, val, function(data) {
				var pendingFind = autoSuggest._pendingFind;
				autoSuggest._loading = false;
				input.removeClass(loadingClass);
				// populate data if we've been given some
				data && populateData(autoSuggest, data);
				if (pendingFind) {
					performFind(autoSuggest, pendingFind);
					autoSuggest._pendingFind = undefined;
				}
			});
		}
	}
	
	/**
		@private
		@function
		@description Creates a data function to load a single url once.
		@param url With no {val} placeholder.
	*/
	function singleLoadUrl(url) {
		var dataFetched,
			currentRequest;
		
		return function(val, callback) {
			// if we've already fetched the data, just call back & return
			if (dataFetched) {
				return callback();
			}
			
			// if we've already sent a request off, just let that one continue
			if ( !currentRequest ) {				
				currentRequest = glow.net.get(url).on('load', function(response) {
					// set data for quick retrieval later
					dataFetched = 1;
					callback(response);
				});
			}
		}
	}
	
	/**
		@private
		@function
		@description Creates a data function to load from a url each time a search is made.
		@param url With {val} placeholder.
	*/
	function multiLoadUrl(url) {
		var currentRequest;
		
		return function(val, callback) {
			var processedUrl = glow.util.interpolate(url, {val:val});
			
			// abort any current request
			currentRequest && currentRequest.abort();
			currentRequest = glow.net.get(processedUrl).on('load', function(response) {
				callback(response);
			});
		}
	}
	
	/**
		@name glow.ui.AutoSuggest#data
		@function
		@description Set the data or datasource to search.
			This gives the AutoSuggest the data to search, or the means to fetch
			the data to search.
			
		@param {string|string[]|Object[]|glow.net.Response|function} data Data or datasource.
		
			<p><strong>String URL</strong></p>
			
			A URL on the same domain can be provided, eg 'results.json?search={val}', where {val} is replaced
			with the search term. If {val} is used, the URL if fetched on each search, otherwise it is only fetched
			once on the first search.
			
			The result is a {@link glow.net.XhrResponse}, by default this is decoded as json. Use
			the 'data' event to convert your incoming data from other types (such as XML).
			
			<p><strong>glow.net.XhrResponse</strong></p>
			
			This will be treated as a json response and decoded to string[] or Object[], see below.
			
			<p><strong>string[] or Object[] dataset</strong></p>
			
			An Array of strings can be provided. Each string will be converted to {name: theString}, leaving
			you with an array of objects.
			
			An Array of objects can be provided, each object is an object that can be matched. By default
			the 'name' property of these objects is searched to determine a match, but {@link glow.ui.AutoSuggest#filter filter} can
			be used to change this.
			
			<p><strong>function</strong></p>
			
			Providing a function means you have total freedom over fetching the data
			for your autoSuggest, sync or async.
			
			Your function will be called by the AutoSuggest whenever a {@link glow.ui.AutoSuggest#find find}
			is performed, and will be passed 2 arguments: the search
			string and a callback.
			
			You can fetch the data however you wish. Once you have the data, pass it
			to the callback to complete the {@link glow.ui.AutoSuggest#find find}.
			Until the callback is called, the AutoSuggest remains in a 'loading' state.
			
			`this` inside the function refers to the AutoSuggest instance.
			
			Your function will be called multiple times, ensure you cancel any existing
			requests before starting a new one.
			
		@example
			// providing a URL
			myAutoSuggest.data('/search?text={val}');
			
		@example
			// providing an array of program names
			myAutoSuggest.data( ['Doctor Who', 'Eastenders', 'The Thick of it', 'etc etc'] );
			
		@example
			// providing an object of user data
			myAutoSuggest.data([
				{name='JaffaTheCake', fullName:'Jake Archibald', photo:'JaffaTheCake.jpg'},
				{name='Bobby', fullName:'Robert Cackpeas', photo:'Bobby.jpg'}
				...
			]);
			
		@example
			// Getting the data via jsonp
			var request;
			
			myAutoSuggest.data(function(val, callback) {
				// abort previous request
				request && request.abort();
				
				request = glow.net.getJsonp('http://blah.com/data?callback={callback}&val=' + val)
					.on('load', function(data) {
						callback(data);
					})
			});
			
		@returns this
	*/
	AutoSuggestProto.data = function(data) {
		/*!debug*/
			if (arguments.length !== 1) {
				glow.debug.warn('[wrong count] glow.ui.Autosuggest#data expects 1 argument, not ' + arguments.length + '.');
			}
		/*gubed!*/
		if (typeof data === 'string') {
			// look for urls without {val}, they get their data once & once only
			if (data.indexOf('{val}') === -1) {
				// replace data with function
				data = singleLoadUrl(data);
			}
			// look for urls with {val}, they get their data on each search
			else {
				// replace data with function
				data = multiLoadUrl(data);
			}
		}
		
		if (typeof data === 'function') {
			setDataFunction(this, data);
		}
		else if (data.push) {
			// clear any data functions set
			this._dataFunc = undefined;
			populateData(this, data);
		}
		
		return this;
	};
	
	/**
		@private
		@function
		@description Generate the output of a find
			
		@param {glow.ui.AutoSuggest} autoSuggest
		@param {Object[]} results Array of filtered results
		@param {string} val The search string
	*/
	function generateOutput(autoSuggest, results, val) {
		var content = autoSuggest.content,
			resultsLen = results.length,
			i = resultsLen,
			listItem,
			itemContent,
			opts = autoSuggest._opts,
			focusable = autoSuggest.focusable;
		
		focusable.active(false);
		
		// if we've got an overlay, we don't bother clearing the list,
		// just hide the overlay to let it animate away nicely
		if ( !resultsLen && autoSuggest.overlay ) {
			autoSuggest._hideOverlay();
			return;
		}
		
		// remove any current results
		content.children().destroy();
		
		while (i--) {
			itemContent = autoSuggest._format( results[i], val );
			listItem = glow('<li class="AutoSuggest-item"></li>')
				.data( 'as_data', results[i] )
				.prependTo(content);
			
			// append HTML or nodes
			(typeof itemContent === 'string') ?
				listItem.html(itemContent) :
				listItem.append(itemContent);
		}
		
		// Activate the focusable if we have results
		if (resultsLen) {
			opts.activateFirst && focusable.active(true);
			// show & position our overlay
			autoSuggest._showOverlay();
		}
		else {
			autoSuggest._hideOverlay();
		}
	}
	
	/**
		@private
		@function
		@description Performs the find operation without calling _dataFunc.
			Or checking _loading or string length. These are done in #find.
			
		@param {glow.ui.AutoSuggest} autoSuggest
		@param {string} str The search string
	*/
	function performFind(autoSuggest, str) {
		var filteredResults = [],
			filteredResultsLen = 0,
			data = autoSuggest._data,
			findEvent = autoSuggest.fire('find', {val: str}),
			resultsEvent,
			caseSensitive = autoSuggest._opts.caseSensitive;
		
		if ( !findEvent.defaultPrevented() ) {
			// pick up any changes a listener has made to the find string
			str = findEvent.val;
			
			str = caseSensitive ? str : str.toLowerCase();
			
			// start filtering the data
			for (var i = 0, len = data.length; i < len; i++) {
				if ( autoSuggest._filter.call(data[i], str, caseSensitive) ) {
					filteredResults[ filteredResultsLen++ ] = data[i];
					
					// break if we have enough results now
					if (filteredResultsLen === autoSuggest._opts.maxResults) {
						break;
					}
				}
			}
			
			// fire result event
			resultsEvent = autoSuggest.fire('results', {results: filteredResults});
			
			if ( resultsEvent.defaultPrevented() ) {
				filteredResults = [];
			}
			else {
				// pick up any changes a listener has made to the results
				filteredResults = resultsEvent.results
			}
			
			// output results
			generateOutput(autoSuggest, filteredResults, findEvent.val);
		}
	}
	
	/**
		@name glow.ui.AutoSuggest#find
		@function
		@description Search the datasource for a given string
			This fetches results from the datasource and displays them. This
			may be an asyncrounous action if data needs to be fetched from
			the server.
		
		@param {string} str String to search for
			{@link glow.ui.AutoSuggest#filter AutoSuggest#filter} is used
			to determine whether results match or not.
			
		@returns this
	*/
	AutoSuggestProto.find = function(str) {
		/*!debug*/
			if (arguments.length !== 1) {
				glow.debug.warn('[wrong count] glow.ui.Autosuggest#find expects 1 argument, not ' + arguments.length + '.');
			}
		/*gubed!*/
		if (str.length >= this._opts.minLength) {
			// refresh/load data if there's a function
			this._dataFunc && this._dataFunc(str);
			
			// can't find if we're loading...
			if (this._loading) {
				// leave it here, _dataFunc will pick it up and call performFind later
				this._pendingFind = str;
			}
			else {
				performFind(this, str);
			}
		}
		else {
			this.hide();
		}
		return this;
	};
	
	/**
		@name glow.ui.AutoSuggest#hide
		@function
		@description Clear the results so the AutoSuggest is no longer visible
			
		@returns this
	*/
	AutoSuggestProto.hide = function() {
		/*!debug*/
			if (arguments.length !== 0) {
				glow.debug.warn('[wrong count] glow.ui.Autosuggest#hide expects 0 arguments, not ' + arguments.length + '.');
			}
		/*gubed!*/
		clearTimeout(this._inputTimeout);
		// generating empty output does the trick
		generateOutput(this, [], '');
		return this;
	};
	
	/**
		@name glow.ui.AutoSuggest#destroy
		@function
		@description Destroy the AutoSuggest.
			Removes all events that cause the AutoSuggest to run. The input
			element will remain on the page.
	*/
	AutoSuggestProto.destroy = function() {
		/*!debug*/
			if (arguments.length !== 0) {
				glow.debug.warn('[wrong count] glow.ui.Autosuggest#destroy expects 0 arguments, not ' + arguments.length + '.');
			}
		/*gubed!*/
		this._data = undefined;
		
		// remove events from the input
		this.input.detach('keypress', this._inputPress)
			.detach('blur', this._inputBlur)
			.detach('onbeforedeactivate', this._inputDeact);
		
		WidgetProto.destroy.call(this);
	};
	
	/**
		@name glow.ui.AutoSuggest#disabled
		@function
		@description Enable/disable the AutoSuggest, or get the disabled state
			When the AutoSuggest is disabled it is not shown.
			
		@param {boolean} [newState] Disable the AutoSuggest?
			'false' will enable a disabled AutoSuggest.
		
		@returns {glow.ui.AutoSuggest|boolean}
			Returns boolean when getting, AutoSuggest when setting
	*/
	
	/**
		@name glow.ui.AutoSuggest#event:data
		@event
		@description Fired when the dataset changes
			This can be the result of calling {@link glow.ui.AutoSuggest#data data} or
			new data has been fetched from the server.
			
			You can use this event to intercept and transform data into the
			correct JSON format.
			
			Cancel this event to ignore the new dataset, and continue
			with the current one.
		@param {glow.events.Event} event Event Object
		@param {*} event.data The new dataset
			You can modify / overwrite this property to alter the dataset.
			
			The type of this object depends on the data source and other listeners
			which may have overwritten / changed the original data.
			
		@example
			myAutoSuggest.data('data.xml?search={val}').on('data', function(event) {
				// When providing a url to .data(), event.data is a glow.net.XhrResponse object 
				// Note: xmlToJson is not a function defined by Glow
				event.data = xmlToJson( event.data.xml() );
			});
	*/
	
	/**
		@name glow.ui.AutoSuggest#event:results
		@event
		@description Fired when the dataset has been filtered but before HTML is output
			You can use this event to sort the dataset and/or add additional items
			
			Cancelling this event is equivalent to setting event.results to an
			empty array.
		@param {glow.events.Event} event Event Object
		@param {string[]|Object[]} event.results The filtered dataset
			You can modify / overwrite this property to alter the results
			
		@example
			myAutoSuggest.on('results', function(event) {
				// sort results by an 'author' property
				event.results = event.results.sort(function(a, b) {
					return a.author > b.author ? 1 : -1;
				});
				
				// Add a 'More...' item to the data set
				event.results.push( {name:'More...'} );
				
				// Behaviour will be added into the 'select' listener to handle what
				// happens when 'More...' is selected
			});
	*/
	
	/**
		@name glow.ui.AutoSuggest#event:select
		@event
		@description Fired when an item in the AutoSuggest is selected.
			You can use this event to react to the user interacting with
			the AutoSuggest
			
			Cancel this event to prevent the default click action.
		@param {glow.events.Event} event Event Object
		@param {string|Object} event.item The item in the dataset that was selected
		@param {glow.NodeList} event.li The list item in the AutoSuggest that was selected
			
		@example
			myAutoSuggest.on('select', function(event) {
				// this assumes our data objects have a 'url' property
				loaction.href = event.item.url;
			});
	*/
	
	/**
		@name glow.ui.AutoSuggest#event:find
		@event
		@description Fired when a search starts.
			Cancel this event to prevent the search.
		
		@param {glow.events.Event} event Event Object.
		@param {string} event.val The search string.
			You can set this to another value if you wish.
	*/
	
	// EXPORT
	glow.ui.AutoSuggest = AutoSuggest;
});
Glow.provide(function(glow) {
	var undefined,
		AutoSuggestProto = glow.ui.AutoSuggest.prototype;
	
	/**
		@name glow.ui.AutoSuggest#bindOpts
		@type Object
		@description The options object passed into #bindInput, with defaults added.
	*/
	
	/**
		@name glow.ui.AutoSuggest#input
		@type glow.NodeList
		@description Refers to the input element to which this is linked to, or an empty NodeList.
			Link an input to an AutoSuggest using {@link glow.ui.AutoSuggest#bindInput bindInput}.
	*/
	AutoSuggestProto.input = glow();
	
	/**
		@name glow.ui.AutoSuggest#overlay
		@type glow.ui.Overlay
		@description The overlay linked to this autosuggest.
			The Overlay is created when {@link glow.ui.AutoSuggest#bindInput bindInput} is
			called.
	*/
	
	/**
		@name glow.ui.AutoSuggest#_inputPress
		@private
		@function
		@description Listener for input's keypress event.
			'this' is the AutoSuggest.
			
			Needed to make this pseudo-private so we could remove the listener later
	*/
	function inputPress(e) {
		var autoSuggest = this,
			focusable = autoSuggest.focusable,
			focusableActive,
			focusableIndex = focusable.activeIndex,
			childrenLength;
			
		// we only care about printable chars and keys that modify input
		if ( e.keyChar || e.key === 'delete' || e.key === 'backspace' ) {
			// look out for printable chars going into the input
			clearTimeout(autoSuggest._inputTimeout);
			
			autoSuggest._inputTimeout = setTimeout(function() {
				autoSuggest.find( getFindValue(autoSuggest) );
			}, autoSuggest._bindOpts.delay * 1000);
		}
		else {
			focusableActive = focusable.active();
			switch (e.key) {
				case 'escape':
					autoSuggest.hide();
					deleteSelectedText(autoSuggest);
					return false;
				case 'up':
					// Is up being pressed on the first item?
					if (focusableActive && !focusableIndex) {
						// deactivate the focusable
						focusable.active(false);
						deleteSelectedText(autoSuggest);
						return false;
					}
					// I'm deliberately not breaking here, want to capture both up & down keys in the next case
				case 'down':
					if ( !focusableActive && (childrenLength = autoSuggest.content.children().length) ) {
						// if the focusable isn't active, activate the first/last item
						focusable.active(e.key == 'up' ? childrenLength - 1 : 0);
						e.stopPropagation();
						return false;
					}
			}
		}
	}
	AutoSuggestProto._inputPress = inputPress;
	
	/**
		@private
		@function
		@description Gets the value to find from the input.
		@returns The value to find.
			This is the same as the input value unless delimiters are used
	*/
	function getFindValue(autoSuggest) {
		var input = autoSuggest.input,
			delim = autoSuggest._bindOpts.delim,
			val = input.val(),
			lastDelimPos,
			caretPos;
		
		// deal with delims
		if (delim) {
			caretPos = getCaretPosition(autoSuggest);
			// get the text before the caret
			val = val.slice(0, caretPos);
			// is there a delimiter before the caret?
			lastDelimPos = val.lastIndexOf(delim);
			// if so, ignore the bits before the caret
			if (lastDelimPos !== -1) {
				val = val.slice( val.lastIndexOf(delim) + delim.length );
			}
		}
		
		return glow.util.trim(val);
	}
	
	/**
		@name glow.ui.AutoSuggest#_inputBlur
		@private
		@function
		@description Listener for input's blur event.
			'this' is the AutoSuggest.
			
			Needed to make this pseudo-private so we could remove the listener later
	*/
	function inputBlur() {
		this.hide();
	}
	AutoSuggestProto._inputBlur = inputBlur;
	
	/**
		@name glow.ui.AutoSuggest#_inputDeact
		@private
		@function
		@description Listener for input's beforedeactivate event.
			'this' is the AutoSuggest.
			
			Prevents IE from bluring the input element when the autosuggest is clicked.
			
			Needed to make this pseudo-private so we could remove the listener later
	*/
	function inputDeact(e) {
		if ( this.container.contains( e.related ) ) {
			return false;
		}
	}
	AutoSuggestProto._inputDeact = inputDeact;
	
	/**
		@private
		@function
		@description Listener for AutoSuggest's select event if opts.autoComplete is true
			This creates the autoComplete behaviour.
			'this' is the AutoSuggest.
	*/
	function completeSelectListener(event) {
		completeInput(this.hide(), event.item.name);
		makeSelection(this, this.input.val().length);
	}
	
	/**
		@private
		@function
		@description Listener for focusable's childActivate event if opts.autoComplete is true.
			This updates the text as the user cycles through items.
		
			'this' is the AutoSuggest
	*/
	function focusablechildActivate(event) {
		if (event.method !== 'hover') {
			completeInput(this, event.item.data('as_data').name, true);
		}
	}
	
	/**
		@private
		@function
		@description Autocomplete value in the input.
		@param {glow.ui.AutoSuggest} autoSuggest
		@param {string} newVal Value to complete to
		@param {boolean} [select=false] Highlight the completed portion?
			This is used while cycling through values
	*/
	function completeInput(autoSuggest, newVal, select) {
		deleteSelectedText(autoSuggest);

		var input = autoSuggest.input,
			oldVal = input.val(),
			caretPos = getCaretPosition(autoSuggest),
			rangeStart = caretPos,
			rangeEnd = newVal.length,
			delim = autoSuggest._bindOpts.delim,
			lastDelimPos,
			firstValPart = '';
		
		// we don't want to overwrite the whole thing if we're using delimiters
		if (delim) {
			lastDelimPos = oldVal.slice(0, caretPos).lastIndexOf(delim);
			if (lastDelimPos !== -1) {
				firstValPart = oldVal.slice(0, lastDelimPos) + delim + ' ';
			}
			newVal = firstValPart + newVal + delim + ' ';
			rangeEnd = newVal.length;
			newVal += oldVal.slice(caretPos);
		}
		input.val(newVal);
		select && makeSelection(autoSuggest, rangeStart, rangeEnd);
	}
	
	
	/**
		@private
		@function
		@description Make a selection in the bound input
		
		@param {glow.ui.AutoSuggest} autoSuggest
		@param {number} start Start point of the selection
		@param {number} [end=start] End point of the selection
	*/
	function makeSelection(autoSuggest, start, end) {
		end = (end === undefined) ? start : end;
		
		var inputElm = autoSuggest.input[0],
			character = 'character',
			range;

		if (!window.opera && inputElm.createTextRange) { // IE
			range = inputElm.createTextRange();
			range.moveStart(character, start);
			range.moveEnd(character, end - inputElm.value.length);
			range.select();
		}
		else { // moz, saf, opera
			inputElm.select();
			inputElm.selectionStart = start;
			inputElm.selectionEnd = end;
		}
	}
	
	/**
		@private
		@function
		@description Get the caret position within the input
	*/
	function getCaretPosition(autoSuggest) {
		var inputElm = autoSuggest.input[0],
			r;
		
		if (glow.env.ie) { // IE
			range = document.selection.createRange();
			range.collapse();
			range.setEndPoint( 'StartToStart', inputElm.createTextRange() );
			r = range.text.length;
		}
		else { // moz, saf, opera
			r = inputElm.selectionStart;
		}
		
		return r;
	}
	
	/**
		@private
		@function
		@description Delete the currently selected text in the input.
			This is used when esc is pressed and in focusablechildActivate
	*/
	function deleteSelectedText(autoSuggest) {
		var inputElm = autoSuggest.input[0],
			val = inputElm.value,
			selectionStart;
		
		if (glow.env.ie) { // IE
			document.selection.createRange().text = '';
		}
		else { // others
			selectionStart = inputElm.selectionStart;
			inputElm.value = val.slice(0, selectionStart) + val.slice(inputElm.selectionEnd);
			inputElm.selectionStart = selectionStart;
		}
	}
	
	/**
		@name glow.ui.AutoSuggest#_showOverlay
		@private
		@function
		@description Shows the overlay, if one is attached.
			Also positions the overlay according to options set.
	*/
	AutoSuggestProto._showOverlay = function() {
		var overlay = this.overlay,
			autoSuggestOpts = this._opts,
			bindOpts = this._bindOpts,
			input = this.input,
			inputOffset;
		
		if (!overlay) { return; }
		
		if (!autoSuggestOpts.width) {
			this.container.width( input[0].offsetWidth );
		}
		
		if (bindOpts.autoPosition) {
			inputOffset = input.offset();
			overlay.container.css({
				top: inputOffset.top + input[0].offsetHeight,
				left: inputOffset.left
			})
		}
		
		overlay.show();
	}
	
	/**
		@name glow.ui.AutoSuggest#_hideOverlay
		@private
		@function
		@description Hide the overlay, if one is attached.
	*/
	AutoSuggestProto._hideOverlay = function() {
		var overlay = this.overlay;
		overlay && overlay.hide();
	}
	
	/**
		@name glow.ui.AutoSuggest#bindInput
		@function
		@description Link this autosuggest to a text input.
			This triggers {@link glow.ui.AutoSuggest#find} when the value in
			the input changes.
			
			The AutoSuggest is placed in an Overlay beneath the input and displayed
			when results are found.
			
			If the input loses focus, or esc is pressed,
			the Overlay will be hidden and results cleared.
			
		@param {selector|glow.NodeList|HTMLElement} input Test input element
		
		@param {Object} [opts] Options
		@param {selector|glow.NodeList} [opts.appendTo] Add the AutoSuggest somewhere in the document rather than an {@link glow.ui.Overlay Overlay}
			By default, the AutoSuggest will be wrapped in an {@link glow.ui.Overlay Overlay} and
			appended to the document's body.
		@param {boolean} [opts.autoPosition=true] Place the overlay beneath the input
			If false, you need to position the overlay's container manually. It's
			recommended to do this as part of the Overlay's show event, so the
			position is updated each time it appears.
		@param {boolean} [opts.autoComplete=true] Update the input when an item is highlighted & selected.
			This will complete the typed text with the result matched.
			
			You can create custom actions by listening for the
			{@link glow.ui.AutoSuggest#event:select 'select' event}
		@param {string} [opts.delim] Delimiting char(s) for selections.
			When defined, the input text will be treated as multiple values,
			separated by this string (with surrounding spaces ignored).
		@param {number} [opts.delay=0.5] How many seconds to delay before searching.
			This prevents searches being made on each key press, instead it
			waits for the input to be idle for a given number of seconds.
		@param {string} [opts.anim] Animate the Overlay when it shows/hides.
			This can be any parameter accepted by {@link glow.ui.Overlay#setAnim Overlay#setAnim}.
		@param {string} [opts.loadingClass] Class name added to the input while data is being loaded.
			This can be used to change the display of the input element while data is being
			fetched from the server. By default, a spinner is displayed in the input.
		
		
		@returns this
	*/
	AutoSuggestProto.bindInput = function(input, opts) {
		/*!debug*/
			if (arguments.length < 1 || arguments.length > 2) {
				glow.debug.warn('[wrong count] glow.ui.AutoSuggest#bindInput expects 1 or 2 arguments, not ' + arguments.length + '.');
			}
			if (opts !== undefined && typeof opts !== 'object') {
				glow.debug.warn('[wrong type] glow.ui.AutoSuggest#bindInput expects object as "opts" argument, not ' + typeof opts + '.');
			}
		/*gubed!*/
		var bindOpts = this._bindOpts = glow.util.apply({
				autoPosition: true,
				autoComplete: true,
				delay: 0.5,
				loadingClass: 'glow200b1-AutoSuggest-loading'
			}, opts),
			appendTo = bindOpts.appendTo,
			container = this.container,
			overlay,
			autoSuggestOpts = this._opts;
			
		// if autocomplete isn't turned off, the browser doesn't let
		// us hear about up & down arrow presses
		this.input = glow(input).attr('autocomplete', 'off')
			.on('keypress', inputPress, this)
			.on('blur', inputBlur, this)
			.on('beforedeactivate', inputDeact, this);
		
		if (bindOpts.autoComplete) {
			this.on('select', completeSelectListener, this)
				.focusable.on('childActivate', focusablechildActivate, this);
		}
		
		// add to document, or...
		if (appendTo) {
			glow(appendTo).append(container);
		}
		// ...make overlay
		else {
			this.overlay = overlay = new glow.ui.Overlay(container)
				.on('hide', overlayHide, this)
				.on('afterShow', overlayAfterShow, this)
				.hide();
			
			// the overlay will reactivate the focusable when needed
			this.focusable.disabled(true);
			
			overlay.container.appendTo(document.body);
			
			// use alternate slide anim
			if (bindOpts.anim === 'slide') {
				bindOpts.anim = altSlideAnim;
			}
			
			bindOpts.anim && overlay.setAnim(bindOpts.anim);
			
			this._tie(overlay);
		}
		
		return this;
	};
	
	/**
		@private
		@function
		@description Alternative slide animation.
			The AutoSuggest uses a different style of slide animation to the
			usual Overlay, this creates it.
	*/
	function altSlideAnim(isShow, callback) {
		var anim,
			container = this.container,
			animOpts = {
				lockToBottom: true
			};
		
		if (isShow) {
			container.height(0);
			anim = container.slideOpen(0.5, animOpts).data('glow_slideOpen')
		}
		else {
			anim = container.slideShut(0.5, animOpts).data('glow_slideShut');
		}
		
		anim.on('complete', callback);
	}
	
	/**
		@private
		@function
		@description Listener for overlay hide.
			'this' is the autoSuggest.
			
			This stops the focusable being interactive during its hide & show animation.
	*/
	function overlayHide() {
		this.focusable.disabled(true);
	}
	
	/**
		@private
		@function
		@description Listener for overlay show.
			'this' is the autoSuggest
	*/
	function overlayAfterShow() {
		var focusable = this.focusable;
		
		focusable.disabled(false);
		
		if (this._opts.activateFirst) {
			focusable.active(true);
		}
	}
});
Glow.provide(function(glow) {
	var undefined,
		CarouselPaneProto,
		WidgetProto = glow.ui.Widget.prototype;
	
	/**
		@name glow.ui.CarouselPane
		@class
		@extends glow.ui.Widget
		@description Create a pane of elements that scroll from one to another.
			This is a component of Carousel.
			
		@param {glow.NodeList|selector|HTMLElement} container Container of the carousel items.
			The direct children of this item will be treated as carousel items. They will
			be positioned next to each other horizontally.
			
			Each item takes up the same horizontal space, equal to the width of the largest
			item (including padding and border) + the largest of its horizontal margins (as set in CSS).
			
			The height of the container will be equal to the height of the largest item (including
			padding and border) + the total of its vertical margins.
			
		@param {object} [opts] Options
			@param {number} [opts.duration=0.2] Duration of scrolling animations in seconds.
			@param {string|function} [opts.tween='easeBoth'] Tween to use for animations.
				This can be a property name of {@link glow.tweens} or a tweening function.
			
			@param {boolean | number} [opts.step=1] Number of items to move at a time.
				If true, the step will be the same size as the spotlight.
			@param {boolean} [opts.loop=false] Loop the carousel from the last item to the first.
			@param {boolean} [opts.page=false] Keep pages in sync by adding space to the end of the carousel.
				Spaces don't exist as physical HTML elements, but simply a gap from the last item
				to the end.
			
			@param {number} [opts.spotlight] The number of items to treat as main spotlighted items.
				A carousel may be wide enough to display 2 whole items, but setting
				this to 1 will result in the spotlight item sitting in the middle, with
				half of the previous item appearing before, and half the next item
				appearing after.
				
				By default, this is the largest number of whole items that can exist in
				the width of the container. Any remaining width will be used to partially
				show the previous/next item.
				
		@example
			new glow.ui.CarouselPane('#carouselItems', {
				duration: 0.4,
				step: 2,
				loop: true
			});
	*/
	function CarouselPane(container, opts) {
		/*!debug*/
			if (!container) {
				glow.debug.warn('[wrong count] glow.ui.CarouselPane - argument "container" is required.');
				return;
			}
			
			if (!container || glow(container).length === 0) {
				glow.debug.warn('[invalid configuration] glow.ui.CarouselPane - "'+container+'" is not a valid element specifier for the container.');
			}
			
			if (opts && opts.spotlight && opts.step && opts.spotlight < opts.step && opts.step !== true) {
				glow.debug.warn('[invalid configuration] glow.ui.CarouselPane - opts.step (' + opts.step +') cannot be greater than opts.spotlight ('+ opts.spotlight + ').');
			}
			
			if (opts && opts.spotlight && opts.step && opts.page && opts.spotlight !== opts.step && opts.step !== true) {
				glow.debug.warn('[invalid configuration] glow.ui.CarouselPane - opts.step (' + opts.step +') cannot be different than opts.spotlight ('+ opts.spotlight + ') if opts.page is true.');
			}
		/*gubed!*/
		
		var that = this;
		
		opts = glow.util.apply({
			duration: 0.2,
			tween: 'easeBoth',
			step: 1,
			loop: false,
			page: false, // add a gap?
			axis: 'x'    // either 'x' or 'y'
		}, opts || {});

		glow.ui.Widget.call(this, 'CarouselPane', opts);
		
		
		if (glow(container).length > 0) { this._init(container, opts); }
	};
	
	glow.util.extend(CarouselPane, glow.ui.Widget); // CarouselPane is a Widget
	CarouselPaneProto = CarouselPane.prototype;     // shortcut
	
	
	
	/**
		Tracks the order and location of all items, including cloned items.
		@private
		@constructor
		@param {glow.NodeList} nodeList The real items to track.
	 */
	function ItemList(nodeList) {
		var thisMeta;
		
		this.range = {min: 0, max: 0};
		this.items = {};
		this.meta = {};
		
		for (var i = 0, leni = nodeList.length; i < leni; i++) {
			this.addItem(i, nodeList.item(i));
		}
	}
	
	ItemList.prototype.addItem = function(index, item, meta) {/*debug*///console.log('ItemList.prototype.addItem('+index+')');
		this.range.min = Math.min(this.range.min, index);
		this.range.max = Math.max(this.range.max, index);
		
		this.items[index] = item;
		this.meta[index] = meta || {};
	}
	
	ItemList.prototype.addMeta = function(index, meta) {/*debug*///console.log('ItemList.prototype.addMeta('+index+', '+meta.offset+')');
		if (this.meta[index]) {
			this.meta[index] = glow.util.apply(this.meta[index], meta);
		}
	}
	
	ItemList.prototype.place = function(top, left) {
		// TODO styleName = this._geom[1]
		for (var p in this.items) {
			if (top !== undefined ) this.items[p].css('top', top);
			this.items[p].css('left', (left === undefined)? this.meta[p].offset : left);
		}
	}
	
	ItemList.prototype.dump = function(c) {
		if (typeof console !== 'undefined') {
			for (var i = c._itemList.range.min, maxi = c._itemList.range.max; i <= maxi; i++) {
				if (c._itemList.meta[i]) {
					console.log('>> '+ i + ': ' + (c._itemList.meta[i].isClone? 'clone':'real') + ' at ' + c._itemList.meta[i].offset + ' ' + c._itemList.items[i][0].children[0].alt);
				}
				else {
					console.log('>> '+ i + ': ' + c._itemList.meta[i]);
				}
			}
		}
	}
	
	ItemList.prototype.swap = function(index1, index2) { /*debug*///console.log('ItemList.prototype.swap('+index1+', '+index2+')');
		this.items[index1].css('left', this.meta[index2].offset);
		this.items[index2].css('left', this.meta[index1].offset);
	}
	
	CarouselPaneProto._init = function(container) { /*debug*///console.log('CarouselPaneProto._init');
		WidgetProto._init.call(this);
		
		// used value vs configured value (they may not be the same). Might be set to spotlight capacity, in _build.
		this._step = this._opts.step;
		
		this._geom = (this._opts.axis === 'y')? ['height', 'top'] : ['width', 'left'];
		
		/**
			@name glow.ui.CarouselPane#stage
			@type glow.NodeList
			@description The container passed in to the constructor for glow.ui.CarouselPane.
		*/
		this.stage = glow(container).item(0);

		this._focusable = this.stage.focusable( {children: '> *', loop: true, setFocus: true} );
		
		
		// what would have been the "content" of this widget, is named "viewport"
		this._viewport = glow('<div class="CarouselPane-viewport"></div>');
		glow(this.stage).wrap(this._viewport);
		
		/**
			@name glow.ui.CarouselPane#items
			@type glow.NodeList
			@description Carousel items.
				This is the same as `myCarouselPane.stage.children()`
		*/
		this.items = this.stage.children();
		this._itemList = new ItemList(this.items);
		
		if (this._opts.spotlight > this.items.length) {
			/*!debug*/
				glow.debug.warn('[invalid configuration] glow.ui.CarouselPane - opts.spotlight (' + this._opts.spotlight +') cannot be greater than the number of items ('+ this.items.length + ').');
			/*gubed!*/
			this._opts.spotlight = this.items.length;
		}
		
		// track what the offset of the current leftmost spotlighted item is
		this._index = 0;
		
		this._build();
	}
	
	CarouselPaneProto._build = function() { /*debug*///console.log('CarouselPaneProto._build');
		WidgetProto._build.call(this, this._viewport);
		
		this.stage.css({
			margin: 0,
			listStyleType: 'none' // useful when content is a list
		});

		this.items.css( {position:'absolute', 'z-index':2} );
		this._itemDimensions = getDimensions(this.items); // get this *after* setting position to absolute
		this.items.css({
			margin: 0,
			width: this._itemDimensions.innerWidth,
			height: this._itemDimensions.innerHeight
		});

		this._wingSize = Math.ceil(this.items.length * this._itemDimensions[this._geom[0]] * 1.5);

		this._viewport.css({
			overflow: 'scroll',
			overflowX: 'hidden', // hide scroll bars
			overflowY: 'hidden',
			position: 'relative',
			padding: 0,
			margin: 0,
			width: this._opts.axis === 'x'? '100%' : this._itemDimensions.width,
			height: this._opts.axis === 'y'? '100%' : this._itemDimensions.height
		});
		
		/**
			@private
			@name glow.ui.CarouselPane#_spot
			@type Object
			@description Information about the spotlight area.
		*/
		this._spot = CarouselPane._getSpot(this._viewport.width(), this.items, this._itemDimensions, this._opts);
		
		/**
			@private
			@name glow.ui.CarouselPane#_step
			@type number
			@description How far to move when going next or prev.
		*/
		if (this._opts.step === true) {
			this._step = this._spot.capacity;
		}
		else if (this._opts.step > this._spot.capacity) {
			/*!debug*/
				glow.debug.warn('[invalid configuration] glow.ui.CarouselPane - opts.step (' + this._opts.step +') cannot be greater than the calculated spotlight ('+ this._spot.capacity + ').');
			/*gubed!*/
			
			this._step = this._spot.capacity;
		}

		if (this._opts.page && this._step !== this._spot.capacity) {
			/*!debug*/
				glow.debug.warn('[invalid configuration] glow.ui.CarouselPane - opts.step (' + this._opts.step +') cannot be different than the spotlight ('+ this._spot.capacity + ') when opts.page is true.');
			/*gubed!*/
			
			this._step = this._spot.capacity;
		}
		
		/**
			@private
			@name glow.ui.CarouselPane#_gap
			@type Object
			@description Information about the gap at the end of the items.
			@property size
			@property count
		*/
		this._gap = getGap(this);
		
		// must set height to anything other than 0, else FF won't *ever* render the stage
		this.stage.css({width: this.stage.width() + this._wingSize * 2, height: '100%'}); // [wing][stage[spot]stage][wing]
		
		layout.call(this);
		
		this._bind();
		
		calculateIndex.call(this);
	}
	
	/**
		@private
		@name getGap
		@description Calculate the size of the empty space at the end of the items
			which may be required to enforce paging.
		@param {glow.ui.CarouselPane} carouselPane
	 */
	function getGap(carouselPane) { /*debug*///console.log('getGap()');
		var gap = { size: 0, count: 0 },
			danglingItemCount = carouselPane.items.length % carouselPane._step;
	
		if (carouselPane._opts.page && carouselPane._step > 1) {
			gap.count = danglingItemCount? carouselPane._spot.capacity - danglingItemCount : 0;
			gap.size = gap.count * carouselPane._itemDimensions[carouselPane._geom[0]];
		}
	
		return gap;
	}
	
	CarouselPaneProto._bind = function() { /*debug*///console.log('CarouselPaneProto._bind');
		var that = this;
		
		WidgetProto._bind.call(that);

		attachEvent(that, that._focusable, 'childActivate', function(e) {
			var itemNumber = e.itemIndex,
				indexes = that.spotlightIndexes(true),
				isVisible = (' '+indexes.join(' ')+' ').indexOf(' '+itemNumber+' ') > -1;

			if (itemNumber !== undefined && !isVisible) {
				that.moveTo(itemNumber, {tween: ''});
				that._index = itemNumber;
			}
		});
		
		this._focusable.on('select', function(e) {
			e.itemIndex = e.item.data('itemIndex');
			that.fire('select', e);
		});
	}
	
	/**
		@private
		@name attachEvent
		@function
		@decription Add an event listener and handler to a node related to this carouselpane.
		Stores a reference to that transaction so each handler can easily be detached later.
		@see glow.ui.CarouselPane-detachEvents
		@param {glow.ui.CarouselPane} carouselPane
		@param {glow.EventTarget} target
		@param {string} name The name of the event to listen for.
		@param {Function} handler
	*/
	function attachEvent(carouselPane, target, name, handler) {
		target.on(name, handler);
		carouselPane._addedEvents = carouselPane._addedEvents || [];
		carouselPane._addedEvents.push( {target:target, name:name, handler:handler} );
	}
	
	/**
		@private
		@name detachEvents
		@function
		@decription Remove all events add via the {@link glow.ui.CarouselPane-attachEvent}.
		@see glow.ui.CarouselPane-removeEvents
		@param {glow.ui.CarouselPane} carouselPane
	*/
	function detachEvents(carouselPane) {
		var i = carouselPane._addedEvents? carouselPane._addedEvents.length : 0,
			e;
		while (i--) {
			e = carouselPane._addedEvents[i];
			e.target.detach(e.name, e.handler);
		}
	}
	
	CarouselPaneProto.updateUi = function() { /*debug*///console.log('updateUi');
		WidgetProto._updateUi.call(this);
		
		// must set height to anything other than 0, else FF won't *ever* render the stage
		this.stage.css({width: this.stage.width() + this._wingSize * 2, height: '100%'}); // [wing][stage[spot]stage][wing]
		
		this._spot = CarouselPane._getSpot(this._viewport.width(), this.items, this._itemDimensions, this._opts);
		
		if (this._opts.step === true) {
			this._step = this._spot.capacity;
		}
		
		layout.call(this);
		
		this._index = 0;
		this.fire('updateUi', {});
	}
	
	/**
		@name glow.ui.CarouselPane#moveStop
		@function
		@description Stop moving the carousel.
			The current animation will end, leaving the carousel
			in step. Note that this is asynchronous: expect this method
			to return before the carousel actually stops.
			
		@returns this
	*/
	CarouselPaneProto.moveStop = function() { /*debug*///console.log('moveStop()');
		// set temporary flag to signal the next animation in the timeline to stop
		this._gliderBrake = true;
	}
	
	/**
		@name glow.ui.CarouselPane#moveStart
		@function
		@description Start moving the carousel in a particular direction.
		
		@param {boolean} [backwards=false] True to move backwards, otherwise move forwards.
		
		@returns this
		@see glow.ui.CarouselPane#moveStop
		
		@example
			nextBtn.on('mousedown', function() {
				myCarouselPane.moveStart();
			}).on('mouseup', function() {
				myCarouselPane.moveStop();
			});
	*/
	CarouselPaneProto.moveStart = function(backwards) { /*debug*///console.log('moveStart('+backwards+')');
		/*!debug*/
			if (arguments.length > 1) {
				glow.debug.warn('[wrong count] glow.ui.moveStart - too many arguments, must be 1 or 0, not '+arguments.length+'.');
			}
		/*gubed!*/
		
		var step = (backwards? -1 : 1) * this._step,
			carouselPane = this;
		
		if (!carouselPane._inMotion) {
			carouselPane._gliderBrake = false;
			
			carouselPane.moveTo(
				carouselPane._index + step,
				{
					callback: function() {
						if (!carouselPane._gliderBrake) {
							if ( // if looping or if there's room to go in the given direction 
								carouselPane._opts.loop ||
								( (backwards && carouselPane._index > 0) || (!backwards && carouselPane._index + carouselPane._spot.capacity < carouselPane.items.length) )
							) {
								if (carouselPane._step === 1) {
									glide.call(carouselPane, backwards);
								}
								else {
									carouselPane.moveStart(backwards); // recursive
								}
							}
						}
					}
				}
			);
		}
		
		return carouselPane;
	}
	
	/**
		@name glow.ui.CarouselPane#moveToggle
		@function
		@description If this CarouselPane is currently moving via moveStart, will call moveStop,
		otherwise will call moveStart.
		@param {boolean} [backwards=false] When calling moveStart, move backwards?
		@returns this
	 */
	CarouselPaneProto.moveToggle = function(backwards) { /*debug*///console.log('moveToggle()');
		/*!debug*/
			if (arguments.length > 1) {
				glow.debug.warn('[wrong count] glow.ui.moveToggle - too many arguments, must be 1 or 0, not '+arguments.length+'.');
			}
		/*gubed!*/
		
		if (this._inMotion && !this._gliderBrake) {
			this.moveStop();
		}
		else {
			this.moveStart(backwards);
		}
		
		return this;
	}
	
	/**
		@private
		@name glide
		@function
		@description Move this using an animation that is continuous, with a linear tween.
		@param {boolean} backwards Glide in a previous-direction?
	 */
	var glide = function(backwards) { /*debug*///console.log('glide('+backwards+')');
		var dir = (backwards? -1 : 1),
			moves = [],
			offset = this.content[0].scrollLeft, // from where is the move starting?
			amount = this._itemDimensions[this._geom[0]], // how many pixels are we moving by?
			from,
			to,
			that = this,
			moveAnim,
			// when to loop back to where we started?
			wrapAt = offset + (backwards? -this._index * amount : (this.items.length - this._index) * amount);
		
		swap.call(this, 'back');

		for (var i = 0, leni = this.items.length; i < leni; i += this._step) {
			// calculate the start and end points of the next move
			from = offset + dir * i * amount;
			to   = offset + dir * (i + this._step) * amount;

			if ( (backwards && from === wrapAt) || (!backwards && to === wrapAt) ) {
				offset -= dir * this.items.length * amount; // wrap
			}

			moveAnim = this.content.anim(
				this._opts.duration,
				{scrollLeft: [from, to]},
				{tween: 'linear', startNow: false}
			)
			.on('start', function() {
				indexMoveTo.call(that);
					
				if ( that.fire('move', { moveBy: dir, currentIndex: that._index }).defaultPrevented() ) {
					glideStop.call(that);
				}
			})
			.on('complete', function() {
				that._index += dir; // assumes move amount will be +/- 1

 				if (
 					that._gliderBrake
 					||
 					( !that._opts.loop && (that._index + that._spot.capacity === that.items.length || that._index === 0) )
 				) {
 					glideStop.call(that);
 					that.fire( 'afterMove', {currentIndex: that._index} );
 				}
			});
			
			moves.push(moveAnim);
		}
		
		this._glider = new glow.anim.Timeline({loop: true});
		glow.anim.Timeline.prototype.track.apply(this._glider, moves);
		
		this._inMotion = true;
		this._gliderBrake = false;
		this._glider.start();
	}
	
	/**
		@private
		@name indexMoveTo
		@function
		@description Calculate what the new index would be and set this._index to that.
		@param {number} index The destination index.
		@returns this._index
		@example
			// items.length is 3
			var newIndex = indexMoveTo(10);
			// newIndex is 1
	 */
	function indexMoveTo(index) {
		if (index !== undefined) { this._index = index; }
		
		// force index to be a number from 0 to items.length
		this._index = this._index % this.items.length;
		while (this._index < 0) { this._index += this.items.length; }
		
		return this._index;
	}
	
	/**
		@private
		@name indexMoveBy
		@function
		@description Calculate what the new index would be and set this._index to that.
		@param {number} delta The amount to change the index by, can be positive or negative.
		@returns this._index
		@example
			// items.length is 3
			// currentIndex is 1
			var newIndex = indexMoveBy(100);
			// newIndex is 2
	 */
	function indexMoveBy(delta) {
		return indexMoveTo.call(this, this._index += delta);
	}
	
	/**
		@private
		@name glideStop
		@description Reset this CarouselPane after a glide is finished.
	 */
	function glideStop() { /*debug*///console.log('glideStop()');
		this._glider.stop();
		this._glider.destroy();
		
		this._inMotion = false;
		this._index = calculateIndex.call(this); // where did we end up?
		
		// in case our clones are showing
		jump.call(this);
		swap.call(this);
	}
	
	/**
		@name glow.ui.CarouselPane#spotlightIndexes
		@function
		@description Gets an array of spotlighted indexes.
			These are the indexes of the nodes within {@link glow.ui.CarouselPane#items}.
			Only item indexes currently visible in the spotlight will be included.
		@private-param {boolean} _real Return only indexes of real items, regardless of what clones are visible.
		@returns {number[]}
	*/
	CarouselPaneProto.spotlightIndexes = function(_real) { /*debug*///console.log('CarouselPaneProto.spotlightIndexes()');
		var indexes = [],
			findex = calculateIndex.call(this),
			index,
			maxi = (this._opts.loop)? this._spot.capacity : Math.min(this._spot.capacity, this.items.length);
		
		// takes into account gaps and wraps
		for (var i = 0; i < maxi; i++) {
			index = _real? (findex + i) : (findex + i)%(this.items.length + this._gap.count);
			// skip gaps
			if (index >= this.items.length || index < 0) {
				continue; // or maybe keep gaps? index = NaN;
			}
			indexes.push(index);
		}		
		return indexes;
	}
	
	/**
		@name glow.ui.CarouselPane#spotlightItems
		@function
		@description Get the currently spotlighted items.
		Only items currently visible in the spotlight will be included.
		@returns {glow.NodeList}
	*/
	CarouselPaneProto.spotlightItems = function() { /*debug*///console.log('CarouselPaneProto.spotlightItems()');
		var items = glow(),
			indexes = this.spotlightIndexes();
		
		// takes into account gaps and wraps
		for (var i = 0, leni = indexes.length; i < leni; i++) {
			items.push( this.items[ indexes[i] ] );
		}
		
		return items;
	}
	
	/**
		@private
		@name calculateIndex
		@function
		@description Calculate the index of the leftmost item in the spotlight.
		@returns {number}
	 */
	function calculateIndex() {
		var cindex = this.content[0].scrollLeft - (this._wingSize +this._spot.offset.left);
		
		cindex += this._spot.offset.left;
		cindex /= this._itemDimensions.width;
		
		return cindex;
	}
	
	/**
		@name glow.ui.CarouselPane#moveTo
		@function
		@description Move the items so a given index is the leftmost active item.
			This method respects the carousel's limits and its step. If it's
			not possible to move the item so it's the leftmost item of the spotlight, it will
			be placed as close to the left as possible.
		
		@param {number} itemIndex Item index to move to.
		
		@param opts
		@param {undefined|string} opts.tween If undefined, use the default animation,
		if empty string then no animation, if non-empty string then use the named tween.
		@privateParam {Function} opts.callback Called when move animation is complete.
		@privateParam {boolean} opts.jump Move without animation and without events.
		
		@returns this
	*/
	CarouselPaneProto.moveTo = function(itemIndex, opts) { /*debug*///glow.debug.log('moveTo('+itemIndex+')');
		var willMove, // trying to move to the same place we already are?
			destination, // in pixels
			tween,
			anim;
		
		if (this._inMotion) {
			return false;
		}
		opts = opts || {};
		
		// will the last item be in the spotlight?
		if (!this._opts.loop && itemIndex > this.items.length - this._spot.capacity) {
			// if opts.page is on then allow a gap at the end, otherwise don't include gap
			itemIndex = this.items.length - this._spot.capacity + (this._opts.page? this._gap.count : 0);
		}
		else if (!this._opts.loop && itemIndex < 0) {
			itemIndex = 0;
		}

		willMove = ( itemIndex !== this._index && canGo.call(this, itemIndex) );
		
		// move event
		if (!opts.jump) { // don't fire move event for jumps
			var e = new glow.events.Event({
				currentIndex: this._index,
				moveBy: (this._index < itemIndex)? (itemIndex - this._index) : (-Math.abs(this._index - itemIndex))
			});
			
			if (!opts.jump && willMove && this.fire('move', e).defaultPrevented() ) {
				return this;
			}
			else {
				itemIndex = this._index + e.moveBy;
			}
		}

		// force items to stay in step when opts.page is on
		if (this._opts.page) {
			itemIndex = Math.floor(itemIndex / this._step) * this._step;
		}
		
		// invalid itemIndex value?
		if (itemIndex > this.items.length + this._step || itemIndex < 0 - this._step) { // moving more than 1 step
			/*!debug*/
				glow.debug.warn('[wrong value]  glow.ui.CarouselPane#moveTo - Trying to moveTo an item ('+itemIndex+') that is more than 1 step (' + this._step +' items) away is not possible.');
			/*gubed!*/
			itemIndex = this._index + (this._index < itemIndex)? -this._step : this._step;
		}

		destination = this._wingSize + itemIndex * this._itemDimensions.width;

		swap.call(this, 'back');
		
		tween = opts.tween || this._opts.tween;
		
		var that = this;
		if (opts.jump === true || opts.tween === '') { // jump
			this.content[0].scrollLeft = destination;
			
			this._index = itemIndex;
			// in case our clones are showing
			jump.call(this);
			swap.call(this);
			
			// force index to be a number from 0 to items.length
			this._index = this._index % (this.items.length  + this._gap.count);
			
			if (!opts.jump && willMove) {
				this.fire('afterMove', {currentIndex: this._index});
			}
			
			this._inMotion = false;
		}
		else if (willMove) {
			this._inMotion = true;
			
			anim = this.content.anim(
				this._opts.duration,
				{
					scrollLeft: destination
				},
				{
					tween: opts.tween || this._opts.tween
				}
			);
			
			this._index = itemIndex;
			
			
			anim.on('complete', function() {
				that._inMotion = false;
				
				// in case our clones are showing
				jump.call(that);
				swap.call(that);
				
				// force index to be a number from 0 to items.length
				that._index = that._index % (that.items.length  + that._gap.count);
				
				that.fire('afterMove', {currentIndex: that._index});
				
				if (opts.callback) {
					opts.callback();
				}
			});
		}
		
		return this;
	}
	
	/**
		@private
		@function
		@name jump
		@description Quickly move forward or back to a new set of items that look the same as
		the current set of items.
	 */
	function jump() { /*debug*///console.log('jump()');
		if (this._index < 0) {
			this.moveTo(this.items.length + this._gap.count + this._index, {jump: true});
		}
		else if (this._index >= this.items.length) {
			this.moveTo(this._index - (this.items.length + this._gap.count), {jump: true});
		}
	}
	
	/**
		Move real items to stand-in for any clones that are in the spotlight, or
		put the real items back again.
		@name swap
		@private
		@param {boolean} back If a truthy value, will move the real items back.
	 */
	function swap(back) { /*debug*///console.log('swap('+back+')');
		var swapItemIndex;
		
		if (!this._opts.loop) { return; } // no clones, so no swap possible
		
		if (back) {
			this._itemList.place();
		}
		else {
			for (var i = 0, leni = this._spot.capacity - this._gap.count; i < leni; i++) {
				swapItemIndex = (this._index + i);
				if (swapItemIndex >= this.items.length) { // a clone needs to have a real item swapped-in
					this._itemList.swap(swapItemIndex, swapItemIndex % this.items.length);
				}
			}
		}
	}
	
	/**
		@name glow.ui.CarouselPane#moveBy
		@function
		@description Move by a number of items.
		
		@param {number} amount Amount and direction to move.
			Negative numbers will move backwards, positive number will move
			forwards.
			
			This method respects the carousel's limits and its step. If it's
			not possible to move the item so it's the leftmost item of the spotlight, it will
			be placed as close to the left as possible.
		
		@returns this
	*/
	CarouselPaneProto.moveBy = function(amount) { /*debug*///console.log('moveBy('+amount+')');
		this.moveTo(this._index + amount);
		return this;
	}
	
	/**
		@name glow.ui.CarouselPane#next
		@function
		@description Move forward by the step.
		@returns this
	*/
	CarouselPaneProto.next = function() { /*debug*///console.log('next()');
		this.moveTo(this._index + this._step);
		return this;
	}
	
	/**
		@name glow.ui.CarouselPane#prev
		@function
		@description Move backward by the step.
		@returns this
	*/
	CarouselPaneProto.prev = function() { /*debug*///console.log('prev()');
		this.moveTo(this._index - this._step);
		return this;
	}
	
	/**
		@private
		@name canGo
		@description Determine if the CarouselPane can go to the desired index.
		@param {number} itemIndex The desired index.
		@returns {boolean}
	 */
	function canGo(itemIndex) { /*debug*///console.log('canGo('+itemIndex+')');
		if (this._opts.loop) { return true; }
		
		// too far prev
		if (itemIndex < 0) {
			return false;
		}

		// too far next
		if (itemIndex - this._step >= this.items.length - this._spot.capacity ) {
			return false;
		}
		return true;
	}
	
	/**
		@private
		@name getDimensions
		@description Calculate the max height and width of all the items.
		@param {glow.NodeList} items
		@returns {Object} With properties `width` and 'height`.
	 */
	function getDimensions(items) {
		var el,
			maxInnerWidth = 0,
			maxInnerHeight = 0,
			maxWidth = 0,
			maxHeight = 0,
			margin = 0,
			marginRight = 0,
			marginLeft = 0,
			marginTop = 0,
			marginBottom = 0;
			
		items.each(function() {
			el = glow(this);
			maxHeight = Math.max(this.offsetHeight, maxHeight);
			maxWidth = Math.max(this.offsetWidth, maxWidth);
			maxInnerWidth = Math.max(el.width(), maxInnerWidth);
			maxInnerHeight = Math.max(el.height(), maxInnerHeight);
			marginRight = Math.max(autoToValue(el.css('margin-right')), marginRight);
			marginLeft = Math.max(autoToValue(el.css('margin-left')), marginLeft);
			marginTop = Math.max(autoToValue(el.css('margin-top')), marginTop);
			marginBottom = Math.max(autoToValue(el.css('margin-bottom')), marginBottom);
		});
		
		// simulate margin collapsing. see: http://www.howtocreate.co.uk/tutorials/css/margincollapsing
		margin = Math.max(marginLeft, marginRight); // the larger of: the largest left matgin and the largest right margin
		return { width: maxWidth+margin, height: maxHeight+marginTop+marginBottom, innerWidth: maxInnerWidth, innerHeight: maxInnerHeight, marginLeft: marginLeft, marginRight: marginRight, marginTop: marginTop, marginBottom: marginBottom };
	}
	
	function autoToValue(v) {
		if (v === 'auto') return 0;
		else return parseInt(v);
	}
	
	/**
		@private
		@name _getSpot
		@description Calculate the bounds for the spotlighted area, within the viewport.
		@private
	 */
	CarouselPane._getSpot = function(viewportWidth, items, itemDimensions, opts) {/*debug*///console.log('CarouselPane._getSpot()');
		var spot = { capacity: 0, top: 0, left: 0, width: 0, height: 0, offset: { top: 0, right: 0, bottom: 0, left: 0 } },
			opts = opts || {}
		
		if (!itemDimensions) { itemDimensions = getDimensions(items); }
		
		if (opts.axis = 'x') {
			if (items.length === 0) {
				spot.capacity = 0;
			}
			else if (opts.spotlight) {
				if (opts.spotlight > items.length) {
					throw new Error('spotlight cannot be larger than item count.');
				}
				spot.capacity = opts.spotlight;
			}
			else {
				spot.capacity = Math.floor( viewportWidth / itemDimensions.width );
			}

			if (spot.capacity > items.length) {
				spot.capacity = items.length;
			}

			spot.width = spot.capacity * itemDimensions.width + Math.min(itemDimensions.marginLeft, itemDimensions.marginRight);
			spot.height = itemDimensions.height
			
			spot.offset.left = Math.floor( (viewportWidth - spot.width) / 2 );
			spot.offset.right = viewportWidth - (spot.offset.left + spot.width);
		}
		else {
			throw Error('y axis (vertical) not yet implemented');
		}
		
		return spot;
	}
	
	function getPosition(itemIndex) { /*debug*///console.log('getPosition('+itemIndex+')');
		position = { top: 0, left: 0 };
		
		// TODO: memoise?
		var size = this._itemDimensions.width,
			offset = this._spot.offset.left + this._wingSize + this._itemDimensions.marginLeft,
			gap = 0;
			
			if (this._opts.page && itemIndex < 0) {
				gap = -(1 + Math.floor( Math.abs(itemIndex+this._gap.count) / this.items.length)) * this._gap.count * size;
			}
			else if (this._opts.page && itemIndex >= this.items.length) {
				gap = Math.floor(itemIndex / this.items.length) * this._gap.count * size;
			}

			position.left = offset + (itemIndex * size) + gap;
			position.top = this._itemDimensions.marginTop;

			return position;
	}
	
	function layout() {/*debug*///console.log('layout()');
		var clone,
			cloneOffset;
					
		this.content[0].scrollLeft = this._wingSize;

		for (var i = 0, leni = this.items.length; i < leni; i++) {
			// items were already added in ItemList constructor, just add meta now
			this._itemList.addMeta(i, {offset:getPosition.call(this, i).left, isClone:false});

			this.items.item(i).data('itemIndex', +i);
		}
		
		if (this._opts.loop) { // send in the clones
			this.stage.get('.carousel-clone').remove(); // kill any old clones
			
			// how many sets of clones (on each side) are needed to fill the off-spotlight portions of the stage?
			var repsMax =  1 + Math.ceil(this._spot.offset.left / (this._itemDimensions.width*this.items.length + this._gap.size));	

			for (var reps = 1; reps <= repsMax; reps++) {
				i = this.items.length;
				while (i--) {
 					// add clones to prev side
 					clone = this.items.item(i).copy();
 					clone.removeClass('carousel-item').addClass('carousel-clone').css({ 'z-index': 1, margin: 0 });
					
 					cloneOffset = getPosition.call(this, 0 - (reps * this.items.length - i)).left;
 					this._itemList.addItem(0 - (reps * this.items.length - i), clone, {isClone:true, offset:cloneOffset});
 					this.stage[0].appendChild(clone[0]);
					
 					// add clones to next side
 					clone = clone.copy();
 					cloneOffset = getPosition.call(this, reps*this.items.length + i).left;
 					this._itemList.addItem(reps*this.items.length + i + this._gap.count, clone, {isClone:true, offset:cloneOffset});
					this.stage[0].appendChild(clone[0]);
 				}
 			}
		}
		
		this.items.addClass('carousel-item');
		// apply positioning to all items and clones
 		this._itemList.place(this._itemDimensions.marginTop, undefined);
	}
	
	/**
		@name glow.ui.CarouselPane#destroy
		@function
		@description Remove listeners and added HTML Elements from this instance.
			CarouselPane items will not be destroyed.
			
		@returns undefined
	*/
	CarouselPaneProto.destroy = function() {
		this.stage.get('.carousel-clone').remove();
		detachEvents(this);
		this.stage.insertBefore(this.container).children().css('position', '');
		WidgetProto.destroy.call(this);
	};
	
	/**
		@name glow.ui.CarouselPane#event:select
		@event
		@description Fires when a carousel item is selected.
			Items are selected by clicking, or pressing enter when a child is in the spotlight.
		
			Canceling this event prevents the default click/key action.
		
		@param {glow.events.Event} event Event Object
		@param {glow.NodeList} event.item Item selected
		@param {number} event.itemIndex The index of the selected item in {@link glow.ui.CarouselPane#items}.
	*/
	
	/**
		@name glow.ui.CarouselPane#event:move
		@event
		@description Fires when the carousel is about to move.
			Canceling this event prevents the carousel from moving.
			
			This will fire for repeated move actions. Ie, this will fire many times
			after #start is called.
		
		@param {glow.events.Event} e Event Object
		@param {number} e.currentIndex Index of the current leftmost item.
		@param {number} e.moveBy The number of items the Carousel will move by.
			This is undefined for 'sliding' moves where the destination isn't known.
			
			This value can be overwritten, resulting in the carousel moving a different amount.
			The carousel step will still be respected.
			
		@example
			// double the amount a carousel will move by
			myCarouselPane.on('move', function(e) {
				e.moveBy *= 2;
			});
	*/

	/**
		@name glow.ui.CarouselPane#event:afterMove
		@event
		@description Fires when the carousel has finished moving.
			Canceling this event prevents the carousel from moving.
			
			This will not fire for repeated move actions. Ie, after #start is
			called this will not fire until the carousel reached an end point
			or when it comes to rest after #stop is called.
			
		@param {glow.events.Event} e Event Object
		@param {number} e.currentIndex Index of the current leftmost item.
			
		@example
			// double the amount a carousel will move by
			myCarouselPane.on('afterMove', function(e) {
				// show content related to this.spotlightItems()[0]
			});
	*/
	
	// EXPORT
	glow.ui.CarouselPane = CarouselPane;
});
Glow.provide(function(glow) {
	var undefined,
		CarouselProto,
		Widget = glow.ui.Widget,
		WidgetProto = Widget.prototype;
	
	/**
		@name glow.ui.Carousel
		@class
		@extends glow.ui.Widget
		@description Create a pane of elements that scroll from one to another.
			
		@param {glow.NodeList|selector|HTMLElement} itemContainer Container of the carousel items.
			The direct children of this item will be treated as carousel items. They will
			be positioned next to each other horizontally.
			
			Each item takes up the same horizontal space, equal to the width of the largest
			item (including padding and border) + the largest of its horizontal margins (as set in CSS).
			
			The height of the container will be equal to the height of the largest item (including
			padding and border) + the total of its vertical margins.
			
		@param {object} [opts] Options
			@param {number} [opts.duration=0.2] Duration of scrolling animations in seconds.
			
			@param {string|function} [opts.tween='easeBoth'] Tween to use for animations.
				This can be a property name of {@link glow.tweens} or a tweening function.
			
			@param {boolean|number} [opts.page=false] Move a whole page at a time.
				If 'true', the page size will be the spotlight size, but you
				can also set this to be an explicit number of items. Space will
				be added to the end of the carousel so pages stay in sync.
				
				If 'false' or 1, the carousel moves one item at a time.
				
			@param {boolean} [opts.loop=false] Loop the carousel from the last item to the first.
			
			@param {number} [opts.spotlight] The number of items to treat as main spotlighted items.
				A carousel may be wide enough to display 2 whole items, but setting
				this to 1 will result in the spotlight item sitting in the middle, with
				half of the previous item appearing before, and half the next item
				appearing after.
				
				By default, this is the largest number of whole items that can exist in
				the width of the container, allowing room for next & previous buttons.
				Any remaining width will be used to partially show the previous/next item
				beneath the next & previous buttons.
				
		@example
			// This creates a carousel out of HTML like...
			// <ol id="carouselItems">
			//   <li>
			//     <a href="anotherpage.html">
			//       <img width="200" height="200" src="img.jpg" alt="" />
			//	   </a>
			//   </li>
			//   ...more list items like above...
			var myCarousel = new glow.ui.Carousel('#carouselItems', {
				loop: true,
				page: true,
			});
			
		@example
			// Make a carousel of thumbnails, which show the full image when clicked.
			// Carousel items look like this...
			// <li>
			//   <a href="fullimage.jpg">
			//     <img src="thumbnail.jpg" width="100" height="100" alt="whatever" />
			//   </a>
			// </li>
			var fullImage = glow('#fullImage'),
				myCarousel = new glow.ui.Carousel('#carouselItems', {
					spotlight: 6
				}).addPageNav('belowCenter').on('select', function(e) {
					fullImage.prop( 'src', e.item.get('a').prop('href') );
					return false;
				});
	*/
	function Carousel(itemContainer, opts) {
		var spot;
		
		Widget.call(this, 'Carousel', opts);
		
		opts = this._opts;
		
		// convert the options for CarouselPane
		if (opts.page) {
			opts.step = opts.page;
			opts.page = true;
		}
		
		this.itemContainer = itemContainer = glow(itemContainer).item(0);
		
		// see if we're going to get enough room for our prev/next buttons
		spot = glow.ui.CarouselPane._getSpot(
			itemContainer.parent().width(),
			itemContainer.children().css('position', 'absolute'),
			0,
			opts
		);
		
		// enfore our minimum back/fwd button size
		if (spot.offset.left < 50) {
			opts.spotlight = spot.capacity - 1;
		}
		
		this._init();
	};
	glow.util.extend(Carousel, glow.ui.Widget);
	CarouselProto = Carousel.prototype;
	
	/**
		@name glow.ui.Carousel#_pane
		@type glow.ui.CarouselPane
		@description The carousel pane used by this Carousel
	*/
	
	/**
		@name glow.ui.Carousel#_prevBtn
		@type glow.NodeList
		@description Element acting as back button
	*/
	/**
		@name glow.ui.Carousel#_nextBtn
		@type glow.NodeList
		@description Element acting as next button
	*/
	
	/**
		@name glow.ui.Carousel#items
		@type glow.NodeList
		@description Carousel items.
	*/
	
	/**
		@name glow.ui.Carousel#itemContainer
		@type glow.NodeList
		@description Parent element of the carousel items.
	*/
	
	// life cycle methods
	CarouselProto._init = function () {
		WidgetProto._init.call(this);
		this._build();
	};
	
	CarouselProto._build = function () {
		var content,
			itemContainer = this.itemContainer,
			pane,
			items,
			spot;
		
		WidgetProto._build.call( this, itemContainer.wrap('<div></div>').parent() );
		content = this.content;
		
		pane = this._pane = new glow.ui.CarouselPane(itemContainer, this._opts);
		spot = pane._spot
		items = this.items = pane.items;
		this.itemContainer = pane.itemContainer;
		
		pane.moveTo(0, {
			tween: null
		});
		
		// add next & prev buttons, autosizing them
		this._prevBtn = glow('<div class="Carousel-prev"><div class="Carousel-btnIcon"></div></div>').prependTo(content).css({
			width: spot.offset.left,
			height: spot.height
		});
		this._nextBtn = glow('<div class="Carousel-next"><div class="Carousel-btnIcon"></div></div>').prependTo(content).css({
			width: spot.offset.right,
			height: spot.height
		});
		
		updateButtons(this);
		
		this._bind();
	};
	
	/**
		@private
		@function
		@description Update the enabled / disabled state of the buttons.
	*/
	function updateButtons(carousel) {
		// buttons are always active for a looping carousel
		if (carousel._opts.loop) {
			return;
		}
		
		var indexes = carousel.spotlightIndexes(),
			lastIndex = indexes[indexes.length - 1],
			lastItemIndex = carousel.items.length - 1;

		// add or remove the disabled class from the buttons
		carousel._prevBtn[ (indexes[0] === 0) ? 'addClass' : 'removeClass' ]('Carousel-prev-disabled');
		carousel._nextBtn[ (lastIndex === lastItemIndex) ? 'addClass' : 'removeClass' ]('Carousel-next-disabled');
	}
	
	/**
		@private
		@function
		@description Listener for CarouselPane's 'select' event.
			'this' is the Carousel
	*/
	function paneSelect(event) {
		this.fire('select', event);
	}
	
	/**
		@private
		@function
		@description Listener for CarouselPane's 'move' event.
			'this' is the Carousel
	*/
	function paneMove(event) {
		var pane = this._pane;
		
		if ( !this.fire('move', event).defaultPrevented() ) {
			this._updateNav( (pane._index + event.moveBy) % this.items.length / pane._step );
		}
	}
	
	/**
		@private
		@function
		@description Listener for CarouselPane's 'afterMove' event.
			'this' is the Carousel
	*/
	function paneAfterMove(event) {
		if ( !this.fire('afterMove', event).defaultPrevented() ) {
			updateButtons(this);
		}
	}
	
	/**
		@private
		@function
		@description Listener for back button's 'mousedown' event.
			'this' is the Carousel
	*/
	function prevMouseDown(event) {
		if (event.button === 0) {
			this._pane.moveStart(true);
			return false;
		}
	}
	
	/**
		@private
		@function
		@description Listener for fwd button's 'mousedown' event.
			'this' is the Carousel
	*/
	function nextMouseDown(event) {
		if (event.button === 0) {
			this._pane.moveStart();
			return false;
		}
	}
	
	/**
		@private
		@function
		@description Stop the pane moving.
			This is used as a listener for various mouse events on the
			back & forward buttons.
			
			`this` is the Carousel.
	*/
	function paneMoveStop() {
		this._pane.moveStop();
	}
	
	CarouselProto._bind = function () {
		var pane = this._pane,
			carousel = this;
		
		this._tie(pane);
		
		pane.on('select', paneSelect, this)
			.on('afterMove', paneAfterMove, this)
			.on('move', paneMove, this);
		
		this._prevBtn.on('mousedown', prevMouseDown, this)
			.on('mouseup', paneMoveStop, this)
			.on('mouseleave', paneMoveStop, this);
		
		this._nextBtn.on('mousedown', nextMouseDown, this)
			.on('mouseup', paneMoveStop, this)
			.on('mouseleave', paneMoveStop, this);
		
		WidgetProto._bind.call(this);
	};
	
	/**
		@name glow.ui.Carousel#spotlightItems
		@function
		@description Get the currently spotlighted items.
		
		@returns {glow.NodeList}
	*/
	CarouselProto.spotlightItems = function() {
		return this._pane.spotlightItems();
	};
	
	/**
		@name glow.ui.Carousel#spotlightIndexes
		@function
		@description Gets an array of spotlighted indexes.
			These are the indexes of the nodes within {@link glow.ui.Carousel#items}.
		
		@returns {number[]}
	*/
	CarouselProto.spotlightIndexes = function() {
		return this._pane.spotlightIndexes();
	};
	
	/**
		@name glow.ui.Carousel#moveTo
		@function
		@description Move the items so a given index is in the spotlight.
		
		@param {number} itemIndex Item index to move to.
		
		@param {boolean} [animate=true] Transition to the item.
			If false, the carousel will switch to the new index.
		
		@returns this
	*/
	CarouselProto.moveTo = function(itemIndex, animate) {
		this._pane.moveTo(itemIndex, animate);
		return this;
	};
	
	/**
		@private
		@function
		@decription Creates the prev & next functions
		@param {number} direction Either 1 or -1
	*/
	function prevNext(direction) {
		return function() {
			this._pane.moveBy(this._pane._step * direction);
			return this;
		}
	}
	
	/**
		@name glow.ui.Carousel#next
		@function
		@description Move to the next page/item
		
		@returns this
	*/
	CarouselProto.next = prevNext(1);
	
	/**
		@name glow.ui.Carousel#prev
		@function
		@description Move to the previous page/item
		
		@returns this
	*/
	CarouselProto.prev = prevNext(-1);
	
	/**
		@name glow.ui.Carousel#destroy
		@function
		@description Remove listeners and styles from this instance.
			Carousel items will not be destroyed.
			
		@returns undefined
	*/
	CarouselProto.destroy = function() {
		// Move the pane outside our widget
		this._pane.container.insertBefore(this.container);
		WidgetProto.destroy.call(this);
	};
	
	/*
		@name glow.ui.Carousel#updateUi
		@function
		@description Refresh the carousel after moving/adding/removing items.
			You can edit the items within the carousel using NodeLists such
			as {@link glow.ui.Carousel#itemContainer}.
			
		@example
			// Add a new carousel item
			myCarousel.itemContainer.append('<li>New Item</li>');
			// Move the new item into position & update page nav etc...
			myCarousel.updateUi();
			
		@returns this
	*/
	// TODO: populate #items here & check back & fwd button sizes
	
	/**
		@name glow.ui.Carousel#event:select
		@event
		@description Fires when a carousel item is selected.
			Items are selected by clicking, or pressing enter when a child is in the spotlight.
		
			Canceling this event prevents the default click/key action.
		
		@param {glow.events.Event} event Event Object
		@param {glow.NodeList} event.item Item selected
		@param {number} event.itemIndex The index of the selected item in {@link glow.ui.Carousel#items}.
	*/
	
	/**
		@name glow.ui.Carousel#event:move
		@event
		@description Fires when the carousel is about to move.
			Canceling this event prevents the carousel from moving.
			
			This will fire for repeated move actions. Ie, this will fire many times
			while the mouse button is held on one of the arrows.
		
		@param {glow.events.Event} event Event Object
		@param {number} event.moveBy The number of items we're moving by.
			This will be positive for forward movements and negative for backward
			movements.
		
			You can get the current index via `myCarousel.spotlightIndexes()[0]`.
	*/
	
	/**
		@name glow.ui.Carousel#event:afterMove
		@event
		@description Fires when the carousel has finished moving.
			Canceling this event prevents the carousel from moving.
			
			This will not fire for repeated move actions. Ie, after #start is
			called this will not fire until the carousel reached an end point
			or when it comes to rest after #stop is called.
			
		@param {glow.events.Event} event Event Object
			
		@example
			// double the amount a carousel will move by
			myCarousel.on('afterMove', function(e) {
				// show content related to this.spotlightIitems()[0]
			});
	*/
	
	// EXPORT
	glow.ui.Carousel = Carousel;
});
Glow.provide(function(glow) {
	var undefined,
		CarouselProto = glow.ui.Carousel.prototype;

	/**
		@name glow.ui.Carousel#_pageNav
		@type glow.NodeList
		@description Element containing pageNav blocks
	*/
	/**
		@name glow.ui.Carousel#_pageNavOpts
		@type Object
		@description Options for the page nav.
			Same as the opts arg for #addPageNav
	*/
	
	/**
		@name glow.ui.Carousel#addPageNav
		@function
		@description Add page navigation to the carousel.
			The page navigation show the user which position they are viewing
			within the carousel.
		
		@param {Object} [opts] Options object.
		@param {string|selector|HTMLElement} [opts.position='belowLast'] The position of the page navigation.
			This can be a CSS selector pointing to an element, or one of the following
			shortcuts:
		
			<dl>
				<dt>belowLast</dt>
				<dd>Display the nav beneath the last spotlight item</dd>
				<dt>belowNext</dt>
				<dd>Display the nav beneath the next button</dd>
				<dt>belowMiddle</dt>
				<dd>Display the nav beneath the carousel, centred</dd>
				<dt>aboveLast</dt>
				<dd>Display the nav above the last spotlight item</dd>
				<dt>aboveNext</dt>
				<dd>Display the nav above the next button</dd>
				<dt>aboveMiddle</dt>
				<dd>Display the nav above the carousel, centred</dd>
			</dl>
			
		@param {boolean} [opts.useNumbers=false] Display as numbers rather than blocks.
		
		@returns this
		
		@example
			new glow.ui.Carousel('#carouselContainer').addPageNav({
				position: 'belowMiddle',
				useNumbers: true
			});
	*/
	CarouselProto.addPageNav = function(opts) {
		opts = glow.util.apply({
			position: 'belowLast'
		}, opts);
		
		var className = 'Carousel-pageNav';
		
		if (opts.useNumbers) {
			className += 'Numbers';
		}
		
		this._pageNav = glow('<div class="' + className + '"></div>')
			.delegate('click', 'div', pageNavClick, this);
		
		this._pageNavOpts = opts;
		
		initPageNav(this);
		
		return this;
	};
	
	/**
		@private
		@function
		@description Listener for one of the page buttons being clicked.
			'this' is the carousel
	*/
	function pageNavClick(event) {
		var targetPage = ( glow(event.attachedTo).text() - 1 ) * this._pane._step;
		this.moveTo(targetPage);
	}
	
	/**
		@private
		@function
		@description Calculate the number of pages this carousel has
	*/
	function getNumberOfPages(carousel) {
		var pane = carousel._pane,
			itemsLength = carousel.items.length,
			step = pane._step;
		
		if (carousel._opts.loop) {
			r = Math.ceil( itemsLength / step );
		}
		else {
			r = 1 + Math.ceil( (itemsLength - pane._spot.capacity) / step );
		}
		
		// this can be less than one if there's less than 1 page worth or items
		return Math.max(r, 0);
	}
	
	/**
		@private
		@function
		@description Position & populate the page nav.
			Its position may need refreshed after updating the carousel ui.
	*/
	function initPageNav(carousel) {
		var pageNav = carousel._pageNav,
			position = carousel._pageNavOpts.position,
			positionY = position.slice(0,5),
			positionX = position.slice(5),
			pane = carousel._pane,
			numberOfPages = getNumberOfPages(carousel),
			htmlStr = '';
		
		// either append or prepend the page nav, depending on option
		carousel.container[ (positionY === 'below') ? 'append' : 'prepend' ](pageNav);
		
		// position in the center for Middle positions, otherwise right
		pageNav.css('text-align', (positionX == 'Middle') ? 'center' : 'right');

		// move it under the last item for *Last positions
		if (positionX === 'Last') {
			pageNav.css( 'margin-right', carousel._nextBtn.width() + pane._itemDimensions.marginRight )
		}
		
		// build the html string
		do {
			htmlStr = '<div>' + numberOfPages + '</div>' + htmlStr;
		} while (--numberOfPages);
		
		pageNav.html(htmlStr);
		carousel._updateNav( pane._index / pane._step );
	}
	
	/**
		@name glow.ui.Carousel#_updateNav
		@function
		@description Activate a particular item on the pageNav
		
		@param {number} indexToActivate
	*/
	CarouselProto._updateNav = function(indexToActivate) {
		if (this._pageNav) {
			var activeClassName = 'active';
			
			this._pageNav.children()
				.removeClass(activeClassName)
				.item(indexToActivate).addClass(activeClassName);	
		}
	}
});
Glow.complete('ui', '2.0.0b1');
