/*!
	Copyright 2010 British Broadcasting Corporation

	Licensed under the Apache License, Version 2.0 (the "License");
	you may not use this file except in compliance with the License.
	You may obtain a copy of the License at

	   http://www.apache.org/licenses/LICENSE-2.0

	Unless required by applicable law or agreed to in writing, software
	distributed under the License is distributed on an "AS IS" BASIS,
	WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
	See the License for the specific language governing permissions and
	limitations under the License.
*/
/**
	@name glow
	@namespace
	@version @VERSION@
	@description The glow library namespace
		The library can also be used as a function, which is a shortcut to
		{@link glow.NodeList}.
		
	@example
		var links = glow('a');
		// is the same as
		var links = new glow.NodeList('a');
*/
if (!window.Glow) { // loading packages via user SCRIPT tags?
	window.Glow = {
		provide: function(f) {
			f(glow);
		},
		complete: function(n, version) {
			glow.version = version;
		}
	};
	
	window.glow = function(nodeListContents) {
		return new glow.NodeList(nodeListContents);
	};
	glow.UID = 'glow' + Math.floor(Math.random() * (1<<30));
	glow.load = function() {
		throw new Error('Method load() is not available without glow.js');
	}
}


Glow.provide(function(glow) {
	/*!debug*/
	var glowbug = {
	errors: []
	,
	log: function(message, fileName, lineNumber) {
		this._add('Log', message, fileName, lineNumber);
	}
	,
	warn: function(message, fileName, lineNumber) {
		this._add('Warn', message, fileName, lineNumber);
	}
	,
	error: function(message, fileName, lineNumber) {
		this._add('Error', message, fileName, lineNumber);
	}
	,
	_add: function(level, message, fileName, lineNumber) {
		var e = new Error(message, fileName, lineNumber);
		
		e.message = message;
		e.name = 'Glow'+level;
		e.level = level.toLowerCase();
		
		var match = /\[([^\]]+)\]/.exec(message);
		if (match) e.type = match[1]; // may be undefined
		else e.type = 'message';
		
		this.errors.push(e);
		
		this.out(e);
	}
	,
	out: function(e) {
		var message = '['+e.level+'] '+e.message;
		
		if (window.console) {
			if (e.level === 'warn' && window.console.warn) {
				console.warn(message);
			}
			else if (e.level === 'error' && window.console.error) {
				console.error(message);
			}
			else if (window.console.log) {
				console.log(message);
			}
		}
		else if (window.opera && opera.postError) {
			opera.postError(message);
		}
		else { // use our own console
			glowbug.console.log(e.level, message);
		}
	}
};

glowbug.console = {
	messages: [],
	log: function(level, message) {
		if (!this._w) {
			try {
				this._w = window.open('', 'report', 'width=350,height=250,menubar=0,toolbar=0,location=no,status=0,scrollbars=1,resizable=1');
				this._w.document.write(
					'<html><head><title>Console<\/title><style>body{background-color: #ddd;} .message{background-color:#FFF;padding:4px;margin:0px;border-bottom:1px solid #ccc;} .warn {background-color: #E5E6B6;} .error{background-color: #D39C9E;}<\/style><\/head>'
					+ '<body style="font: 11px monaco"><code id="messages"><\/code><\/body><\/html>'
				)
				this._w.document.close();
			}
			catch(ignored) {
				this._w = null;
			}
		}
		
		if (this._w) {
			var p = this._w.document.createElement('P');
			p.className = 'message ' + level;
			p.innerHTML = message;
			this._w.document.getElementById('messages').appendChild(p);
			
			var dh = this._w.document.body.scrollHeight
			var ch = this._w.document.body.clientHeight
			if (dh > ch) { this._w.scrollTo(0, dh-ch); }
		}
	}
}
	if (typeof glowbug != 'undefined') { glow.debug = glowbug; }
	/*gubed!*/
});
Glow.provide(function(glow) {
	/**
		@name glow.env
		@namespace
		@description Information about the browser and characteristics
	*/
	
	// parse the useragent string, setting NaN if match isn't found
	var ua = navigator.userAgent.toLowerCase(),
		nanArray = [0, NaN],
		opera  = (/opera[\s\/]([\w\.]+)/.exec(ua) || nanArray)[1],
		ie     = opera ? NaN : (/msie ([\w\.]+)/.exec(ua) || nanArray)[1],
		gecko  = (/rv:([\w\.]+).*gecko\//.exec(ua) || nanArray)[1],
		webkit = (/applewebkit\/([\w\.]+)/.exec(ua) || nanArray)[1],
		khtml  = (/khtml\/([\w\.]+)/.exec(ua) || nanArray)[1],
		toNumber = parseFloat,
		env = {};
	
	/**
		@name glow.env.gecko
		@type number
		@description Gecko version number to one decimal place (eg 1.9) or NaN on non-gecko browsers.
			The most popular browser using the Gecko engine is Firefox.
		
		@see <a href="http://en.wikipedia.org/wiki/Gecko_(layout_engine)#Usage">Versions of Gecko used by browsers</a>
		
		@example
			if (glow.env.gecko < 1.9) {
				// runs in Firefox 2 and other browsers that use Gecko earlier than 1.9
			}
	*/
	env.gecko = toNumber(gecko);
	
	/**
		@name glow.env.ie
		@type number
		
		@description IE version number to one decimal place (eg 6.0) or NaN on non-IE browsers.
			This number will also be populated for browser based on IE's trident engine
			
		@example
			if (glow.env.ie < 9) {
				// runs in IE pre-9.0
				glow('#content').css('background', 'deadmoomin.png');
			}
	*/
	env.ie = toNumber(ie);
	
	/**
		@name glow.env.opera
		@type number
		
		@description Opera version number to one decimal place (eg 10.0) or NaN on non-Opera browsers.
		
		@example
			if (glow.env.opera < 10) {
				// runs in Opera pre-10.0
			}
	*/
	env.opera = toNumber(opera);
	
	/**
		@name glow.env.webkit
		@type number
		
		@description Webkit version number to one decimal place (eg 531.9) or NaN on non-Webkit browsers.
			Safari and Google Chrome are the most popular browsers using Webkit.
			
		@see <a href="http://en.wikipedia.org/wiki/Safari_version_history#Release_history">Versions of Webkit used by Safari</a>
		@see <a href="http://en.wikipedia.org/wiki/Google_Chrome#Release_history">Versions of Webkit used by Google Chrome</a>
			
		@example
			if (glow.env.webkit < 526) {
				// runs in Safari pre-4.0, and Chrome pre-1.0
			}
	*/
	env.webkit = toNumber(webkit);

	/**
		@name glow.env.khtml
		@type number
		
		@description KHTML version number to one decimal place or NaN on non-KHTML browsers.
			Konqueror is the most popular browsers using KHTML.
	*/
	env.khtml = toNumber(khtml);
	
	/**
		@name glow.env.standardsMode
		@type boolean
		@description True if the browser reports itself to be in 'standards mode'
			Otherwise, the browser is in 'quirks mode'
			
		@see <a href="http://en.wikipedia.org/wiki/Quirks_mode">Quirks Mode vs Standards Mode</a>
	*/
	env.standardsMode = document.compatMode != "BackCompat" && (!env.ie || env.ie >= 6);
	
	/**
		@name glow.env.version
		@type string
		@description Version number of the browser in use as a string.
			This caters for version numbers that aren't 'real' numbers, like "7b" or "1.9.1"
	*/
	env.version = ie || gecko || webkit || opera || khtml || '';
	
	// export
	glow.env = env;
});
// start-source: core/ready.js
/*debug*///log.info('executing core/ready.js');
Glow.provide(
	function(glow) {
		var readyQueue = [],
			domReadyQueue = [],
			blockersActive = 0,
			processingReadyQueue = false;
			
		glow._readyBlockers = {};
		
 		/*debug*///log.info('overwriting Glow ready with glow.ready');	
		glow.ready = function (f) { /*debug*///log.info('glow.ready()');
			if (this.isReady) {
				f();
			}
			else {
				readyQueue.push(f);
			}
			return glow;
		};
		
		glow.onDomReady = function(f) {
			//just run function if already ready
			if (glow.isDomReady) {
				f();
			}
			else {
				domReadyQueue.push(f);
			}
		};
		
		glow._addReadyBlock = function(name) { /*debug*///log.info('_addReadyBlock('+name+')');
			if (typeof glow._readyBlockers[name] === 'undefined') { glow._readyBlockers[name] = 0; }
			glow._readyBlockers[name]++;
			glow.isReady = false;
			blockersActive++; /*debug*///log.info('  &#187; blockersActive '+blockersActive+'.');
			return glow;
		}
			
		glow._removeReadyBlock = function(name) { /*debug*///log.info('_removeReadyBlock('+name+')');
			if (glow._readyBlockers[name]) {
				glow._readyBlockers[name]--;
				blockersActive--;  /*debug*///log.info('  &#187; blockersActive '+blockersActive+'.');
				// if we're out of blockers
				if (!blockersActive) {
					// call our queue
					glow.isReady = true;
					runReadyQueue();
				}
			}
			return glow;
		}
		
		// add blockers for any packages that started loading before core (this package) was built
		if (glow._build) { // only defined when using big Glow
			for (var i = 0, len = glow._build.loading.length; i < len; i++) {
				glow._addReadyBlock('glow_loading_'+glow._build.loading[i]);
			}
			
			for (var j = 0, lenj = glow._build.callbacks.length; j < lenj; j++) {
				if (glow._addReadyBlock) { glow._addReadyBlock('glow_loading_loadedcallback'); }
			}
		}
		
		function runDomReadyQueue() { /*debug*///log.info('runDomReadyQueue()');
			glow.isDomReady = true;
			// run all functions in the array
			for (var i = 0, len = domReadyQueue.length; i < len; i++) {
				domReadyQueue[i]();
			}
		}
	
		function runReadyQueue() { /*debug*///log.info('runReadyQueue()');
			// if we're already processing the queue, just exit, the other instance will take care of it
			if (processingReadyQueue) { return; }
			
			/*debug*///log.info('readyQueue: '+readyQueue.length);
			processingReadyQueue = true;
			while (readyQueue.length) {
				var callback = readyQueue.shift();
				/*debug*///log.info('callback: '+callback);
				callback(glow);
				
				// check if the previous function has created a blocker
				if (blockersActive) { break; }
			}
			processingReadyQueue = false;
		}
		
		
		/**
			@private
			@function
			@name bindReady
			@description Add listener to document to detect when page is ready.
		 */
		var bindReady = function() { // use `var bindReady= function` form instead of `function bindReady()` to prevent FireBug 'cannot access optimized closure' error
			//don't do this stuff if the dom is already ready
			if (glow.isDomReady) { return; }
			glow._addReadyBlock('glow_domReady'); // wait for dom to be ready
			
			function onReady() { /*debug*///log.info('onReady()');
				runReadyQueue();
				glow._removeReadyBlock('glow_domReady');
			}
					
			if (document.readyState == 'complete') { // already here!
				 /*debug*///log.info('already complete');
				onReady();
			}
			else if (glow.env.ie && document.attachEvent) { /*debug*///log.info('bindready() - document.attachEvent');
				// like IE
				
				// not an iframe...
				if (document.documentElement.doScroll && window == top) {
					(function() {  /*debug*///log.info('doScroll');
						try {
							document.documentElement.doScroll('left');
						}
						catch(error) {
							setTimeout(arguments.callee, 0);
							return;
						}
				
						// and execute any waiting functions
						onReady();
					})();
				}
				else {
					// an iframe...
					document.attachEvent(
						'onreadystatechange',
						function() { /*debug*///log.info('onreadystatechange');
							if (document.readyState == 'complete') {
								document.detachEvent('onreadystatechange', arguments.callee);
								onReady();
							}
						}
					);
				}
			}
			else if (document.readyState) { /*debug*///log.info('bindready() - document.readyState');
				// like pre Safari
				(function() { /*debug*///log.info('loaded|complete');
					if ( /loaded|complete/.test(document.readyState) ) {
						onReady();
					}
					else {
						setTimeout(arguments.callee, 0);
					}
				})();
			}
			else if (document.addEventListener) {/*debug*///log.info('bindready() - document.addEventListener');
				// like Mozilla, Opera and recent webkit
				document.addEventListener( 
					'DOMContentLoaded',
					function(){ /*debug*///log.info('glow DOMContentLoaded');
						document.removeEventListener('DOMContentLoaded', arguments.callee, false);
						onReady();
					},
					false
				);
			}
			else {
				throw new Error('Unable to bind glow ready listener to document.');
			}
		};
	
		glow.notSupported = ( // here are the browsers we don't support
			glow.env.ie < 6 ||
			(glow.env.gecko < 1.9 && !/^1\.8\.1/.test(glow.env.version)) ||
			glow.env.opera < 9 ||
			glow.env.webkit < 412
		);
		// deprecated
		glow.isSupported = !glow.notSupported;
		
		// block 'ready' if browser isn't supported
		if (glow.notSupported) {
			glow._addReadyBlock('glow_browserSupport');
		}
		
		bindReady();
	}
);
// end-source: core/ready.js
/**
	@name glow.util
	@namespace
	@description Core JavaScript helpers
*/
Glow.provide(function(glow) {
	var util = {},
		undefined,
		TYPES = {
			UNDEFINED : "undefined",
			OBJECT    : "object",
			NUMBER    : "number",
			BOOLEAN   : "boolean",
			STRING    : "string",
			ARRAY     : "array",
			FUNCTION  : "function",
			NULL      : "null"
		},
		/*
		PrivateProperty: TEXT
			hash of strings used in encoding/decoding
		*/
		TEXT = {
			AT    : "@",
			EQ    : "=",
			DOT   : ".",
			EMPTY : "",
			AND   : "&",
			OPEN  : "(",
			CLOSE : ")"
		},
		/*
		PrivateProperty: JSON
			nested hash of strings and regular expressions used in encoding/decoding Json
		*/
		JSON = {
			HASH : {
				START     : "{",
				END       : "}",
				SHOW_KEYS : true
			},

			ARRAY : {
				START     : "[",
				END       : "]",
				SHOW_KEYS : false
			},

			DATA_SEPARATOR   : ",",
			KEY_SEPARATOR    : ":",
			KEY_DELIMITER    : "\"",
			STRING_DELIMITER : "\"",

			SAFE_PT1 : /^[\],:{}\s]*$/,
			SAFE_PT2 : /\\./g,
			SAFE_PT3 : /\"[^\"\\\n\r]*\"|true|false|null|-?\d+(?:\.\d*)?(:?[eE][+\-]?\d+)?/g,
			SAFE_PT4 : /(?:^|:|,)(?:\s*\[)+/g
		};
	/**
		@private
		@name glow.util-_getType
		@param {Object} object The object to be tested.
		@returns {string} The data type of the object.
	*/
	function _getType(object) {
		var typeOfObject = typeof object,
			constructorStr,
			type;

		if (object === null) { return 'null'; } // warn: won't work across frames?
		else if (isFunction(object)) { return 'Function'; }
		else if (isArray(object)) { return 'Array'; }
		else if (typeOfObject === 'object') {
			
			constructorStr = object.constructor.toString();

			if ( /^function (\S+?)\(/.test(constructorStr) ) {
				type = RegExp.$1;
				if (type === 'Object') { return 'object'; }
				else { return type; }
			}
		}

		return typeOfObject;
	}

	function isArray(o) {
		return {}.toString.call(o) === '[object Array]';
	}
	
	function isFunction(o) {
		return {}.toString.call(o) === '[object Function]';
	}

	/**
		@name glow.util.getType
		@function
		@description Get the native type or constructor name of an object.
			This allows you to safely get the type of an object, even
			if it came from another frame.
			
		@param {Object} object Object to get the type of.
			
		@example
			glow.util.getType( null ); // 'null'
			glow.util.getType( undefined ); // 'undefined'
			glow.util.getType('Hello'); // 'string'
			glow.util.getType( {} ); // 'Object'
			glow.util.getType(12); // 'number'
			glow.util.getType( [] ); // 'Array'
			glow.util.getType( function(){} ); // 'Function'
			glow.util.getType( glow('#whatever') ); // 'NodeList'
			
		@example
			var MyConstructor = function() {},
				obj = new MyConstructor;
			
			glow.util.getType(obj); // ''
			// The above returns an empty string as the constructor
			// is an anonymous function and therefore has no name
	*/
	util.getType = _getType;
	
	/**
		@name glow.util.apply
		@function
		@description Copies properties from one object to another
			All properties from 'source' will be copied onto
			'destination', potentially overwriting existing properties
			on 'destination'.
			
			Properties from 'source's prototype chain will not be copied.
		
		@param {Object} [destination] Destination object.
			If this object is undefined or falsey, a new object will be created.
		
		@param {Object} [source] Properties of this object will be copied onto the destination
			If this object is undefined or falsey, a new object will be created.
		
		@returns {Object} The destination object.
		
		@example
			var obj = glow.util.apply({foo: "hello", bar: "world"}, {bar: "everyone"});
			//results in {foo: "hello", bar: "everyone"}
	*/
	util.apply = function(destination, source) {
		destination = destination || {};
		source = source || {};
		
		/*!debug*/
			if (typeof destination != 'object') {
				glow.debug.warn('[wrong type] glow.util.apply expects argument "destination" to be of type object, not ' + typeof destination + '.');
			}
			if (typeof source != 'object') {
				glow.debug.warn('[wrong type] glow.util.apply expects argument "source" to be of type object, not ' + typeof source + '.');
			}
		/*gubed!*/
		for (var i in source) {
			if ( source.hasOwnProperty(i) ) {
				destination[i] = source[i];
			}
		}
		return destination;
	};
	
	/**
		@name glow.util.extend
		@function
		@description Copies the prototype of one object to another
			The 'subclass' can also access the 'base class' via subclass.base

		@param {Function} sub Class which inherits properties.
		@param {Function} base Class to inherit from.
		@param {Object} additionalProperties An object of properties and methods to add to the subclass.

		@example
			function MyClass(arg) {
				this.prop = arg;
			}
			MyClass.prototype.showProp = function() {
				alert(this.prop);
			};
			function MyOtherClass(arg) {
				//call the base class's constructor
				MyOtherClass.base.apply(this, arguments);
			}
			glow.util.extend(MyOtherClass, MyClass, {
				setProp: function(newProp) {
					this.prop = newProp;
				}
			});

			var test = new MyOtherClass("hello");
			test.showProp(); // alerts "hello"
			test.setProp("world");
			test.showProp(); // alerts "world"
	*/
	util.extend = function(sub, base, additionalProperties) {
		/*!debug*/
			if (arguments.length < 2) {
				glow.debug.warn('[wrong count] glow.util.extend expects at least 2 arguments, not '+arguments.length+'.');
			}
			if (typeof sub != 'function') {
				glow.debug.error('[wrong type] glow.util.extend expects argument "sub" to be of type function, not ' + typeof sub + '.');
			}
			if (typeof base != 'function') {
				glow.debug.error('[wrong type] glow.util.extend expects argument "base" to be of type function, not ' + typeof base + '.');
			}
		/*gubed!*/
		var f = function () {}, p;
		f.prototype = base.prototype;
		p = new f();
		sub.prototype = p;
		p.constructor = sub;
		sub.base = base;
		if (additionalProperties) {
			util.apply(sub.prototype, additionalProperties);
		}
	};
	
	/**
		@name glow.util.escapeRegex
		@function
		@description Escape special regex chars from a string

		@param {string} str String to escape
		
		@returns {string} Escaped string
		
		@example
			var str = glow.util.escapeRegex('[Hello. Is this escaped?]');
			// Outputs:
			// \[Hello\. Is this escaped\?\]
	*/
	util.escapeRegex = function(str) {
		/*!debug*/
			if (arguments.length !== 1) {
				glow.debug.warn('[wrong count] glow.util.escapeRegex expects 1 argument, not '+arguments.length+'.');
			}
		/*gubed!*/
		return String(str).replace(/[.*+?^${}()|[\]\/\\]/g, '\\$&');
	};
	
	/**
		@name glow.util.encodeUrl
		@function
		@description Encodes an object for use as a query string.
		
			Returns a string representing the object suitable for use 
			as a query string, with all values suitably escaped.
			It does not include the initial question mark. Where the 
			input field was an array, the key is repeated in the output.
		
		@param {Object} object The object to be encoded.
		
			This must be a hash whose values can only be primitives or 
			arrays of primitives.
		
		@returns {String}
		
		@example
			var getRef = glow.util.encodeUrl({foo: "Foo", bar: ["Bar 1", "Bar2"]});
			// will return "foo=Foo&bar=Bar%201&bar=Bar2"
	*/
	util.encodeUrl = function (object) {
		var type = _getType(object),
			paramsList = [],
			listLength = 0;
		
		/*!debug*/
			if (typeof object !== 'object') {
				throw new Error('glow.util.encodeUrl: cannot encode item');
			}
		/*gubed!*/
		
		for (var key in object) {
			type = _getType( object[key] );

			/*!debug*/
				if (type !== 'Array' || type !== 'string') {
					glow.debug.warn('[wrong type] glow.util.encodeUrl expected Array or String value for "' + key + '", not ' + type + '.');
				}
			/*gubed!*/
			if (type === 'Array') {
				for(var i = 0, l = object[key].length; i < l; i++) {
					/*!debug*/
						if (_getType(object[key])[i] !== 'string') {
							glow.debug.warn('[wrong type] glow.util.encodeUrl expected string value for "' + key + '" value at index ' + i + ', not ' + _getType(object[key])[i] + '.');
						}
					/*gubed!*/
					paramsList[listLength++] = key + '=' + encodeURIComponent(object[key][i]);
				}
			}
			else { // assume string
				paramsList[listLength++] = key + '=' + encodeURIComponent(object[key]);
			}
		}

		return paramsList.join('&');
	};
	
	/**
		@name glow.util.decodeUrl
		@function
		@description Decodes a query string into an object.
		
			Returns an object representing the data given by the query 
			string, with all values suitably unescaped. All keys in the 
			query string are keys of the object. Repeated keys result 
			in an array.
		
		@param {String} string The query string to be decoded.
		
			It should not include the initial question mark.
		
		@returns {Object}
		
		@example
			var getRef = glow.util.decodeUrl("foo=Foo&bar=Bar%201&bar=Bar2");
			// will return the object {foo: "Foo", bar: ["Bar 1", "Bar2"]}
	*/
	util.decodeUrl = function(text) {
		/*!debug*/
			if (arguments.length !== 1) {
				glow.debug.warn('[wrong count] glow.util.decodeUrl expects 1 argument, not '+arguments.length+'.');
			}
			if (typeof text !== 'string') {
				glow.debug.warn('[wrong type] glow.util.decodeUrl expects argument "text" to be of type string, not ' + typeof text + '.');
			}
		/*gubed!*/
		
		var result = {},
			keyValues = text.split(/[&;]/),
			thisPair,
			key,
			value;

		for(var i = 0, leni = keyValues.length; i < leni; i++) {
			thisPair = keyValues[i].split('=');
			
			if (thisPair.length < 2) {
				key = keyValues[i];
				value = '';
			}
			else {
				key   = '' + decodeURIComponent(thisPair[0]);
				value = '' + decodeURIComponent(thisPair[1]);
			}
			
			// will be either: undefined, string or [object Array]
			switch (typeof result[key]) {
				case 'string':
					result[key] = [result[key], value];
					break;
				case 'undefined':
					result[key] = value;
					break;
				default:
					result[key].push(value);
			}
		}

		return result;
	};
	
	/**
			@name glow.util.encodeJson
			@function
			@description Encodes an object into a string JSON representation.
			
				Returns a string representing the object as JSON.
			
			@param {Object} object The object to be encoded.
			 
				This can be arbitrarily nested, but must not contain 
				functions or cyclical structures.
			
			@returns {Object}
			
			@example
				var myObj = {foo: "Foo", bar: ["Bar 1", "Bar2"]};
				var getRef = glow.util.encodeJson(myObj);
				// will return '{"foo": "Foo", "bar": ["Bar 1", "Bar2"]}'
			*/
	util.encodeJson = function(object, options){
		function _encode(object, options)
				{
					if(_getType(object) == TYPES.ARRAY) {
						var type = JSON.ARRAY;
					} else {
						var type = JSON.HASH;
					}

					var serial = [type.START];
					var len = 1;
					var dataType;
					var notFirst = false;

					for(var key in object) {
						dataType = _getType(object[key]);

						if(dataType != TYPES.UNDEFINED) { /* ignore undefined data */
							if(notFirst) {
								serial[len++] = JSON.DATA_SEPARATOR;
							}
							notFirst = true;

							if(type.SHOW_KEYS) {
								serial[len++] = JSON.KEY_DELIMITER;
								serial[len++] = key;
								serial[len++] = JSON.KEY_DELIMITER;
								serial[len++] = JSON.KEY_SEPARATOR;
							}

							switch(dataType) {
								case TYPES.FUNCTION:
									throw new Error("glow.data.encodeJson: cannot encode item");
									break;
								case TYPES.STRING:
								default:
									serial[len++] = JSON.STRING_DELIMITER;
									serial[len++] = glow.lang.replace(object[key], SLASHES.TEST, _replaceSlashes);
									serial[len++] = JSON.STRING_DELIMITER;
									break;
								case TYPES.NUMBER:
								case TYPES.BOOLEAN:
									serial[len++] = object[key];
									break;
								case TYPES.OBJECT:
								case TYPES.ARRAY:
									serial[len++] = _encode(object[key], options);
									break;
								case TYPES.NULL:
									serial[len++] = TYPES.NULL;
									break;
							}
						}
					}
					serial[len++] = type.END;

					return serial.join(TEXT.EMPTY);
				}

				options = options || {};
				var type = _getType(object);

				if((type == TYPES.OBJECT) || (type == TYPES.ARRAY)) {
					return _encode(object, options);
				} else {
					throw new Error("glow.data.encodeJson: cannot encode item");
				}
		
	};
	/**
			@name glow.util.decodeJson
			@function
			@description Decodes a string JSON representation into an object.
				
				Returns a JavaScript object that mirrors the data given.
			
			@param {String} string The string to be decoded.
				Must be valid JSON. 
			
			@param {Object} opts
			
					Zero or more of the following as properties of an object:
					@param {Boolean} [opts.safeMode=false] Whether the string should only be decoded if it is  deemed "safe". 
					The json.org regular expression checks are used. 
			
			@returns {Object}
			
			@example
				var getRef = glow.util.decodeJson('{foo: "Foo", bar: ["Bar 1", "Bar2"]}');
				// will return {foo: "Foo", bar: ["Bar 1", "Bar2"]}
			
				var getRef = glow.util.decodeJson('foobar', {safeMode: true});
				// will throw an error
			*/
	util.decodeJson = function(text, options){
		if(_getType(text) != TYPES.STRING) {
					throw new Error("glow.data.decodeJson: cannot decode item");
				}

				options = options || {};
				options.safeMode = options.safeMode || false;

				var canEval = true;

				if(options.safeMode) {
					canEval = (JSON.SAFE_PT1.test(text.replace(JSON.SAFE_PT2, TEXT.AT).replace(JSON.SAFE_PT3, JSON.ARRAY.END).replace(JSON.SAFE_PT4, TEXT.EMPTY)));
				}

				if(canEval) {
					try {
						return eval(TEXT.OPEN + text + TEXT.CLOSE);
					}
					catch(e) {/* continue to error */}
				}

				throw new Error("glow.data.decodeJson: cannot decode item");
	};
	/**
		@name glow.util.trim
		@function
		@description Removes leading and trailing whitespace from a string
	
		@param {string} str String to trim
	
		@returns {String}
	
			String without leading and trailing whitespace
	
		@example
			glow.util.trim("  Hello World  "); // "Hello World"
	*/
	util.trim = function(str) {
		//this optimisation from http://blog.stevenlevithan.com/archives/faster-trim-javascript
		return str.trim ? str.trim() : str.replace(/^\s*((?:[\S\s]*\S)?)\s*$/, '$1');
	};
	
	/**
		@name glow.util.interpolate
		@function
		@description Replaces placeholders in a string with data from an object
		
		@param {String} template The string containing {placeholders}
		@param {Object} data Object containing the data to be merged in to the template
			<p>The object can contain nested data objects and arrays, with nested object properties and array elements are accessed using dot notation. eg foo.bar or foo.0.</p>
			<p>The data labels in the object cannot contain characters used in the template delimiters, so if the data must be allowed to contain the default { and } delimiters, the delimters must be changed using the option below.</p>
		@param {Object} opts Options object
			@param {String} [opts.delimiter="{}"] Alternative label delimiter(s) for the template
				The first character supplied will be the opening delimiter, and the second the closing. If only one character is supplied, it will be used for both ends.
			@param {Boolean} [opts.escapeHtml=false] Escape any special html characters found in the data object
				Use this to safely inject data from the user into an HTML template. The glow.dom module
				must be present for this feature to work (an error will be thrown otherwise).
		
		@returns {String}
		
		@example
			var data = {
				name: "Domino",
				colours: ["black", "white"],
				family: {
					mum: "Spot",
					dad: "Patch",
					siblings: []
				}
			};
			var template = "My cat's name is {name}. His colours are {colours.0} & {colours.1}. His mum is {family.mum}, his dad is {family.dad} and he has {family.siblings.length} brothers or sisters.";
			var result = glow.util.interpolate(template, data);
			// result == "My cat's name is Domino. His colours are black & white. His mum is Spot, his dad is Patch and he has 0 brothers or sisters."
		
		@example
			var data = {
				name: 'Haxors!!1 <script src="hackhackhack.js"></script>'
			}
			var template = '<p>Hello, my name is {name}</p>';
			var result = glow.util.interpolate(template, data, {
				escapeHtml: true
			});
			// result == '<p>Hello, my name is Haxors!!1 &lt;script src="hackhackhack.js"&gt;&lt;/script&gt;</p>'
	*/
	util.interpolate = function(template, data, opts) {
		var placeHolderRx,
			leftDelimiter,
			rightDelimiter,
			// div used for html escaping
			div;
	
		opts = opts || {};
		
		// make sure the dom module is around
		if (opts.escapeHtml) {
			div = glow('<div></div>');
		}
	
		if (opts.delimiter == undefined) {
			placeHolderRx = /\{[^{}]+\}/g;
		}
		else {
			leftDelimiter = opts.delimiter.substr(0, 1).replace(regexEscape, "\\$1");
			rightDelimiter = opts.delimiter.substr(1, 1).replace(regexEscape, "\\$1") || leftDelimiter;
			placeHolderRx = new RegExp(leftDelimiter + "[^" + leftDelimiter + rightDelimiter + "]+" + rightDelimiter, "g");
		}
	
		return template.replace(placeHolderRx, function (placeholder) {
			var key = placeholder.slice(1, -1),
				keyParts = key.split("."),
				val,
				i = 0,
				len = keyParts.length;
			
			if (key in data) {
				// need to be backwards compatible with "flattened" data.
				val = data[key]; 
			}
			else {
				// look up the chain
				val = data;
				for (; i < len; i++) {
					if (keyParts[i] in val) {
						val = val[ keyParts[i] ];
					}
					else {
						return placeholder;
					}
				}
			}
			
			if (opts.escapeHtml) {
				val = div.text(val).html();
			}
			return val;
		});
	};
	
	/**
		@example
			glow.util.cookie(key); // get value for key
			glow.util.cookie({key: val, key2: val2}, opts); // set all keys, vals
			glow.util.cookie(key, val, opts); // set key, val
			glow.util.cookie(); // get all keys, vals
			
			// use value of undefined
	*/
	util.cookie = function(key, value, opts) {
		/*!debug*/
			if (arguments.length > 3) {
				glow.debug.warn('[wrong count] glow.util.cookie expects 3 or less arguments, not '+arguments.length+'.');
			}
			
			if (arguments.length === 1 && _getType(key) !== 'string' && _getType(key) !== 'object') {
				glow.debug.warn('[wrong type] glow.util.cookie expects argument "key" to be of type string or object, not ' + _getType(key) + '.');
			}
			
			if (
				arguments.length === 2
				&&
				(
					! (_getType(key) === 'string' && _getType(value) === 'string')
					||
					! (_getType(key) === 'object' && _getType(value) === 'object')
				)
			) {
				glow.debug.warn('[wrong type] glow.util.cookie expects arguments to be (key, val) or (keyVals, opts).');
			}
			
			if (arguments.length === 3 && _getType(key) !== 'string' && _getType(value) !== 'string' && _getType(opts) !== 'object') {
				glow.debug.warn('[wrong type] glow.util.cookie expects argument "key" and "value" to be strings and "options" to be an object.');
			}
			
			if (opts && opts.debug && (typeof opts.expires !== 'number' || !opts.expires.toUTCString)) {
				glow.debug.warn('[wrong type] glow.util.cookie expects opts.expires to be a number or a Date.');
			}
		/*gubed!*/
		
		
		var date = '',
			expires = '',
			path = '',
			domain = '',
			secure = '',
			keyValues,
			thisPair,
			key,
			val,
			cookieValues;
		
		if (opts) {
			if (opts.expires) {
				if (typeof opts.expires === 'number') {
					date = new Date();
					date.setTime(date.getTime() + (opts.expires * 24 * 60 * 60 * 1000)); // opts.expires days
				}
				else { // is already a Date
					date = opts.expires;
				}
				expires = '; expires=' + date.toUTCString();
			}
		   
			path = opts.path ? '; path=' + (opts.path) : '';
			domain = opts.domain ? '; domain=' + (opts.domain) : '';
			secure = opts.secure ? '; secure' : '';
		}
		else {
			opts = {};
		}
		
		if (typeof key === 'string' && typeof value === 'string') { // a single setter
			document.cookie = key + '=' + encodeURIComponent(value) + expires + path + domain + secure;
		}
		else if (typeof key === 'object') { // an all setter
			for (var p in key) {
				document.cookie = p + '=' + encodeURIComponent(key[p]) + expires + path + domain + secure;
			}
		}
		else { // a getter
			cookieValues = {};
			if (document.cookie && document.cookie != '') {
				keyValues = document.cookie.split(/; ?/);
				for (var i = 0, leni = keyValues.length; i < leni; i++) {
					thisPair = keyValues[i].split('=');
					
					cookieValues[thisPair[0]] = decodeURIComponent(thisPair[1]);
				}
			}
			
			if (typeof key === 'string') { // a single getter
				return cookieValues[key];
			}
			else if (typeof key === 'undefined') { // an all getter
				return cookieValues;
			}
		}
	};
	
	util.removeCookie = function(key) {
		util.cookie(key, '', {expires: -1});
	};
	
	// export
	glow.util = util;
});
Glow.provide(function(glow) {
	/**
	@name glow.events
	@namespace
	@description Handling custom events
	*/
	var events = {};
		
	/* storage variables */
	
	var eventListeners = {}, // eventName: [ [callback, thisVal], ... ] 
		eventId = 1,
		objIdCounter = 1, 
		eventKey = '__eventId' + glow.UID; 

	
	/**
	@name glow.events.addListeners
	@function
	@param {Object[]} attachTo Array of objects to add listeners to.
	@param {string} name Name of the event to listen for.
		Event names are case sensitive.
	@param {function} callback Function to call when the event is fired.
		The callback will be passed a single event object. The type of this
		object depends on the event (see documentation for the event
		you're listening to).
	@param {Object} [thisVal] Value of 'this' within the callback.
		By default, this is the object being listened to.
	@see glow.events.Target#fire
	@description Convenience method to add listeners to many objects at once.
		If you want to add a listener to a single object, use its
		'on' method.
	*/
	events.addListeners = function (attachTo, name, callback, thisVal) {
		var listenerIds = [],
			objIdent,
			listener,
			eventsOnObject,
			currentListeners;
	
		//attach the event for each element, return an array of listener ids
		var i = attachTo.length;
		while (i--) {
			objIdent = attachTo[i][eventKey];
			if (!objIdent){
				objIdent = attachTo[i][eventKey] = objIdCounter++;
			}
					
			listener = [ callback, thisVal ];
			eventsOnObject = eventListeners[objIdent];
			if(!eventsOnObject){
				eventsOnObject = eventListeners[objIdent] = {};
			}
					
			currentListeners = eventsOnObject[name];
			if(!currentListeners){
				eventsOnObject[name] = [listener];
			}
			else{
				currentListeners[currentListeners.length] = listener;
			}							
		}
		return events;
	};
	
	events._getPrivateEventKey = function(node) {
		if (!node[eventKey]) {
			node[eventKey] = objIdCounter++;
		}
		
		return node[eventKey];
	}
	
	/**
	@name glow.events.fire
	@function
	@param {Object[]} items      Array of objects to add listeners to
	@param {string}   eventName  Name of the event to fire
	@param {glow.events.Event|Object} [event] Event object to pass into listeners.
       You can provide a simple object of key-value pairs which will
       be added as properties on the glow.events.Event instance.
		
	@description Convenience method to fire events on multiple items at once.
		If you want to fire events on a single object, use its
		'fire' method.
	*/
		
	events.fire = function (items, eventName, event) {
		if (! event) {
			event = new events.Event();
		}
		else if ( event.constructor === Object ) {
			event = new events.Event( event )
		}
		
		// for loop, because order matters!
		for(var i = 0, len = items.length; i < len; i++) { 
			callListeners(items[i], eventName, event);
		}
			
		return event;
	};

	
	/**
	 @name glow.events-callListeners
	 @private
	*/
	function callListeners(item, eventName, event, thisVal) {
		var objIdent = item[eventKey],
			listenersForEvent,
			returnedVal;			
		
		// set the attachedTo value for this event
		event.attachedTo = event.attachedTo || item;
		
		if (!objIdent || !eventListeners[objIdent]) {
			return event;
		}
		
		listenersForEvent = eventListeners[objIdent][eventName];
			
		if (!listenersForEvent) {
			return event;
		}
		// Slice to make sure we get a unique copy.
		listenersForEvent = listenersForEvent.slice(0);
		for (var i = 0, len = listenersForEvent.length; i < len; i++){
			returnedVal = listenersForEvent[i][0].call((listenersForEvent[i][1] || thisVal || item), event);
			if (returnedVal === false){
				event.preventDefault();
			}
		}
			
		return event;
	}
	events._callListeners = callListeners;
		
		
	/**
	@name glow.events.removeAllListeners
	@function
	@param {Object[]} items Items to remove events from		    
	@description Removes all listeners attached to a given object.
		This removes not only listeners you added, but listeners others
		added too. For this reason it should only be used as part of a cleanup
		operation on objects that are about to be destroyed.
	*/
	
	events.removeAllListeners = function (items) {
		var objIdent,
		i = items.length;		
		
		while(i--){
			
			objIdent = items[i][eventKey];
			
			if (!objIdent) {
				return false;
			}
			else {
				delete eventListeners[objIdent];
			}
		}

		return true;
	};


	/**
	@name glow.events.removeListeners
	@function
	@param {Object[]} items Items to remove events from.
	@param {string} eventName Name of the event to remove.
	@param {function} callback A reference to the original callback used when the listener was added.
	@decription Removes listeners for an event.
	*/
	events.removeListeners = function (item, eventName, callback) { /* TODO: items! */
		var objIdent,
			listenersForEvent,
			i = item.length;
		
	
		while(i--){
			
			objIdent = item[i][eventKey];
				
			if(!objIdent || !eventListeners[objIdent]){
				return events;
			}
			
		
			listenersForEvent = eventListeners[objIdent][eventName];
			if(!listenersForEvent){
				return events;
			}
			
			// for loop, because order matters
			for(var j = 0, lenj = listenersForEvent.length; j < lenj; j++){						
				if (listenersForEvent[j][0] === callback){
					listenersForEvent.splice(j, 1);
					break;
				}
		
			}
		}
		
		return events;			
	};
	
	/**
		Copies the events from one NodeList to another
		@private
		@name glow.events._copyEvents
		@see glow.NodeList#clone
		@function
	*/
	events._copyDomEvents = function(from, to){
		var objIdent,
			i = from.length,
			j, jLen,
			listeners,
			listenersForEvent,
			eventName,
			toItem;
		
		// loop over elements
		while(i--){
			objIdent = from[i][eventKey];
			listeners = eventListeners[objIdent];
			
			if (objIdent){
				toItem = to.slice(i, i+1);
				
				// loop over event names (of listeners attached)
				for ( eventName in listeners ) {
					listenersForEvent = listeners[eventName];
					
					// loop over individual listeners and add them to the 'to' item
					// loop forward to preserve event order
					for (j = 0, jLen = listenersForEvent.length; j < jLen; j++) {
						// add listener
						toItem.on( eventName, listenersForEvent[j][0], listenersForEvent[j][1] );
					}
				}
			}
		}
	}
	/**
	@name glow.events._getListeners
	@private
	@function
	@param {Object} item Item to find events for
	@decription Returns a list of listeners attached for the given item.
	
	*/	
	events._getListeners = function(item){
		var objIdent = item[eventKey];
			
		if (!objIdent) {
			return {};
		}
		else {
			// todo: need to return listeners in a sensible format
			return eventListeners[objIdent];
		}
			
	};
	
	///**
	//@name glow.events.hasListener
	//@function
	//@param {Object[]} item  Item to find events for
	//@param {String}   eventName  Name of the event to match
	//@decription Returns true if an event is found for the item supplied
	//
	//*/
	//
	//glow.events.hasListener = function (item, eventName) {
	//	var objIdent,
	//		listenersForEvent;
	//		
	//	for (var i = 0, len = item.length; i < len; i++) {	
	//		objIdent = item[i][eventKey];
	//			
	//		if (!objIdent || !eventListeners[objIdent]) {
	//			return false;
	//		}
	//				
	//		listenersForEvent = eventListeners[objIdent][eventName];
	//		if (!listenersForEvent) {
	//			return false;
	//		}
	//		else {
	//			return true;							
	//		}					
	//	}
	//	
	//	return false;			
	//};
	
	/**
	@name glow.events.Target
	@class
	@description An object that can have event listeners and fire events.
		Extend this class to make your own objects have 'on' and 'fire'
		methods.
		
	@example
		// Ball is our constructor
		function Ball() {
			// ...
		}
		       
		// make Ball inherit from Target
		glow.util.extend(Ball, glow.events.Target, {
			// additional methods for Ball here, eg:
			bowl: function() {
				// ...
			}
		});
		       
		// now instances of Ball can receive event listeners
		var myBall = new Ball();
		myBall.on('bounce', function() {
			alert('BOING!');
		});
		       
		// and events can be fired from Ball instances
		myBall.fire('bounce');
	*/
	
	events.Target = function () {
			
	};
	var targetProto = events.Target.prototype;
		
	/**
	@name glow.events.Target.extend
	@function
	@param {Object} obj Object to add Target instance methods to.
		
	@description Convenience method to add Target instance methods onto an object.
		If you want to add Target methods to a class, extend glow.events.Target instead.
		       
	@example
		var myApplication = {};
		       
		glow.events.Target.extend(myApplication);
		       
		// now myApplication can fire events...
		myApplication.fire('load');
		       
		// and other objects can listen for those events
		myApplication.on('load', function(e) {
			alert('App loaded');
		});
	*/
	
	events.Target.extend = function (obj) {
		glow.util.apply( obj, glow.events.Target.prototype );
	};
		
	/**
	@name glow.events.Target#on
	@function
	@param {string} eventName Name of the event to listen for.
	@param {function} callback Function to call when the event fires.
		The callback is passed a single event object. The type of this
		object depends on the event (see documentation for the event
		you're listening to).
	@param {Object} [thisVal] Value of 'this' within the callback.
		By default, this is the object being listened to.
		
	@description Listen for an event
		
	@returns this
		
	@example
		myObj.on('show', function() {
		    // do stuff
		});
	*/
	
	targetProto.on = function(eventName, callback, thisVal) {
		glow.events.addListeners([this], eventName, callback, thisVal);
		return this;
	}
		
	/**
	@name glow.events.Target#detach
	@function
	@param {string} eventName Name of the event to remove.
	@param {function} callback Callback to detach.
	@param {Object} [thisVal] Value of 'this' within the callback.
		By default, this is the object being listened to.
	@description Remove an event listener.
		
	@returns this Target object
		
	@example
		function showListener() {
		    // ...
		}
		       
		// add listener
		myObj.on('show', showListener);
		       
		// remove listener
		myObj.detach('show', showListener);
		       
	@example
		// note the following WILL NOT WORK
		       
		// add listener
		myObj.on('show', function() {
		    alert('hi');
		});
		       
		// remove listener
		myObj.detach('show', function() {
			alert('hi');
		});
		       
		// this is because both callbacks are different function instances
	
	*/
		
	targetProto.detach = function(eventName, callback) {
		glow.events.removeListeners(this, eventName, callback);
		return this;
	}
		
	/**
	@name glow.events.Target#fire
	@function
	@param {string} eventName Name of the event to fire.
	@param {glow.events.Event|Object} [event] Event object to pass into listeners.
		    You can provide a simple object of key-value pairs which will
		    be added as properties of a glow.events.Event instance.
		
	@description Fire an event.
		
	@returns glow.events.Event
		
	@example
		myObj.fire('show');
		       
	@example
		// adding properties to the event object
		myBall.fire('bounce', {
		    velocity: 30
		});
	       
	@example
		// BallBounceEvent extends glow.events.Event but has extra methods
		myBall.fire( 'bounce', new BallBounceEvent(myBall) );
	*/
	
	targetProto.fire = function(eventName, event) {
		if (! event) {
			event = new events.Event();
		}
		else if ( event.constructor === Object ) {
			event = new events.Event( event )
		}
		
		return callListeners(this, eventName, event);
	}
		
	/**
	@name glow.events.Event
	@class
	@param {Object} [properties] Properties to add to the Event instance.
		Each key-value pair in the object will be added to the Event as
		properties.
	       
	@description Describes an event that occurred.
		You don't need to create instances of this class if you're simply
		listening to events. One will be provided as the first argument
		in your callback.
	       
	@example
		// creating a simple event object
		var event = new glow.events.Event({
			velocity: 50,
			direction: 180
		});
		       
		// 'velocity' and 'direction' are simple made-up properties
		// you may want to add to your event object
		       
	@example
		// inheriting from glow.events.Event to make a more
		// specialised event object
		       
		function RocketEvent() {
			// ...
		}
		       
		// inherit from glow.events.Event
		glow.util.extend(RocketEvent, glow.events.Event, {
			getVector: function() {
				return // ...
			}
		});
		       
		// firing the event
		rocketInstance.fire( 'landingGearDown', new RocketEvent() );
		       
		// how a user would listen to the event
		rocketInstance.on('landingGearDown', function(rocketEvent) {
			var vector = rocketEvent.getVector();
		});
	*/
		
	events.Event = function(obj) {			
		if (obj) {
			glow.util.apply(this, obj);
		}
	};
	var eventProto = events.Event.prototype;
	/**
	@name glow.events.Event#attachedTo
	@type {Object}
	@description The object the listener was attached or delegated to.
	*/

		
	/**
	@name glow.events.Event#preventDefault
	@function
	@description Prevent the default action of the event.
		Eg, if the click event on a link is cancelled, the link
		is not followed.
		       
		Returning false from an event listener has the same effect
		as calling this function.
		       
		For custom events, it's down to whatever fired the event
		to decide what to do in this case. See {@link glow.events.Event#defaultPrevented defaultPrevented}
		       
	@example
		myLinks.on('click', function(event) {
			event.preventDefault();
		});
		       
		// same as...
		       
		myLinks.on('click', function(event) {
			return false;
		});
	*/
	
	eventProto.preventDefault = function () {	
		this._defaultPrevented = true;		
	};

		
	/**
	@name glow.events.Event#defaultPrevented
	@function
	@description Has the default been prevented for this event?
		This should be used by whatever fires the event to determine if it should
		carry out of the default action.
		
	@returns {Boolean} Returns true if {@link glow.events.Event#preventDefault preventDefault} has been called for this event.
		
	@example
		// fire the 'show' event
		// read if the default action has been prevented
		if ( overlayInstance.fire('show').defaultPrevented() == false ) {
		    // go ahead and show
		}
	*/
	
	eventProto.defaultPrevented = function () {
		return this._defaultPrevented;
	};

	
	/* Export */
	glow.events = events;
});
Glow.provide(function(glow) {
	var document = window.document,
		undef = undefined,
		domEventHandlers = [], // like: domEventHandlers[uniqueId][eventName].count, domEventHandlers[uniqueId][eventName].callback
		// shortcuts to aim compression
		events = glow.events,
		_callListeners = events._callListeners,
		_getPrivateEventKey = events._getPrivateEventKey,
		// used for feature detection
		supportsActivateDeactivate = (document.createElement('div').onactivate !== undefined);
	
	/** 
		@name glow.events.DomEvent
		@constructor
		@extends glow.events.Event
		
		@param {Event|string} nativeEvent A native browser event read properties from, or the name of a native event.
		
		@param {Object} [properties] Properties to add to the Event instance.
		   Each key-value pair in the object will be added to the Event as
		   properties
		
		@description Describes a DOM event that occurred
		   You don't need to create instances of this class if you're simply
		   listening to events. One will be provided as the first argument
		   in your callback.
	*/
	function DomEvent(e, properties) {
		/** 
			@name glow.events.DomEvent#nativeEvent
			@type {Event | MouseEvent | UIEvent}
			@description The native event object provided by the browser.
		 */
		this.nativeEvent = e;
		
		/** 
			@name glow.events.DomEvent#type
			@type {string}
			@description The native type of the event, like 'click' or 'keydown'.
		 */
		this.type = e.type;
		
		/** 
			@name glow.events.DomEvent#source
			@type {HTMLElement}
			@description The element that the event originated from.
				For example, you could attach a listener to an <ol> element to listen for
				clicks. If the user clicked on an <li> the source property would be the
				<li> element, and {@link glow.DomEvent#attachedTo attachedTo} would be
				the <ol>.
		*/
		this.source = e.target || e.srcElement || undefined;
		
		// some rare cases crop up in Firefox where the source is a text node
		if (this.source && this.source.nodeType === 3) {
			this.source = this.source.parentNode;
		}
		
		/** 
			@name glow.events.DomEvent#related
			@type {HTMLElement}
			@description A related HTMLElement
				For mouseover / mouseenter events, this will refer to the previous element
				the mouse was over.
				
				For mouseout / mouseleave events, this will refer to the element the mouse
				is now over.
		*/
		this.related = e.relatedTarget || (this.type == 'mouseover' ? e.fromElement : e.toElement);
		
		/** 
			@name glow.events.DomEvent#shiftKey
			@type {boolean | undefined}
			@description Was the shift key pressed during the event?
		*/
		this.shiftKey = (e.shiftKey === undef)? undef : !!e.shiftKey;
		
		/** 
			@name glow.events.DomEvent#altKey
			@type {boolean | undefined}
			@description Was the alt key pressed during the event?
		*/
		this.altKey = (e.altKey === undef)? undef : !!e.altKey;
		
		/** 
			@name glow.events.DomEvent#ctrlKey
			@type {boolean | undefined}
			@description Was the ctrl key pressed during the event?
		*/
		this.ctrlKey = (e.ctrlKey === undef)? undef : !!e.ctrlKey;
		
		/**
			@name glow.events.DomEvent#button
			@type {number | undefined}
			@description A number representing which button was pressed.
				0 for the left button, 1 for the middle button or 2 for the right button.
		*/
		this.button = glow.env.ie ? (e.button & 1 ? 0 : e.button & 2 ? 2 : 1) : e.button;
		
		/** 
			@name glow.events.DomEvent#mouseTop
			@type {number}
			@description The vertical position of the mouse pointer in the page in pixels.
		*/
		/** 
			@name glow.events.DomEvent#mouseLeft
			@type {number}
			@description The horizontal position of the mouse pointer in the page in pixels.
		*/
		if (e.pageX !== undef || e.pageY !== undef) {
			this.mouseTop = e.pageY;
			this.mouseLeft = e.pageX;
		}
		else if (e.clientX !== undef || e.clientY !== undef) {
			this.mouseTop = e.clientY + document.body.scrollTop + document.documentElement.scrollTop;
			this.mouseLeft = e.clientX + document.body.scrollLeft + document.documentElement.scrollLeft;
		}
		
		/** 
			@name glow.events.DomEvent#wheelData
			@type {number}
			@description The number of clicks the mouse wheel moved.
				Up values are positive, down values are negative.
		*/
		if (this.type == 'mousewheel') {
			// this works in latest opera, but have read that it needs to be switched in direction
			// if there was an opera bug, I can't find which version it was fixed in
			this.wheelDelta =
				e.wheelDelta ? e.wheelDelta / 120 :
				e.detail ? - e.detail / 3 :
				0;
		}
		
		for (var key in properties) {
			this[key] = properties[key];
		}
	}
	
	glow.util.extend(DomEvent, events.Event, {
		// no docs for this as it simply adds DOM behaviour to glow.events.Event#preventDefault
		preventDefault: function() {
			var nativeEvent = this.nativeEvent;
			if (nativeEvent) {
				nativeEvent.preventDefault && nativeEvent.preventDefault();
				nativeEvent.returnValue = false;
			}
			// call the original method
			events.Event.prototype.preventDefault.call(this);
			return this;
		},
		/**
			@name glow.events.DomEvent#stopPropagation
			@function
			@description Stop an event bubbling any further.
				For instance, if you had 2 click listeners, one on a link and
				one on a parent element, if you stopped the event propogating in the
				link listener, the event will never be fired on the parent element.
			
			@returns this
		*/
		stopPropagation: function() {
			var nativeEvent = this.nativeEvent;
			
			if (nativeEvent) {
				// the ie way
				nativeEvent.cancelBubble = true;
				// the proper way
				nativeEvent.stopPropagation && nativeEvent.stopPropagation();
			}
			return this;
		}
	});
	
	/**
		Add listener for an event fired by the browser.
		@private
		@name glow.events._addDomEventListener
		@see glow.NodeList#on
		@function
	*/
	events._addDomEventListener = function(nodeList, eventName) {
		var i = nodeList.length, // TODO: should we check that this nodeList is deduped?
			attachTo,
			id;
	
		while (i--) {
			attachTo = nodeList[i];

			id = _getPrivateEventKey(attachTo);

			// check if there is already a handler for this kind of event attached
			// to this node (which will run all associated callbacks in Glow)
			if (!domEventHandlers[id]) { domEventHandlers[id] = {}; }

			if (domEventHandlers[id][eventName] && domEventHandlers[id][eventName].count > 0) { // already have handler in place
				domEventHandlers[id][eventName].count++;
				continue;
			}

			// no bridge in place yet
			domEventHandlers[id][eventName] = { count:1 };
			
			// attach a handler to tell Glow to run all the associated callbacks
			(function(attachTo) {
				var handler = domHandle(attachTo, eventName);
				
				if (attachTo.addEventListener) { // like DOM2 browsers	
					attachTo.addEventListener(handler.domName, handler, (eventName === 'focus' || eventName === 'blur')); // run in bubbling phase except for focus and blur, see: http://www.quirksmode.org/blog/archives/2008/04/delegating_the.html
				}
				else if (attachTo.attachEvent) { // like IE
					attachTo.attachEvent('on' + handler.domName, handler);
				}
				// older browsers?
				
				domEventHandlers[id][eventName].callback = handler;
			})(attachTo);
		}
	}
	
	function domHandle(attachTo, eventName) {
		var handler;
		
		if (eventName === 'mouseenter' || eventName === 'mouseleave') {
			// mousenter and mouseleave handle their own delegation as its non-standard
			handler = function(nativeEvent) {
				var domEvent = new DomEvent(nativeEvent),
					container,
					selector,
					elementsToTest = _getDelegateMatches(attachTo, eventName, domEvent);
				
				// add this element to the delegates
				elementsToTest.push( [attachTo] );
				
				for (var i = 0, leni = elementsToTest.length; i < leni; i++) {
					container = elementsToTest[i][0];
					selector = elementsToTest[i][1];
					
					if (!new glow.NodeList(container).contains(domEvent.related)) {
						_callListeners(attachTo, selector ? eventName + '/' + selector : eventName, domEvent, container); // fire() returns result of callback
					}
				}
				return !domEvent.defaultPrevented();
			};
			
			handler.domName = (eventName === 'mouseenter') ? 'mouseover' : 'mouseout';
		}
		// handle blur & focus differently for IE so it bubbles
		else if ( supportsActivateDeactivate && (eventName === 'focus' || eventName === 'blur') ) {
			// activate and deactivate are like focus and blur but bubble
			// However, <body> and <html> also activate so we need to fix that
			handler = function(nativeEvent) {
				var nodeName = nativeEvent.srcElement.nodeName;
				if (nodeName !== 'HTML' && nodeName !== 'BODY') {
					_callDomListeners( attachTo, eventName, new DomEvent(nativeEvent) );
				}
			}
			
			handler.domName = (eventName === 'focus') ? 'activate' : 'deactivate';
		}
		else {
			handler = function(nativeEvent) {
				var domEvent = new DomEvent(nativeEvent);
				_callDomListeners(attachTo, eventName, domEvent); // fire() returns result of callback
				
				return !domEvent.defaultPrevented();
			};
			
			handler.domName = eventName;
		}
		
		return handler;
	}
	
	
	/**
		Remove listener for an event fired by the browser.
		@private
		@name glow.events._removeDomEventListener
		@see glow.NodeList#detach
		@function
	*/
	events._removeDomEventListener = function(nodeList, eventName) {
		var i = nodeList.length,
			attachTo,
			id,
			bridge,
			handler;
			
		while (i--) {
			attachTo = nodeList[i];
			
			// skip if there is no bridge for this kind of event attached
			id = _getPrivateEventKey(attachTo);
			if (!domEventHandlers[id] || !domEventHandlers[id][eventName]) { continue; }

			bridge = domEventHandlers[id][eventName];
			
			// one less listener associated with this event
			if ( !--bridge.count ) {
				// no more listeners associated with this event
				handler = bridge.callback;
				
				if (attachTo.removeEventListener) { // like DOM2 browsers	
					attachTo.removeEventListener(handler.domName, handler, (eventName === 'focus' || eventName === 'blur')); // run in bubbling phase except for focus and blur, see: http://www.quirksmode.org/blog/archives/2008/04/delegating_the.html
				}
				else if (attachTo.detachEvent) { // like IE
					attachTo.detachEvent('on' + handler.domName, handler);
				}
				domEventHandlers[id][eventName] = undefined;
			}
		}
	}

// see: http://developer.yahoo.com/yui/3/event/#eventsimulation
// see: http://developer.yahoo.com/yui/docs/YAHOO.util.UserAction.html
// 	function simulateDomEvent(nodeList, domEvent) {
// 		var i = nodeList.length,
// 			eventName = domEvent.type,
// 			nativeEvent,
// 			node,
// 			fire;
// 		
// 		if (document.createEvent) {
// 			var nativeEvent = document.createEvent('MouseEvent'); // see: 
// 			nativeEvent.initEvent(eventName, true, true);
// 			
// 			fire = function(el) {
// 				return !el.dispatchEvent(nativeEvent);
// 			}
// 		}
// 		else {
// 			fire = function(el) {
// 				var nativeEvent = document.createEventObject(); 
// 				return el.fireEvent('on'+eventName, nativeEvent);
// 			}
// 		}
// 		
// 		while (i--) {
// 			node = nodeList[i];
// 			if (node.nodeType !== 1) { continue; }
// 			fire(node);
// 		}
// 	}

	/*
		The following is a proposal for dealing with event delegation without
		multiple bridges. This allows us to have only one listener per element per event
		therefore only one search for delegates per event.
	*/
	
	// structure:
	// delegates[eventId][eventName][selector] = number of delegates listening for that selector (for that event on that element)
	var delegates = {}
	
	/**
		@name glow.events._registerDelegate
		@private
		@function
		@description Register a delegated event
			This allows selectors for a given element & eventName to be retrieved later
		
		@param {glow.NodeList} nodeList Elements to register
		@param {string} eventName
		@param {string} selector Selector to match for the delegate
	*/
	events._registerDelegate = function(nodeList, eventName, selector) {
		var id,
			i = nodeList.length,
			delegatesForEvent;
		
		while (i--) {
			id = _getPrivateEventKey( nodeList[i] );
			delegates[id] = delegates[id] || {};
			delegatesForEvent = delegates[id][eventName] = delegates[id][eventName] || {};
			// increment the count or set it to 1
			delegatesForEvent[selector] = delegatesForEvent[selector] + 1 || 1;
		}
	};
	
	/**
		@name glow.events._unregisterDelegate
		@private
		@function
		@description Unregister a delegated event
		
		@param {glow.NodeList} nodeList Elements to unregister
		@param {string} eventName
		@param {string} selector Selector to match for the delegate
	*/
	events._unregisterDelegate = function(nodeList, eventName, selector) {
		var id,
			selectorCounts,
			i = nodeList.length;
		
		while (i--) {
			id = _getPrivateEventKey( nodeList[i] );
			if ( !delegates[id] || !( selectorCounts = delegates[id][eventName] ) ) { continue; }
			
			// either decrement the count or delete the entry
			if ( selectorCounts[selector] && --selectorCounts[selector] === 0 ) {
				delete selectorCounts[selector];
			}
		}
	};
	
	/**
		@name glow.events._getDelegateMatches
		@private
		@function
		@description Get the elements which qualify for a delegated event
		
		@param {HTMLElement} element Element the listener is attached to
		@param {string} eventName
		@param {glow.events.DomEvent} event DOM event for the original event
			The events source will be used as a place to start searching
			
		@returns {Array[]} An array of arrays like [matchedNode, selectorMatched]
	*/
	var _getDelegateMatches = events._getDelegateMatches = function(element, eventName, event) {
		var id = _getPrivateEventKey(element),
			selectorCounts,
			selector,
			node,
			r = [];
		
		// get delegated listeners
		if ( delegates[id] && ( selectorCounts = delegates[id][eventName] ) ) {
			for (selector in selectorCounts) {
				node = event.source;
				// if the source matches the selector
				while (node && node !== element) {
					if (glow._sizzle.matches( selector, [node] ).length) {
						r.push( [node, selector] );
						break;
					}
					
					node = node.parentNode;
				}
			}
		}
		return r;
	}
	
	/**
		@name glow.events._callDomListeners
		@private
		@function
		@description Call delegated listeners and normal listeners for an event
			Events that don't bubble (like mouseenter and mouseleave) need
			to handle their own delegation rather than use this.
		
		@param {HTMLElement} element Element to fire event on
		@param {string} eventName
		@param {glow.events.DomEvent} event
			
		@returns {glow.events.DomEvent} Original event passed in
	*/
	var _callDomListeners = events._callDomListeners = function(element, eventName, event) {
		var delegateMatches = _getDelegateMatches(element, eventName, event);
		
		// call delegated listeners
		for (var i = 0, leni = delegateMatches.length; i < leni; i++) {
			event.attachedTo = delegateMatches[i][0];
			_callListeners( element, eventName + '/' + delegateMatches[i][1], event, delegateMatches[i][0] );
		}
		
		// call non-delegated listeners
		event.attachedTo = element;
		_callListeners(element, eventName, event);
		
		return event;
	}
	
	// export
	events.DomEvent = DomEvent;
});
Glow.provide(function(glow) {
	var document = window.document,
		undefined,
        keyboardEventProto,
		env = glow.env,
		// the keyCode for the last keydown (returned to undefined on keyup)
		activeKey,
		// the charCode for the last keypress (returned to undefined on keyup & keydown)
		activeChar,
		DomEvent = glow.events.DomEvent,
		_callDomListeners = glow.events._callDomListeners,
		_getPrivateEventKey = glow.events._getPrivateEventKey,
		// object of event names & listeners, eg:
		// {
		//    eventId: [
		//        2, // the number of glow listeners added for this node
		//        keydownListener,
		//        keypressListener,
		//        keyupListener
		//    ]
		// }
		// This lets us remove these DOM listeners from the node when the glow listeners reaches zero
		eventKeysRegistered = {}; 
	
	/** 
		@name glow.events.KeyboardEvent
		@constructor
		@extends glow.events.DomEvent
		
		@description Describes a keyboard event.
		   You don't need to create instances of this class if you're simply
		   listening to events. One will be provided as the first argument
		   in your callback.
		   
		@param {Event} nativeEvent A native browser event read properties from.
		
		@param {Object} [properties] Properties to add to the Event instance.
		   Each key-value pair in the object will be added to the Event as
		   properties.
	*/
	function KeyboardEvent(nativeEvent) {
		if (activeKey) {
			this.key = keyCodeToId(activeKey);
		}
		if (activeChar) {
			this.keyChar = String.fromCharCode(activeChar);
		}
		DomEvent.call(this, nativeEvent);
	}
    
    glow.util.extend(KeyboardEvent, DomEvent, {
        /** 
            @name glow.events.KeyboardEvent#key
            @type {string}
            @description The key pressed
				This is a string representing the key pressed.
				
				Alphanumeric keys are represented by 0-9 and a-z (always lowercase). Other safe cross-browser values are:
				
				<ul>
					<li>backspace</li>
					<li>tab</li>
					<li>return</li>
					<li>shift</li>
					<li>alt</li>
					<li>escape</li>
					<li>space</li>
					<li>pageup</li>
					<li>pagedown</li>
					<li>end</li>
					<li>home</li>
					<li>left</li>
					<li>up</li>
					<li>right</li>
					<li>down</li>
					<li>insert</li>
					<li>delete</li>
					<li>;</li>
					<li>=</li>
					<li>-</li>
					<li>f1</li>
					<li>f2</li>
					<li>f3</li>
					<li>f4</li>
					<li>f5</li>
					<li>f6</li>
					<li>f7</li>
					<li>f8</li>
					<li>f9</li>
					<li>f10</li>
					<li>f11</li>
					<li>f12</li>
					<li>numlock</li>
					<li>scrolllock</li>
					<li>pause</li>
					<li>,</li>
					<li>.</li>
					<li>/</li>
					<li>[</li>
					<li>\</li>
					<li>]</li>
				</ul>
				
				Some keys may trigger actions in your browser and operating system, some
				are not cancelable.
                
            @example
				glow(document).on('keypress', function(event) {
					switch (event.key) {
						case 'up':
							// do stuff
							break;
						case 'down':
							// do stuff
							break;
					}
				});
        */
        key: '',
        /** 
            @name glow.events.KeyboardEvent#keyChar
            @type {string}
            @description The character entered.
                This is only available during 'keypress' events.
                
                If the user presses shift and 1, event.key will be "1", but event.keyChar
                will be "!".
                
            @example
				// only allow numbers to be entered into the ageInput field
				glow('#ageInput').on('keypress', function(event) {
					// Convert keyChar to a number and see if we get
					// a valid number back
					return !isNaN( Number(event.keyChar) );
				});
        */
        keyChar: ''
    });
	
	/** 
		@private
		@description Add a listener onto a DOM element
		
		@param {HTMLElement} elm
		@param {string} name Event name, without 'on' at the start
		@param {function} callback Callback for the event
	*/
	function addListener(elm, name, callback) {
		if (elm.addEventListener) { // like DOM2 browsers	
			elm.addEventListener(name, callback, false);
		}
		else if (elm.attachEvent) { // like IE
			elm.attachEvent('on' + name, callback);
		}
	}
	
	/** 
		@private
		@description Removes a listener onto a DOM element
		
		@param {HTMLElement} elm
		@param {string} name Event name, without 'on' at the start
		@param {function} callback Callback for the event
	*/
	function removeListener(elm, name, callback) {
		if (elm.removeEventListener) { // like DOM2 browsers	
			elm.removeEventListener(name, callback, false);
		}
		else if (elm.detachEvent) { // like IE
			elm.detachEvent('on' + name, callback);
		}
	}
	
	/** 
		@private
		@description Do we expect the browser to fire a keypress after a given keydown?
			Also fills in activeChar for webkit.

		@param {number} keyCode The keyCode from a keydown listener.
		@param {boolean} defaultPrevented Was the keydown prevented?
	*/
	function expectKeypress(keyCode, defaultPrevented) {
		var keyName;
		
		// for browsers that fire keypress for the majority of keys
		if (env.gecko || env.opera || env.webkit < 525) {
			return !noKeyPress[keyCode];
		}
		
		// for browsers that only fire keypress for printable chars
		keyName = keyCodeToId(keyCode);
		
		// is this a printable char?
		if (keyName.length === 1 || keyName === 'tab' || keyName === 'space') {
			// webkit doesn't fire keypress if the keydown has been prevented
			// take a good guess at the active char for webkit
			activeChar = ( keyNameToChar[keyName] || keyName ).charCodeAt(0);
			return !(env.webkit && defaultPrevented);
		}
		return false;
	}
	
	/** 
		@private
		@description Add the key listeners for firing glow's normalised key events.
		
		@param {HTMLElement} attachTo Element to attach listeners to.
		
		@returns {Object[]} An entry for eventKeysRegistered.
	*/
	function addDomKeyListeners(attachTo) {
		var keydownHandler,
			keypressHandler,
			keyupHandler,
			// Even though the user may only be interested in one key event,
			// we need all 3 listeners to normalise any of them.
			// Hash of which keys are down, keyed by keyCode
			// Like: {123: true, 124: false}
			keysDown = {};
		
		keydownHandler = function(nativeEvent) {
			var keyCode = nativeEvent.keyCode,
				preventDefault,
				preventDefaultKeyPress;
			
			// some browsers repeat this event while a key is held down, we don't want to do that
			if ( !keysDown[keyCode] ) {
				activeKey = keyCode;
				activeChar = undefined;
				preventDefault = _callDomListeners( attachTo, 'keydown', new KeyboardEvent(nativeEvent) ).defaultPrevented();
				keysDown[keyCode] = true;
			}
			// we want to fire a keyPress event here if the browser isn't going to fire one itself
			if ( !expectKeypress(keyCode, preventDefault) ) {
				preventDefaultKeyPress = _callDomListeners( attachTo, 'keypress', new KeyboardEvent(nativeEvent) ).defaultPrevented();
			}
			// return false if either the keydown or fake keypress event was cancelled
			return !(preventDefault || preventDefaultKeyPress);
		};
		
		keypressHandler = function(nativeEvent) {
			var keyName, preventDefault;
			// some browsers store the charCode in .charCode, some in .keyCode
			activeChar = nativeEvent.charCode || nativeEvent.keyCode;
			keyName = keyCodeToId(activeKey);
			
			// some browsers fire this event for non-printable chars, look at the previous keydown and see if we're expecting a printable char
			if ( keyName.length > 1 && keyName !== 'tab' && keyName !== 'space' ) {
				// non-printable chars usually have an ID length greater than 1
				activeChar = undefined;
			}

			preventDefault = _callDomListeners( attachTo, 'keypress', new KeyboardEvent(nativeEvent) ).defaultPrevented();
			return !preventDefault;
		};
		
		keyupHandler = function(nativeEvent) {
			var keyCode = nativeEvent.keyCode,
				preventDefault;
			
			// set the active key so KeyboardEvent picks it up
			activeKey = keyCode;
			activeChar = undefined;
			preventDefault = _callDomListeners( attachTo, 'keyup', new KeyboardEvent(nativeEvent) ).defaultPrevented();
			keysDown[keyCode] = false;
			activeKey = undefined;
			return !preventDefault;
		};
		
		// add listeners to the dom
		addListener(attachTo, 'keydown',  keydownHandler);
		addListener(attachTo, 'keypress', keypressHandler);
		addListener(attachTo, 'keyup',    keyupHandler);
		
		return [1, keydownHandler, keypressHandler, keyupHandler];
	}
	
	/**
		@name glow.events._addKeyListener
		@private
		@function
		@description Add DOM listeners for key events fired by the browser.
			Won't add more than one.
		
		@param {glow.NodeList} nodeList Elements to add listeners to.
		
		@see glow.NodeList#on
	*/
	glow.events._addKeyListener = function(nodeList) {
		var i = nodeList.length,
			attachTo,
			eventKey;
	
		while (i--) {
			attachTo = nodeList[i];

			// get the ID for this event
			eventKey = _getPrivateEventKey(attachTo);
			
			// if we've already attached DOM listeners for this, don't add them again
			if ( eventKeysRegistered[eventKey] ) {
				// increment the number of things listening to this
				// This lets us remove these DOM listeners from the node when
				// the glow listeners reaches zero
				eventKeysRegistered[eventKey][0]++;
				continue;
			}
			else {
				eventKeysRegistered[eventKey] = addDomKeyListeners(attachTo);
			}
		}
	}
	
	/**
		@name glow.events._removeKeyListener
		@private
		@function
		@description Remove DOM listeners for key events fired by the browser
			Avoids removing DOM listeners until all Glow listeners have been removed
		
		@param {glow.NodeList} nodeList Elements to remove listeners from
		
		@see glow.NodeList#detach
	*/
	glow.events._removeKeyListener = function(nodeList) {
		var i = nodeList.length,
			attachTo,
			eventKey,
			eventRegistry;
		
		while (i--) {
			attachTo = nodeList[i];
			
			// get the ID for this event
			eventKey = _getPrivateEventKey(attachTo);
			eventRegistry = eventKeysRegistered[eventKey];
			// exist if there are no key events registered for this node
			if ( !eventRegistry ) {
				continue;
			}
			if ( --eventRegistry[0] === 0 ) {
				// our glow listener count is zero, we have no need for the dom listeners anymore
				removeListener( attachTo, 'keydown',   eventRegistry[1] );
				removeListener( attachTo, 'keypress',  eventRegistry[2] );
				removeListener( attachTo, 'keyup',     eventRegistry[3] );
				eventKeysRegistered[eventKey] = undefined;
			}
		}
	}
	/**
		@private
		@function
		@description convert a keyCode to a string name for that key
		
		@param {number} keyCode
		
		@returns {string} ID for that key. Is a letter a-z, number 0-9, or id from 'keyIds'
	*/
	function keyCodeToId(keyCode) {
		// key codes for 0-9 A-Z are the same as their char codes
		if ( (keyCode >= keyCodeA && keyCode <= keyCodeZ) || (keyCode >= keyCode0 && keyCode <= keyCode9) ) {
			return String.fromCharCode(keyCode).toLowerCase();
		}
		return keyIds[keyCode] || 'unknown' + keyCode;
	}
	
	// keyCode to key name translation
	var keyCodeA = 65,
		keyCodeZ = 90,
		keyCode0 = 48,
		keyCode9 = 57,
		// key codes for non-alphanumeric keys
		keyIds = {
			8: 'backspace',
			9: 'tab',
			13: 'return',
			16: 'shift',
			17: 'control',
			18: 'alt',
			19: 'pause',
			27: 'escape',
			32: 'space',
			33: 'pageup',
			34: 'pagedown',
			35: 'end',
			36: 'home',
			37: 'left',
			38: 'up',
			39: 'right',
			40: 'down',
			//44: 'printscreen', // Only fires keyup in firefox, IE. Doesn't fire in webkit, opera.
			45: 'insert',
			46: 'delete',
			59: ';',
			61: '=',
			//91: 'meta',
			//93: 'menu', // no keycode in opera, doesn't fire in Chrome
			
			// these are number pad numbers, but Opera doesn't distinguish them from normal number keys so we normalise on that
				96: '0', 
				97: '1',
				98: '2',
				99: '3',
				100: '4',
				101: '5',
				102: '6',
				103: '7',
				104: '8',
				105: '9',
				//106: '*', // opera fires 2 keypress events
				//107: '+', // opera fires 2 keypress events
				109: '-', // opera sees - as insert, but firefox 3.0 see the normal - key the same as the numpad one
				//110: '.', // opera sees this as n
				111: '/',
			// end of numpad
			
			112: 'f1',
			113: 'f2',
			114: 'f3',
			115: 'f4',
			116: 'f5',
			117: 'f6',
			118: 'f7',
			119: 'f8',
			120: 'f9',
			121: 'f10',
			122: 'f11',
			123: 'f12',
			144: 'numlock',
			145: 'scrolllock',
			188: ',',
			189: '-',
			190: '.',
			191: '/',
			192: "'",
			219: '[',
			220: '\\',
			221: ']',
			222: '#', // opera sees # key as 3. Pah.
			223: '`',
			//224: 'meta', // same as [ in opera
			226: '\\' // this key appears on a US layout in webkit windows
		},
		// converting key names to chars, for key names greater than 1 char
		keyNameToChar = {
			space: ' ',
			tab: '\t'
		}
		noKeyPress = {};
	
	// corrections for particular browsers :(
	if (env.gecko) {
		keyIds[107] = '=';
		
		noKeyPress = {
			16: 1,  // shift
			17: 1,  // control
			18: 1,  // alt
			144: 1, // numlock
			145: 1  // scrolllock
		};
	}
	else if (env.opera) {
		keyIds[42] = '*';
		keyIds[43] = '+';
		keyIds[47] = '/';
		keyIds[222] = "'";
		keyIds[192] = '`';
		
		noKeyPress = {
			16: 1,  // shift
			17: 1,  // control
			18: 1   // alt
		};
	}
	else if (env.webkit || env.ie) {
		keyIds[186] = ';';
		keyIds[187] = '=';
	}
	
	// export
	glow.events.KeyboardEvent = KeyboardEvent;
});
Glow.provide(function(glow) {
	var NodeListProto, undefined,
		// shortcuts to aid compression
		document = window.document,
		arraySlice = Array.prototype.slice,
		arrayPush = Array.prototype.push;
	
	/**
		@name glow.NodeList
		@constructor
		@description An array-like collection of DOM Nodes
			It is recommended to create a NodeList using the shortcut function {@link glow}.
			
		@param {string | glow.NodeList | Node | Node[] | Window} contents Items to populate the NodeList with.
			This parameter will be passed to {@link glow.NodeList#push}.
			
			Strings will be treated as CSS selectors unless they start with '<', in which
			case they'll be treated as an HTML string.
			
		@example
			// empty NodeList
			var myNodeList = glow();

		@example
			// using glow to return a NodeList then chaining methods
			glow('p').addClass('eg').append('<div>Hello!</div>');
			
		@example
			// creating an element from a string
			glow('<div>Hello!</div>').appendTo('body');
		
		@see <a href="http://wiki.github.com/jeresig/sizzle/">Supported CSS selectors</a>
	*/
	function NodeList(contents) {
		// call push if we've been given stuff to add
		contents && this.push(contents);
	}
	NodeListProto = NodeList.prototype;
	
	/**
		@name glow.NodeList#length
		@type Number
		@description Number of nodes in the NodeList
		@example
			// get the number of paragraphs on the page
			glow('p').length;
	*/
	NodeListProto.length = 0;
	
	/**
		@name glow.NodeList._strToNodes
		@private
		@function
		@description Converts a string to an array of nodes
		
		@param {string} str HTML string
		
		@returns {Node[]} Array of nodes (including text / comment nodes)
	*/
	NodeList._strToNodes = (function() {
		var	tmpDiv = document.createElement('div'),
			// these wraps are in the format [depth to children, opening html, closing html]
			tableWrap = [1, '<table>', '</table>'],
			emptyWrap = [0, '', ''],
			// Easlier Webkits won't accept <link> & <style> elms to be the only child of an element,
			// it steals them and hides them in the head for some reason. Using
			// broken html fixes it for some reason
			paddingWrap = glow.env.webkit < 526 ? [0, '', '</div>'] : [1, 'b<div>', '</div>'],
			trWrap = [3, '<table><tbody><tr>', '</tr></tbody></table>'],
			wraps = {
				caption: tableWrap,
				thead: tableWrap,
				th: trWrap,
				colgroup: tableWrap,
				tbody: tableWrap,
				tr: [2, '<table><tbody>', '</tbody></table>'],
				td: trWrap,
				tfoot: tableWrap,
				option: [1, '<select multiple="multiple">', '</select>'],
				legend: [1, '<fieldset>', '</fieldset>'],
				link: paddingWrap,
				script: paddingWrap,
				style: paddingWrap,
				'!': paddingWrap
			};

		function strToNodes(str) {
			var r = [],
				tagName = ( /^<([\w!]+)/.exec(str) || [] )[1],
				// This matches str content with potential elements that cannot
				// be a child of <div>.  elmFilter declared at top of page.
				wrap = wraps[tagName] || emptyWrap,
				nodeDepth = wrap[0],
				childElm = tmpDiv,
				exceptTbody,
				rLen = 0,
				firstChild;

			// Create the new element using the node tree contents available in filteredElm.
			childElm.innerHTML = (wrap[1] + str + wrap[2]);

			// Strip newElement down to just the required elements' parent
			while(nodeDepth--) {
				childElm = childElm.lastChild;
			}

			// pull nodes out of child
			if (wrap === tableWrap && str.indexOf('<tbody') === -1) {
				// IE7 (and earlier) sometimes gives us a <tbody> even though we didn't ask for one
				while (firstChild = childElm.firstChild) {
					if (firstChild.nodeName != 'TBODY') {
						r[rLen++] = firstChild;
					}
					childElm.removeChild(firstChild);
				}
			}
			else {
				while (firstChild = childElm.firstChild) {
					r[rLen++] = childElm.removeChild(firstChild);
				}
			}

			return r;
		}

		return strToNodes;
	})();
	
	// takes a collection and returns an array
	var collectionToArray = function(collection) {
		return arraySlice.call(collection, 0);
	};
	
	try {
		// look out for an IE bug
		arraySlice.call( document.documentElement.childNodes, 0 );
	}
	catch(e) {
		collectionToArray = function(collection) {
			// We can't use this trick on IE collections that are com-based, like HTMLCollections
			// Thankfully they don't have a constructor, so that's how we detect those
			if (collection instanceof Object) {
				return arraySlice.call(collection, 0);
			}
			var i   = collection.length,
				arr = [];
				
			while (i--) {
				arr[i] = collection[i];
			}
			return arr;
		}
	}
	
	/**
		@name glow.NodeList#push
		@function
		@description Adds nodes to the NodeList
		
		@param {string | Node | Node[] | glow.NodeList} nodes Node(s) to add to the NodeList
			Strings will be treated as CSS selectors or HTML strings.
		
		@returns {glow.NodeList}
		
		@example
			myNodeList.push('<div>Foo</div>').push('h1');
	*/
	NodeListProto.push = function(nodes) {
		/*!debug*/
			if (arguments.length !== 1) {
				glow.debug.warn('[wrong count] glow.NodeList#push expects 1 argument, not '+arguments.length+'.');
			}
		/*gubed!*/
		
		if (nodes) {
			if (typeof nodes === 'string') {
				// if the string begins <, treat it as html, otherwise it's a selector
				if (nodes.charAt(0) === '<') {
					nodes = NodeList._strToNodes(nodes);
				}
				else {
					nodes = glow._sizzle(nodes)
				}
				arrayPush.apply(this, nodes);
			}
			
			else if ( nodes.nodeType || nodes.window == nodes ) {
				if (this.length) {
					arrayPush.call(this, nodes);
				}
				else {
					this[0] = nodes;
					this.length = 1;
				}
			}
			else if (nodes.length !== undefined) {
				if (nodes.constructor != Array) {
					// convert array-like objects into an array
					nodes = collectionToArray(nodes);
				}
				arrayPush.apply(this, nodes);
			}
			/*!debug*/
			else {
				glow.debug.warn('[wrong type] glow.NodeList#push: Ignoring unexpected argument type, failing silently');
			}
			/*gubed!*/
		}
		/*!debug*/
		else {
			glow.debug.warn('[wrong type] glow.NodeList#push: Ignoring false argument type, failing silently');
		}
		/*gubed!*/
		return this;
	};
	
	/**
		@name glow.NodeList#eq
		@function
		@description Compares this NodeList to another
			Returns true if both NodeLists contain the same items in the same order
		
		@param {Node | Node[] | glow.NodeList} nodeList The NodeList to compare to.
		
		@returns {boolean}
		
		@see {@link glow.NodeList#is} for testing if a NodeList item matches a selector
		
		@example
			// the following returns true
			glow('#blah').eq( document.getElementById('blah') );
	*/
	NodeListProto.eq = function(nodeList) {
		/*!debug*/
			if (arguments.length !== 1) {
				glow.debug.warn('[wrong count] glow.NodeList#eq expects 1 argument, not ' + arguments.length + '.');
			}
			if (typeof nodeList !== 'object') {
				glow.debug.warn('[wrong type] glow.NodeList#eq expects object argument, not ' + typeof nodeList + '.');
			}
		/*gubed!*/
		
		var len = this.length,
			i = len;
		
		// normalise param to NodeList
		if ( !(nodeList instanceof NodeList) ) {
			nodeList = new NodeList(nodeList);
		}
		
		// quickly return false if lengths are different
		if (len != nodeList.length) {
			return false;
		}
		
		// loop through and return false on inequality
		while (i--) {
			if (this[i] !== nodeList[i]) {
				return false;
			}
		}
		
		return true;
	};
	
	/**
		@name glow.NodeList#slice
		@function
		@description Get a section of an NodeList
			Operates in the same way as an Array's slice method
		
		@param {number} start Start index
			If negative, it specifies a position measured from the end of the list
		
		@param {number} [end] End index
			By default, this is the end of the list. A negative end specifies
			a position measured from the end of the list.
		
		@returns {glow.NodeList} A new sliced NodeList
		
		@example
		var myNodeList = glow("<div></div><p></p>");
		myNodeList.slice(1, 2); // selects the paragraph
		myNodeList.slice(-1); // same thing, selects the paragraph
	*/
	NodeListProto.slice = function(/*start, end*/) {
		return new NodeList( arraySlice.apply(this, arguments) );
	};
	
	/**
		@name glow.NodeList#sort
		@function
		@description Sort the elements in the list.
			Items will already be in document order if a CSS selector
			was used to fetch them.
		
		@param {Function} [func] Function to determine sort order
			This function will be passed 2 elements (elementA, elementB). The function
			should return a number less than 0 to sort elementA lower than elementB
			and greater than 0 to sort elementA higher than elementB.
			
			If no function is provided, elements will be sorted in document order.
		
		@returns {glow.NodeList} A new sorted NodeList
		
		@example
			//get links in alphabetical (well, lexicographical) order
			var links = glow("a").sort(function(elementA, elementB) {
				return glow(elementA).text() < glow(elementB).text() ? -1 : 1;
			})
	*/
	NodeListProto.sort = function(func) {
		var items = collectionToArray(this),
			sortedElms = func ? items.sort(func) : glow._sizzle.uniqueSort(items);
		
		return new NodeList(sortedElms);
	};
	
	/**
		@name glow.NodeList#item
		@function
		@description Get a single item from the list as an NodeList
			Negative numbers can be used to get items from the end of the
			list.
		
		@param {number} index The numeric index of the node to return.
		
		@returns {glow.NodeList} A new NodeList containing a single item
		
		@example
			// get the html from the fourth element
			myNodeList.item(3).html();
			
		@example
			// add a class name to the last item
			myNodeList.item(-1).addClass('last');
	*/
	NodeListProto.item = function(index) {
		/*!debug*/
			if ( arguments.length !== 1 ) {
				glow.debug.warn('[wrong count] glow.NodeList#item expects 1 argument, got ' + arguments.length);
			}
		/*gubed!*/
		// TODO: test which of these methods is faster (use the current one unless significantly slower)
		return this.slice(index, (index + 1) || this.length);
		// return new NodeList( index < 0 ? this[this.length + index] : this[index] );
	};
	
	/**
		@name glow.NodeList#each
		@function
		@description Calls a function for each node in the list.
		
		@param {Function} callback The function to call for each node.
			The function will be passed 2 arguments, the index of the current item,
			and the NodeList being iterated over.
			
			Inside the function 'this' refers to the Node.
			
			Returning false from this function stops further iterations
		
		@returns {glow.NodeList}
		
		@example
			// add "link number: x" to each link, where x is the index of the link
			glow("a").each(function(i, nodeList) {
				glow(this).append(' link number: ' + i);
			});
		@example
			// breaking out of an each loop
			glow("a").each(function(i, nodeList) {
				// do stuff
				if ( glow(this).hasClass('whatever') ) {
					// we don't want to process any more links
					return false;
				}
			});
	*/
	NodeListProto.each = function(callback) {
		/*!debug*/
			if ( arguments.length !== 1 ) {
				glow.debug.warn('[wrong count] glow.NodeList#each expects 1 argument, got ' + arguments.length);
			}
			if (typeof callback != 'function') {
				glow.debug.warn('[wrong type] glow.NodeList#each expects "function", got ' + typeof callback);
			}
		/*gubed!*/
		for (var i = 0, len = this.length; i<len; i++) {
			if ( callback.call(this[i], i, this) === false ) {
				break;
			}
		}
		return this;
	};
	
	/**
		@name glow.NodeList#filter
		@function
		@description Filter the NodeList
		 
		@param {Function|string} test Filter test
			If a string is provided it's treated as a CSS selector. Elements
			which match the CSS selector are added to the new NodeList.
			
			If 'test' is a function, it will be called per node in the NodeList.
			
			The function is passed 2 arguments, the index of the current item,
			and the ElementList being itterated over.
			
			Inside the function 'this' refers to the node.
			Return true to add the element to the new NodeList.
		 
		@returns {glow.NodeList} A new NodeList containing the filtered nodes
		 
		@example
			// return images with a width greater than 320
			glow("img").filter(function () {
				return glow(this).width() > 320;
			});
		
		@example
			// Get items that don't have an alt attribute
			myElementList.filter(':not([alt])');
	*/
	NodeListProto.filter = function(test) {
		/*!debug*/
			if ( arguments.length !== 1 ) {
				glow.debug.warn('[wrong count] glow.NodeList#filter expects 1 argument, got ' + arguments.length);
			}
			if ( !/^(function|string)$/.test(typeof test) ) {
				glow.debug.warn('[wrong type] glow.NodeList#each expects function/string, got ' + typeof test);
			}
		/*gubed!*/
		var r = [],
			ri = 0;
		
		if (typeof test === 'string') {
			r = glow._sizzle.matches(test, this);
		}
		else {	
			for (var i = 0, len = this.length; i<len; i++) {
				if ( test.call(this[i], i, this) ) {
					r[ri++] = this[i];
				}
			}
		}
		
		return new NodeList(r);
	};

	
	/**
		@name glow.NodeList#is
		@function
		@description Tests if the first element matches a CSS selector

		@param {string} selector CSS selector
		
		@returns {boolean}
		
		@example
			if ( myNodeList.is(':visible') ) {
				// ...
			}
	*/
	NodeListProto.is = function(selector) {
		/*!debug*/
			if ( arguments.length !== 1 ) {
				glow.debug.warn('[wrong count] glow.NodeList#is expects 1 argument, got ' + arguments.length);
			}
			if ( typeof selector !== 'string' ) {
				glow.debug.warn('[wrong type] glow.NodeList#is expects string, got ' + typeof selector);
			}
		/*gubed!*/
		if ( !this[0] ) {
			return false;
		}
		return !!glow._sizzle.matches( selector, [ this[0] ] ).length;
	};
	
	// export
	glow.NodeList = NodeList;
});
Glow.provide(function(glow) {
	var undef
		, NodeListProto = glow.NodeList.prototype
	
		/**
			@private
			@name glow.NodeList-dom0PropertyMapping
			@description Mapping of HTML attribute names to DOM0 property names.
		*/
		, dom0PropertyMapping = { // keys must be lowercase
			'class'     : 'className',
			'for'       : 'htmlFor',
			'maxlength' : 'maxLength'
		}
		
		/**
			@private
			@name glow.NodeList-dataPropName
			@type String
			@description The property name added to the DomElement by the NodeList#data method.
		*/
		, dataPropName = '_uniqueData' + glow.UID
		
		/**
			@private
			@name glow.NodeList-dataIndex
			@type String
			@description The value of the dataPropName added by the NodeList#data method.
		*/
		, dataIndex = 1 // must be a truthy value
			
		/**
			@private
			@name glow.NodeList-dataCache
			@type Object
			@description Holds the data used by the NodeList#data method.
			
			The structure is like:
			[
				{
					myKey: "my data"
				}
			]
		*/
		, dataCache = [];
			
	/**
	@name glow.NodeList#addClass
	@function
	@description Adds a class to each node.

	@param {string} name The name of the class to add.

	@returns {glow.NodeList}

	@example
		glow("#login a").addClass("highlight");
	*/
	NodeListProto.addClass = function(name) {
		var i = this.length;
		
		/*!debug*/
			if (arguments.length !== 1) {
				glow.debug.warn('[wrong count] glow.NodeList#addClass expects 1 argument, not '+arguments.length+'.');
			}
			else if (typeof arguments[0] !== 'string') {
				glow.debug.warn('[wrong type] glow.NodeList#addClass expects argument 1 to be of type string, not '+typeof arguments[0]+'.');
			}
		/*gubed!*/
		
		while (i--) {
			if (this[i].nodeType === 1) {
				_addClass(this[i], name);
			}
		}
		
		return this;
	};
	
	function _addClass(node, name) { // TODO: handle classnames separated by non-space characters?
		if ( (' ' + node.className + ' ').indexOf(' ' + name + ' ') === -1 ) {
			node.className += (node.className? ' ' : '') + name;
		}
	}
	
	/**
	@name glow.NodeList#attr
	@function
	@description Gets or sets attributes.

		When getting an attribute, it is retrieved from the first
		node in this NodeList. Setting attributes applies the change
		to each element in this NodeList.

		To set an attribute, pass in the name as the first
		parameter and the value as a second parameter.

		To set multiple attributes in one call, pass in an object of
		name/value pairs as a single parameter.

		For browsers that don't support manipulating attributes
		using the DOM, this method will try to do the right thing
		(i.e. don't expect the semantics of this method to be
		consistent across browsers as this is not possible with
		currently supported browsers).

	@param {string | Object} name The name of the attribute, or an object of name/value pairs
	@param {string} [value] The value to set the attribute to.

	@returns {string | undefined | glow.NodeList}

		When setting attributes this method returns its own NodeList, otherwise
		returns the attribute value. The attribute name is always treated as
		case-insensitive. When getting, the returned value will be of type string unless
		that particular attribute was never set and there is no default value, in which
		case the returned value will be an empty string.

	@example
		var myNodeList = glow(".myImgClass");

		// get an attribute
		myNodeList.attr("class");

		// set an attribute
		myNodeList.attr("class", "anotherImgClass");

		// set multiple attributes
		myNodeList.attr({
		  src: "a.png",
		  alt: "Cat jumping through a field"
		});
	 */
	 // see: http://tobielangel.com/2007/1/11/attribute-nightmare-in-ie/
	NodeListProto.attr = function(/*arguments*/) {
		var args = arguments,
			argsLen = args.length,
			thisLen = this.length,
			keyvals,
			name = keyvals = args[0], // using this API: attr(name) or attr({key: val}) ?
			dom0Property = '',
			node,
			attrNode;
		
		/*!debug*/
			if (arguments.length === 2 && typeof arguments[0] !== 'string') {glow.debug.warn('[wrong type] glow.NodeList#attr expects name to be of type string, not '+typeof arguments[0]+'.'); }
			else if (arguments.length === 1 && (typeof arguments[0] !== 'string' && arguments[0].constructor !== Object)) {glow.debug.warn('[wrong type] glow.NodeList#attr expects argument 1 to be of type string or an instance of Object.'); }
			else if (arguments.length === 0 ||  arguments.length > 2) { glow.debug.warn('[wrong count] glow.NodeList#attr expects 1 or 2 arguments, not '+arguments.length+'.'); }
		/*gubed!*/
		
		if (this.length === 0) { // is this an empty nodelist?
			return (argsLen > 1)? this : undef;
		}
		
		if (typeof keyvals === 'object') { // SETting value from {name: value} object
			for (name in keyvals) {
				if (!keyvals.hasOwnProperty(name)) { continue; }
				
				// in IE6 and IE7 the attribute name needs to be translated into dom property name
				if (glow.env.ie < 8) {
					dom0Property = dom0PropertyMapping[name.toLowerCase()];
				}
				
				var i = thisLen;
				while (i--) {
					node = this[i];
					
					if (node.nodeType !== 1) { continue; }
					
					if (dom0Property) {
						node[dom0Property] = keyvals[name];
					}
					else {
						node.setAttribute(name, keyvals[name], 0); // IE flags, 0: case-insensitive
					}
				}
			}
			
			return this;
		}
		else {
			node = this[0];
				
			if (node.nodeType !== 1) {
				return (argsLen > 1)? this : undef;
			}

			if (argsLen === 1) { // GETting value from name. see http://reference.sitepoint.com/javascript/Element/getAttribute
				if ( glow.env.ie && (name === 'href' || name === 'src') ) {
					value = node.getAttribute(name, 2); // IE flags, 0: case-insensitive + 2: exactly as set
					return (value === null)? '' : value;
				}
				else if (node.attributes[name]) { // in IE node.getAttributeNode sometimes returns unspecified default values so we look for specified attributes if we can
					return (!node.attributes[name].specified)? '' : node.attributes[name].value;
				}
				else if (node.getAttributeNode) { // in IE getAttribute() does not always work so we use getAttributeNode if we can
					attrNode = node.getAttributeNode(name, 0);
					return (attrNode === null)? '' : attrNode.value;
				}
				else {
					value = node.getAttribute(name, 2); // IE flags, 0: case-insensitive + 2: exactly as set
					return (value === null)? '' : value;
				}	
			}
			else { // SETting a single value like attr(name, value), normalize to an keyval object
				if (glow.env.ie < 8) {
					dom0Property = dom0PropertyMapping[name.toLowerCase()];
				}
				
				if (dom0Property) {
					node[dom0Property] = args[1];
				}
				else {
					node.setAttribute(name, args[1], 0); // IE flags, 0: case-insensitive
				}
				return this;
			}
		}
	};
	/**
		Copies the data from one nodelist to another
		@private
		@name glow.NodeList._copyData
		@see glow.NodeList#clone
		@function
	*/
	glow.NodeList._copyData = function(from, to){
		var i = to.length,
			data;
		
		while (i--) {
			data = dataCache[ from[i][dataPropName] ];
			data && to.slice(i, i+1).data(data);
		}
	}

	/**
	@name glow.NodeList#data
	@function
	@description Use this to safely attach arbitrary data to any DOM Element.
	
	This method is useful when you wish to avoid memory leaks that are possible when adding your own data directly to DOM Elements.
	
	When called with no arguments, will return glow's entire data store for the first node in this NodeList.
	
	Otherwise, when given a name, will return the associated value from the first node in this NodeList.
	
	When given both a name and a value, will store that data on every node in this NodeList.
	
	Optionally you can pass in a single object composed of multiple name, value pairs.
	
	@param {string|Object} [key] The name of the value in glow's data store.
	@param {Object} [val] The value you wish to associate with the given name.
	@see glow.NodeList#removeData
	@example
	
	glow("p").data("tea", "milky");
	var colour = glow("p").data("tea"); // milky
	@returns {Object} When setting a value this method can be chained, as in that case it will return itself.
	@see glow.NodeList#removeData
	*/
	NodeListProto.data = function (key, val) { /*debug*///console.log("data("+key+", "+val+")");
		var args = arguments,
			argsLen = args.length,
			keyvals = key, // like: data({key: val}) or data(key, val)
			index,
			node;
			
		/*!debug*/
			if (arguments.length === 2 && typeof arguments[0] !== 'string') {glow.debug.warn('[wrong type] glow.NodeList#data expects name argument to be of type string.'); }
			else if (arguments.length === 1 && (typeof arguments[0] !== 'string' && arguments[0].constructor !== Object)) {glow.debug.warn('[wrong type] glow.NodeList#data expects argument 1 to be of type string or an instance of Object.'); }
			else if (arguments.length > 2) { glow.debug.warn('[wrong count] glow.NodeList#data expects 0, 1 or 2 arguments.'); }
		/*gubed!*/
		
		if (argsLen > 1) { // SET key, val on every node
			var i = this.length;
			while (i--) {
				node = this[i];
				if (node.nodeType !== 1) { continue; }
				
				index = node[''+dataPropName];

				if (!index) { // assumes index is always > 0
					index = dataIndex++;
					
					node[dataPropName] = index;
					dataCache[index] = {};
				}
				dataCache[index][key] = val;
			}
			
			return this; // chainable with (key, val) signature
		}
		else if (typeof keyvals === 'object') { // SET keyvals on every node
			var i = this.length;
			while (i--) {
				node = this[i];
				if (node.nodeType !== 1) { continue; }
				
				index = node[dataPropName];
				if (!index) { // assumes index is always > 0
					index = dataIndex++;
					
					node[dataPropName] = index;
					dataCache[index] = {};
				}
				for (key in keyvals) {
					dataCache[index][key] = keyvals[key];
				}
			}
			
			return this; // chainable with ({key, val}) signature
		}
		else { // GET from first node
			node = this[0];
			if (node === undef || node.nodeType !== 1) { return undef; }
				
			if ( !(index = node[dataPropName]) ) {
				return undef;
			}

			if (key !== undef) {
				return dataCache[index][key];
			}
			
			// get the entire data cache object for this node
			return dataCache[index];
		}
	};
	
	/**
	@name glow.NodeList#hasAttr
	@function
	@description Does the node have a particular attribute?
		
		The first node in this NodeList is tested.
		
	@param {string} name The name of the attribute to test for.

	@returns {boolean|undefined} Returns undefined if the first node is not an element,
	or if the NodeList is empty, otherwise returns true/false to indicate if that attribute exists
	on the first element.

	@example
		if ( glow("#myImg").hasAttr("alt") ){
			// ...
		}
	*/
	NodeListProto.hasAttr = function(name) {
		var node;
		
		/*!debug*/
			if (arguments.length !== 1) { glow.debug.warn('[wrong count] glow.NodeList#hasAttr expects 1 argument.'); }
			else if (typeof arguments[0] !== 'string') {glow.debug.warn('[wrong type] glow.NodeList#hasAttr expects argument 1 to be of type string.'); }
		/*gubed!*/
		
		node = this[0];
		
		if (this.length && node.nodeType === 1) {
			if (node.attributes[name]) { // is an object in  IE, or else: undefined in IE < 8, null in IE 8
				return !!node.attributes[name].specified;
			}
			
			if (node.hasAttribute) { return node.hasAttribute(name); } // like FF, Safari, etc
			else { return node.attributes[name] !== undef; } // like IE7
		}
	};
	
	/**
	@name glow.NodeList#hasClass
	@function
	@description Does the node have a particular class?

		The first node in this NodeList is tested.

	@param {string} name The name of the class to test for.

	@returns {boolean}

	@example
		if ( glow("#myInput").hasClass("errored") ){
			// ...
		}
	*/
	NodeListProto.hasClass = function (name) {
		/*!debug*/
			if (arguments.length !== 1) { glow.debug.warn('[wrong count] glow.NodeList#hasClass expects 1 argument.'); }
			else if (typeof arguments[0] !== 'string') {glow.debug.warn('[wrong type] glow.NodeList#hasClass expects argument 1 to be of type string.'); }
		/*gubed!*/
		
		if (this.length && this[0].nodeType === 1) {
			return ( (' ' + this[0].className + ' ').indexOf(' ' + name + ' ') > -1 );
		}
	};
	
	/**
	@name glow.NodeList#prop
	@function
	@description Gets or sets node properties.
	
		This function gets / sets node properties, to get attributes,
		see {@link glow.NodeList#attr NodeList#attr}.
		
		When getting a property, it is retrieved from the first
		node in this NodeList. Setting properties to each element in
		this NodeList.
		
		To set multiple properties in one call, pass in an object of
		name/value pairs.
		
	@param {string | Object} name The name of the property, or an object of name/value pairs
	@param {string} [value] The value to set the property to.

	@returns {string | glow.NodeList}

		When setting properties it returns the NodeList, otherwise
		returns the property value.

	@example
		var myNodeList = glow("#formElement");

		// get the node name
		myNodeList.prop("nodeName");

		// set a property
		myNodeList.prop("_secretValue", 10);

		// set multiple properties
		myNodeList.prop({
			checked: true,
			_secretValue: 10
		});
	*/
	NodeListProto.prop = function(name, val) {
		var hash = name,
			argsLen = arguments.length;
		
		/*!debug*/
			if (arguments.length === 1 && (typeof name !== 'string' && name.constructor !== Object)) {glow.debug.warn('[wrong type] glow.NodeList#prop expects argument 1 to be of type string or Object.'); }
			else if (arguments.length === 2 && typeof name !== 'string') {glow.debug.warn('[wrong type] glow.NodeList#prop expects name to be of type string.'); }
			else if (arguments.length === 0 || arguments.length > 2) { glow.debug.warn('[wrong count] glow.NodeList#prop expects 1 or 2 arguments.'); }
		/*gubed!*/
		
		if (this.length === 0) return;
		
		if (argsLen === 2 && typeof name === 'string') {
			for (var i = 0, ilen = this.length; i < ilen; i++) {
				if (this[i].nodeType === 1) { this[i][name] = val; }
			}
			return this;
		}
		else if (argsLen === 1 && hash.constructor === Object) {
			for (var key in hash) {
				for (var i = 0, ilen = this.length; i < ilen; i++) {
					if (this[i].nodeType === 1) { this[i][key] = hash[key]; }
				}
			}
			return this;
		}
		else if (argsLen === 1 && typeof name === 'string') {
			if (this[0].nodeType === 1) { return this[0][name]; }
		}
		else {
			throw new Error('Invalid parameters.');
		}
	};
	
	/**
	@name glow.NodeList#removeAttr
	@function
	@description Removes an attribute from each node.

	@param {string} name The name of the attribute to remove.

	@returns {glow.NodeList}

	@example
		glow("a").removeAttr("target");
	*/
	NodeListProto.removeAttr = function (name) {
		var dom0Property;
		
		/*!debug*/
			if (arguments.length !== 1) { glow.debug.warn('[wrong count] glow.NodeList#removeAttr expects 1 argument.'); }
			else if (typeof arguments[0] !== 'string') {glow.debug.warn('[wrong type] glow.NodeList#removeAttr expects argument 1 to be of type string.'); }
		/*gubed!*/
	
		for (var i = 0, leni = this.length; i < leni; i++) {
			if (this[i].nodeType === 1) {
				if (glow.env.ie < 8) {
					if ( (dom0Property = dom0PropertyMapping[name.toLowerCase()]) ) {
						this[i][dom0Property] = '';
					}
				}
				
				if (this[i].removeAttribute) this[i].removeAttribute(name);
			}
		}
		return this;
	};
	
	/**
	@name glow.NodeList#removeClass
	@function
	@description Removes a class from each node.

	@param {string} name The name of the class to remove.

	@returns {glow.NodeList}

	@example
		glow("#footer #login a").removeClass("highlight");
	*/
	NodeListProto.removeClass = function(name) {
		var node;
					
		/*!debug*/
			if (arguments.length !== 1) { glow.debug.warn('[wrong count] glow.NodeList#removeClass() expects 1 argument.'); }
			else if (typeof arguments[0] !== 'string') {glow.debug.warn('[wrong type] glow.NodeList#removeClass() expects argument 1 to be of type string.'); }
		/*gubed!*/
		
		var i = this.length;
		while (i--) {
			node = this[i];
			if (node.className) {
				_removeClass(node, name);
			}
		}
		return this;
	};
	
	function _removeClass(node, name) {
		var oldClasses = node.className.split(' '),
			newClasses = [];
			
		oldClasses = node.className.split(' ');
		newClasses = [];
		
		var i = oldClasses.length;
		while (i--) {
			if (oldClasses[i] !== name) {
				oldClasses[i] && newClasses.unshift(oldClasses[i]); // unshift to maintain original order
			}
		}
		node.className = (newClasses.length)? newClasses.join(' ') : '';
	}
	
	/**
	@name glow.NodeList#removeData
	@function
	@description Removes data previously added by {@link glow.NodeList#data} from each node in this NodeList.
	
	When called with no arguments, will delete glow's entire data store for each node in this NodeList.
	
	Otherwise, when given a name, will delete the associated value from each node in this NodeList.
	
	@param {string} [key] The name of the value in glow's data store.
	@see glow.NodeList#data
	*/
	NodeListProto.removeData = function(key) {
		var elm,
			i = this.length,
			index;
			// uses private scoped variables: dataCache, dataPropName
		
		/*!debug*/
			if (arguments.length > 1) { glow.debug.warn('[wrong count] glow.NodeList#removeData expects 0 or 1 arguments.'); }
			else if (arguments.length === 1 && typeof arguments[0] !== 'string') {glow.debug.warn('[wrong type] glow.NodeList#removeData expects argument 1 to be of type string.'); }
		/*gubed!*/
		
		while (i--) {
			elm = this[i];
			index = elm[dataPropName];
			
			if (index !== undef) {
				switch (arguments.length) {
					case 0:
						dataCache[index] = undef;
						elm[dataPropName] = undef;
						try {
							delete elm[dataPropName]; // IE 6 goes wobbly here
						}
						catch(e) { // remove expando from IE 6
							elm.removeAttribute && elm.removeAttribute(dataPropName);
						}
						break;
					case 1:
						dataCache[index][key] = undef;
						delete dataCache[index][key];
						break;
				}
			}
		}
		
		return this; // chainable
	};
	
	/**
	@name glow.NodeList#toggleClass
	@function
	@description Toggles a class on each node.

	@param {string} name The name of the class to toggle.

	@returns {glow.NodeList}

	@example
		glow(".onOffSwitch").toggleClass("on");
	 */
	NodeListProto.toggleClass = function(name) {
		var node;
		
		/*!debug*/
			if (arguments.length !== 1) { glow.debug.warn('[wrong count] glow.NodeList#toggleClass() expects 1 argument.'); }
			else if (typeof arguments[0] !== 'string') {glow.debug.warn('[wrong type] glow.NodeList#toggleClass() expects argument 1 to be of type string.'); }
		/*gubed!*/
		
		for (var i = 0, leni = this.length; i < leni; i++) {
			node = this[i];
			if (node.className) {
				if ( (' ' + node.className + ' ').indexOf(' ' + name + ' ') > -1 ) {
					_removeClass(node, name);
				}
				else {
					_addClass(node, name);
				}
			}
		}
		
		return this;
	};
	
	/**
	@name glow.NodeList#val
	@function
	@description Gets or sets form values for the first node.
		The returned value depends on the type of element, see below:

		<dl>
		<dt>Radio button or checkbox</dt>
		<dd>If checked, then the contents of the value property, otherwise an empty string.</dd>
		<dt>Select</dt>
		<dd>The contents of value property of the selected option</dd>
		<dt>Select (multiple)</dt>
		<dd>An array of selected option values.</dd>
		<dt>Other form elements</dt>
		<dd>The value of the input.</dd>
		</dl>

		Getting values from a form:

		If the first element in the NodeList is a form, then an
		object is returned containing the form data. Each item
		property of the object is a value as above, apart from when
		multiple elements of the same name exist, in which case the
		it will contain an array of values.

		Setting values for form elements:

		If a value is passed and the first element of the NodeList
		is a form element, then the form element is given that value.
		For select elements, this means that the first option that
		matches the value will be selected. For selects that allow
		multiple selection, the options which have a value that
		exists in the array of values/match the value will be
		selected and others will be deselected.

		Checkboxes and radio buttons will be checked only if the value is the same
		as the one you provide.

		Setting values for forms:

		If the first element in the NodeList is a form and the
		value is an object, then each element of the form has its
		value set to the corresponding property of the object, using
		the method described above.

	@param {string | Object} [value] The value to set the form element/elements to.

	@returns {glow.NodeList | string | Object}

		When used to set a value it returns the NodeList, otherwise
		returns the value as described above.

	@example
		// get a value from an input with the id 'username'
		var username = glow("#username").val();

	@example			
		// get values from a form
		var userDetails = glow("form").val();

	@example
		// set a value
		glow("#username").val("example username");

	@example
		// set values in a form
		glow("form").val({
			username : "another",
			name     : "A N Other"
		});
	*/
	NodeListProto.val = function(){		
		var args = arguments,
			val = args[0],
			i = 0,
			length = this.length;

		if (args.length === 0) {
			return this[0].nodeName == 'FORM' ?
				formValues(this[0]) :
				elementValue(this[0]);
		}
		if (this[0].nodeName == 'FORM') {
			if (! typeof val == 'object') {
				throw 'value for FORM must be object';
			}
			setFormValues(this[0], val);
		} else {
			for (; i < length; i++) {
				setValue(this[i], val);
			}
		}
		return this;		
	};
	
	/*
	 @name elementValue
	 @private
	 @returns the value of the form element
	*/
	function elementValue (el) {
		var elType = el.type,
			elChecked = el.checked,
			elValue = el.value,
			vals = [],
			i = 0;

		if (elType == 'radio') {
			return elChecked ? elValue : '';
		}
			
		else if (elType == 'checkbox') {
			return elChecked ? elValue : '';
		}
			
		else if (elType == 'select-one') {
			return el.selectedIndex > -1 ? el.options[el.selectedIndex].value : '';
		}
			
		else if (elType == 'select-multiple') {
			for (var length = el.options.length; i < length; i++) {
				if (el.options[i].selected) {
					vals[vals.length] = el.options[i].value;
				}
			}
			return vals;
		}
			
		else {
			return elValue;
		}
	}
		
	/*
	@name: setValue
	@description Set the value of a form element.  Returns values that weren't able to set if array of vals passed (for multi select). Otherwise true if val set, false if not
	@returns val or bool
	@private
	*/
	function setValue (el, val) {
		var i = 0,
			length,
			n = 0,
			nlen,
			elOption,
			optionVal;

		if (el.type == 'select-one') {
			for (length = el.options.length; i < length; i++) {
				if (el.options[i].value == val) {
					el.selectedIndex = i;
					return true;
				}
			}
			return false;
		} else if (el.type == 'select-multiple') {
			var isArray = !!val.push;
			for (i = 0, length = el.options.length; i < length; i++) {
				elOption = el.options[i];
				optionVal = elOption.value;
				if (isArray) {
					elOption.selected = false;
					for (nlen = val.length; n < nlen; n++) {
						if (optionVal == val[n]) {
							elOption.selected = true;
							val.splice(n, 1);
							break;
						}
					}
				} else {
					return elOption.selected = val == optionVal;
				}
			}
			return false;
		} else if (el.type == 'radio' || el.type == 'checkbox') {
			el.checked = val == el.value;
			return val == el.value;
		} else {
			el.value = val;
			return true;
		}
	}
		
	/*
	@name setFormValues
	@description Set values of a form to those in passed in object.
	@private
	*/
	function setFormValues (form, vals) {
		var prop, currentField,
			fields = {},
			storeType, i = 0, n, len, foundOne, currentFieldType;

		for (prop in vals) {
			currentField = form[prop];
			if (currentField && currentField[0] && !currentField.options) { // is array of fields
				//normalise values to array of vals
				vals[prop] = vals[prop] && vals[prop].push ? vals[prop] : [vals[prop]];
				//order the fields by types that matter
				fields.radios = [];
				fields.checkboxesSelects = [];
				fields.multiSelects = [];
				fields.other = [];

				for (i = 0; currentField[i]; i++) {
					currentFieldType = currentField[i].type;
					if (currentFieldType == 'radio') {
						storeType = 'radios';
				} else if (currentFieldType == 'select-one' || currentFieldType == 'checkbox') {
						storeType = 'checkboxesSelects';
					} else if (currentFieldType == 'select-multiple') {
						storeType = 'multiSelects';
					} else {
						storeType = 'other';
					}
					//add it to the correct array
					fields[storeType][fields[storeType].length] = currentField[i];
				}

				for (i = 0; fields.multiSelects[i]; i++) {
					vals[prop] = setValue(fields.multiSelects[i], vals[prop]);
				}
				for (i = 0; fields.checkboxesSelects[i]; i++) {
					setValue(fields.checkboxesSelects[i], '');
					for (n = 0, len = vals[prop].length; n < len; n++) {
						if (setValue(fields.checkboxesSelects[i], vals[prop][n])) {
							vals[prop].slice(n, 1);
							break;
						}
					}
				}
				for (i = 0; fields.radios[i]; i++) {
					fields.radios[i].checked = false;
					foundOne = false;
					for (n = 0, len = vals[prop].length; n < len; n++) {
						if (setValue(fields.radios[i], vals[prop][n])) {
							vals[prop].slice(n, 1);
							foundOne = true;
							break;
						}
						if (foundOne) { break; }
					}
				}
				for (i = 0; fields.other[i] && vals[prop][i] !== undefined; i++) {
					setValue(fields.other[i], vals[prop][i]);
				}
			} else if (currentField && currentField.nodeName) { // is single field, easy
				setValue(currentField, vals[prop]);
			}
		}
	}

	/*
	@name formValues
	@description Get an object containing form data.
	@private
	*/
	function formValues (form) {
		var vals = {},
			radios = {},
			formElements = form.elements,
			i = formElements.length,
			name,
			formElement,
			j,
			radio,
			nodeName;
		while (i--) {
			formElement = formElements[i];
			nodeName = formElement.nodeName.toLowerCase();
			name = formElement.name;
				
			// fieldsets & objects come back as form elements, but we don't care about these
			// we don't bother with fields that don't have a name
			// switch to whitelist?
			if (
				nodeName == 'fieldset' ||
				nodeName == 'object' ||
				!name
			) { continue; }
			if (formElement.type == 'checkbox' && ! formElement.checked) {
				if (! name in vals) {
					vals[name] = undefined;
				}
			} else if (formElement.type == 'radio') {
				
				if (radios[name]) {
					radios[name][radios[name].length] = formElement;
				} else {
					radios[name] = [formElement];
				}
			} else {
				var value = elementValue(formElement);
				if (name in vals) {
					if (vals[name].push) {
						vals[name][vals[name].length] = value;
					} else {
						vals[name] = [vals[name], value];
					}
				} else {
					vals[name] = value;
				}
			}
		}
		for (i in radios) {
			var length,
			j = 0;
			for (length = radios[i].length; j < length; j++) {
				radio = radios[i][j];
				name = radio.name;
				if (radio.checked) {
					vals[radio.name] = radio.value;
					break;
				}
			}
			if (! name in vals) { alert('15 if name in vals'); vals[name] = undefined; }
		}
		return vals;
	}
});
/*!
 * Sizzle CSS Selector Engine - v1.0
 *  Copyright 2009, The Dojo Foundation
 *  Released under the MIT, BSD, and GPL Licenses.
 *  More information: http://sizzlejs.com/
 */
(function(){

var chunker = /((?:\((?:\([^()]+\)|[^()]+)+\)|\[(?:\[[^\[\]]*\]|['"][^'"]*['"]|[^\[\]'"]+)+\]|\\.|[^ >+~,(\[\\]+)+|[>+~])(\s*,\s*)?((?:.|\r|\n)*)/g,
	done = 0,
	toString = Object.prototype.toString,
	hasDuplicate = false,
	baseHasDuplicate = true;

// Here we check if the JavaScript engine is using some sort of
// optimization where it does not always call our comparision
// function. If that is the case, discard the hasDuplicate value.
//   Thus far that includes Google Chrome.
[0, 0].sort(function(){
	baseHasDuplicate = false;
	return 0;
});

var Sizzle = function(selector, context, results, seed) {
	results = results || [];
	context = context || document;

	var origContext = context;

	if ( context.nodeType !== 1 && context.nodeType !== 9 ) {
		return [];
	}
	
	if ( !selector || typeof selector !== "string" ) {
		return results;
	}

	var parts = [], m, set, checkSet, extra, prune = true, contextXML = Sizzle.isXML(context),
		soFar = selector, ret, cur, pop, i;
	
	// Reset the position of the chunker regexp (start from head)
	do {
		chunker.exec("");
		m = chunker.exec(soFar);

		if ( m ) {
			soFar = m[3];
		
			parts.push( m[1] );
		
			if ( m[2] ) {
				extra = m[3];
				break;
			}
		}
	} while ( m );

	if ( parts.length > 1 && origPOS.exec( selector ) ) {
		if ( parts.length === 2 && Expr.relative[ parts[0] ] ) {
			set = posProcess( parts[0] + parts[1], context );
		} else {
			set = Expr.relative[ parts[0] ] ?
				[ context ] :
				Sizzle( parts.shift(), context );

			while ( parts.length ) {
				selector = parts.shift();

				if ( Expr.relative[ selector ] ) {
					selector += parts.shift();
				}
				
				set = posProcess( selector, set );
			}
		}
	} else {
		// Take a shortcut and set the context if the root selector is an ID
		// (but not if it'll be faster if the inner selector is an ID)
		if ( !seed && parts.length > 1 && context.nodeType === 9 && !contextXML &&
				Expr.match.ID.test(parts[0]) && !Expr.match.ID.test(parts[parts.length - 1]) ) {
			ret = Sizzle.find( parts.shift(), context, contextXML );
			context = ret.expr ? Sizzle.filter( ret.expr, ret.set )[0] : ret.set[0];
		}

		if ( context ) {
			ret = seed ?
				{ expr: parts.pop(), set: makeArray(seed) } :
				Sizzle.find( parts.pop(), parts.length === 1 && (parts[0] === "~" || parts[0] === "+") && context.parentNode ? context.parentNode : context, contextXML );
			set = ret.expr ? Sizzle.filter( ret.expr, ret.set ) : ret.set;

			if ( parts.length > 0 ) {
				checkSet = makeArray(set);
			} else {
				prune = false;
			}

			while ( parts.length ) {
				cur = parts.pop();
				pop = cur;

				if ( !Expr.relative[ cur ] ) {
					cur = "";
				} else {
					pop = parts.pop();
				}

				if ( pop == null ) {
					pop = context;
				}

				Expr.relative[ cur ]( checkSet, pop, contextXML );
			}
		} else {
			checkSet = parts = [];
		}
	}

	if ( !checkSet ) {
		checkSet = set;
	}

	if ( !checkSet ) {
		Sizzle.error( cur || selector );
	}

	if ( toString.call(checkSet) === "[object Array]" ) {
		if ( !prune ) {
			results.push.apply( results, checkSet );
		} else if ( context && context.nodeType === 1 ) {
			for ( i = 0; checkSet[i] != null; i++ ) {
				if ( checkSet[i] && (checkSet[i] === true || checkSet[i].nodeType === 1 && Sizzle.contains(context, checkSet[i])) ) {
					results.push( set[i] );
				}
			}
		} else {
			for ( i = 0; checkSet[i] != null; i++ ) {
				if ( checkSet[i] && checkSet[i].nodeType === 1 ) {
					results.push( set[i] );
				}
			}
		}
	} else {
		makeArray( checkSet, results );
	}

	if ( extra ) {
		Sizzle( extra, origContext, results, seed );
		Sizzle.uniqueSort( results );
	}

	return results;
};

Sizzle.uniqueSort = function(results){
	if ( sortOrder ) {
		hasDuplicate = baseHasDuplicate;
		results.sort(sortOrder);

		if ( hasDuplicate ) {
			for ( var i = 1; i < results.length; i++ ) {
				if ( results[i] === results[i-1] ) {
					results.splice(i--, 1);
				}
			}
		}
	}

	return results;
};

Sizzle.matches = function(expr, set){
	return Sizzle(expr, null, null, set);
};

Sizzle.find = function(expr, context, isXML){
	var set;

	if ( !expr ) {
		return [];
	}

	for ( var i = 0, l = Expr.order.length; i < l; i++ ) {
		var type = Expr.order[i], match;
		
		if ( (match = Expr.leftMatch[ type ].exec( expr )) ) {
			var left = match[1];
			match.splice(1,1);

			if ( left.substr( left.length - 1 ) !== "\\" ) {
				match[1] = (match[1] || "").replace(/\\/g, "");
				set = Expr.find[ type ]( match, context, isXML );
				if ( set != null ) {
					expr = expr.replace( Expr.match[ type ], "" );
					break;
				}
			}
		}
	}

	if ( !set ) {
		set = context.getElementsByTagName("*");
	}

	return {set: set, expr: expr};
};

Sizzle.filter = function(expr, set, inplace, not){
	var old = expr, result = [], curLoop = set, match, anyFound,
		isXMLFilter = set && set[0] && Sizzle.isXML(set[0]);

	while ( expr && set.length ) {
		for ( var type in Expr.filter ) {
			if ( (match = Expr.leftMatch[ type ].exec( expr )) != null && match[2] ) {
				var filter = Expr.filter[ type ], found, item, left = match[1];
				anyFound = false;

				match.splice(1,1);

				if ( left.substr( left.length - 1 ) === "\\" ) {
					continue;
				}

				if ( curLoop === result ) {
					result = [];
				}

				if ( Expr.preFilter[ type ] ) {
					match = Expr.preFilter[ type ]( match, curLoop, inplace, result, not, isXMLFilter );

					if ( !match ) {
						anyFound = found = true;
					} else if ( match === true ) {
						continue;
					}
				}

				if ( match ) {
					for ( var i = 0; (item = curLoop[i]) != null; i++ ) {
						if ( item ) {
							found = filter( item, match, i, curLoop );
							var pass = not ^ !!found;

							if ( inplace && found != null ) {
								if ( pass ) {
									anyFound = true;
								} else {
									curLoop[i] = false;
								}
							} else if ( pass ) {
								result.push( item );
								anyFound = true;
							}
						}
					}
				}

				if ( found !== undefined ) {
					if ( !inplace ) {
						curLoop = result;
					}

					expr = expr.replace( Expr.match[ type ], "" );

					if ( !anyFound ) {
						return [];
					}

					break;
				}
			}
		}

		// Improper expression
		if ( expr === old ) {
			if ( anyFound == null ) {
				Sizzle.error( expr );
			} else {
				break;
			}
		}

		old = expr;
	}

	return curLoop;
};

Sizzle.error = function( msg ) {
	throw "Syntax error, unrecognized expression: " + msg;
};

var Expr = Sizzle.selectors = {
	order: [ "ID", "NAME", "TAG" ],
	match: {
		ID: /#((?:[\w\u00c0-\uFFFF\-]|\\.)+)/,
		CLASS: /\.((?:[\w\u00c0-\uFFFF\-]|\\.)+)/,
		NAME: /\[name=['"]*((?:[\w\u00c0-\uFFFF\-]|\\.)+)['"]*\]/,
		ATTR: /\[\s*((?:[\w\u00c0-\uFFFF\-]|\\.)+)\s*(?:(\S?=)\s*(['"]*)(.*?)\3|)\s*\]/,
		TAG: /^((?:[\w\u00c0-\uFFFF\*\-]|\\.)+)/,
		CHILD: /:(only|nth|last|first)-child(?:\((even|odd|[\dn+\-]*)\))?/,
		POS: /:(nth|eq|gt|lt|first|last|even|odd)(?:\((\d*)\))?(?=[^\-]|$)/,
		PSEUDO: /:((?:[\w\u00c0-\uFFFF\-]|\\.)+)(?:\((['"]?)((?:\([^\)]+\)|[^\(\)]*)+)\2\))?/
	},
	leftMatch: {},
	attrMap: {
		"class": "className",
		"for": "htmlFor"
	},
	attrHandle: {
		href: function(elem){
			return elem.getAttribute("href");
		}
	},
	relative: {
		"+": function(checkSet, part){
			var isPartStr = typeof part === "string",
				isTag = isPartStr && !/\W/.test(part),
				isPartStrNotTag = isPartStr && !isTag;

			if ( isTag ) {
				part = part.toLowerCase();
			}

			for ( var i = 0, l = checkSet.length, elem; i < l; i++ ) {
				if ( (elem = checkSet[i]) ) {
					while ( (elem = elem.previousSibling) && elem.nodeType !== 1 ) {}

					checkSet[i] = isPartStrNotTag || elem && elem.nodeName.toLowerCase() === part ?
						elem || false :
						elem === part;
				}
			}

			if ( isPartStrNotTag ) {
				Sizzle.filter( part, checkSet, true );
			}
		},
		">": function(checkSet, part){
			var isPartStr = typeof part === "string",
				elem, i = 0, l = checkSet.length;

			if ( isPartStr && !/\W/.test(part) ) {
				part = part.toLowerCase();

				for ( ; i < l; i++ ) {
					elem = checkSet[i];
					if ( elem ) {
						var parent = elem.parentNode;
						checkSet[i] = parent.nodeName.toLowerCase() === part ? parent : false;
					}
				}
			} else {
				for ( ; i < l; i++ ) {
					elem = checkSet[i];
					if ( elem ) {
						checkSet[i] = isPartStr ?
							elem.parentNode :
							elem.parentNode === part;
					}
				}

				if ( isPartStr ) {
					Sizzle.filter( part, checkSet, true );
				}
			}
		},
		"": function(checkSet, part, isXML){
			var doneName = done++, checkFn = dirCheck, nodeCheck;

			if ( typeof part === "string" && !/\W/.test(part) ) {
				part = part.toLowerCase();
				nodeCheck = part;
				checkFn = dirNodeCheck;
			}

			checkFn("parentNode", part, doneName, checkSet, nodeCheck, isXML);
		},
		"~": function(checkSet, part, isXML){
			var doneName = done++, checkFn = dirCheck, nodeCheck;

			if ( typeof part === "string" && !/\W/.test(part) ) {
				part = part.toLowerCase();
				nodeCheck = part;
				checkFn = dirNodeCheck;
			}

			checkFn("previousSibling", part, doneName, checkSet, nodeCheck, isXML);
		}
	},
	find: {
		ID: function(match, context, isXML){
			if ( typeof context.getElementById !== "undefined" && !isXML ) {
				var m = context.getElementById(match[1]);
				return m ? [m] : [];
			}
		},
		NAME: function(match, context){
			if ( typeof context.getElementsByName !== "undefined" ) {
				var ret = [], results = context.getElementsByName(match[1]);

				for ( var i = 0, l = results.length; i < l; i++ ) {
					if ( results[i].getAttribute("name") === match[1] ) {
						ret.push( results[i] );
					}
				}

				return ret.length === 0 ? null : ret;
			}
		},
		TAG: function(match, context){
			return context.getElementsByTagName(match[1]);
		}
	},
	preFilter: {
		CLASS: function(match, curLoop, inplace, result, not, isXML){
			match = " " + match[1].replace(/\\/g, "") + " ";

			if ( isXML ) {
				return match;
			}

			for ( var i = 0, elem; (elem = curLoop[i]) != null; i++ ) {
				if ( elem ) {
					if ( not ^ (elem.className && (" " + elem.className + " ").replace(/[\t\n]/g, " ").indexOf(match) >= 0) ) {
						if ( !inplace ) {
							result.push( elem );
						}
					} else if ( inplace ) {
						curLoop[i] = false;
					}
				}
			}

			return false;
		},
		ID: function(match){
			return match[1].replace(/\\/g, "");
		},
		TAG: function(match, curLoop){
			return match[1].toLowerCase();
		},
		CHILD: function(match){
			if ( match[1] === "nth" ) {
				// parse equations like 'even', 'odd', '5', '2n', '3n+2', '4n-1', '-n+6'
				var test = /(-?)(\d*)n((?:\+|-)?\d*)/.exec(
					match[2] === "even" && "2n" || match[2] === "odd" && "2n+1" ||
					!/\D/.test( match[2] ) && "0n+" + match[2] || match[2]);

				// calculate the numbers (first)n+(last) including if they are negative
				match[2] = (test[1] + (test[2] || 1)) - 0;
				match[3] = test[3] - 0;
			}

			// TODO: Move to normal caching system
			match[0] = done++;

			return match;
		},
		ATTR: function(match, curLoop, inplace, result, not, isXML){
			var name = match[1].replace(/\\/g, "");
			
			if ( !isXML && Expr.attrMap[name] ) {
				match[1] = Expr.attrMap[name];
			}

			if ( match[2] === "~=" ) {
				match[4] = " " + match[4] + " ";
			}

			return match;
		},
		PSEUDO: function(match, curLoop, inplace, result, not){
			if ( match[1] === "not" ) {
				// If we're dealing with a complex expression, or a simple one
				if ( ( chunker.exec(match[3]) || "" ).length > 1 || /^\w/.test(match[3]) ) {
					match[3] = Sizzle(match[3], null, null, curLoop);
				} else {
					var ret = Sizzle.filter(match[3], curLoop, inplace, true ^ not);
					if ( !inplace ) {
						result.push.apply( result, ret );
					}
					return false;
				}
			} else if ( Expr.match.POS.test( match[0] ) || Expr.match.CHILD.test( match[0] ) ) {
				return true;
			}
			
			return match;
		},
		POS: function(match){
			match.unshift( true );
			return match;
		}
	},
	filters: {
		enabled: function(elem){
			return elem.disabled === false && elem.type !== "hidden";
		},
		disabled: function(elem){
			return elem.disabled === true;
		},
		checked: function(elem){
			return elem.checked === true;
		},
		selected: function(elem){
			// Accessing this property makes selected-by-default
			// options in Safari work properly
			elem.parentNode.selectedIndex;
			return elem.selected === true;
		},
		parent: function(elem){
			return !!elem.firstChild;
		},
		empty: function(elem){
			return !elem.firstChild;
		},
		has: function(elem, i, match){
			return !!Sizzle( match[3], elem ).length;
		},
		header: function(elem){
			return (/h\d/i).test( elem.nodeName );
		},
		text: function(elem){
			return "text" === elem.type;
		},
		radio: function(elem){
			return "radio" === elem.type;
		},
		checkbox: function(elem){
			return "checkbox" === elem.type;
		},
		file: function(elem){
			return "file" === elem.type;
		},
		password: function(elem){
			return "password" === elem.type;
		},
		submit: function(elem){
			return "submit" === elem.type;
		},
		image: function(elem){
			return "image" === elem.type;
		},
		reset: function(elem){
			return "reset" === elem.type;
		},
		button: function(elem){
			return "button" === elem.type || elem.nodeName.toLowerCase() === "button";
		},
		input: function(elem){
			return (/input|select|textarea|button/i).test(elem.nodeName);
		}
	},
	setFilters: {
		first: function(elem, i){
			return i === 0;
		},
		last: function(elem, i, match, array){
			return i === array.length - 1;
		},
		even: function(elem, i){
			return i % 2 === 0;
		},
		odd: function(elem, i){
			return i % 2 === 1;
		},
		lt: function(elem, i, match){
			return i < match[3] - 0;
		},
		gt: function(elem, i, match){
			return i > match[3] - 0;
		},
		nth: function(elem, i, match){
			return match[3] - 0 === i;
		},
		eq: function(elem, i, match){
			return match[3] - 0 === i;
		}
	},
	filter: {
		PSEUDO: function(elem, match, i, array){
			var name = match[1], filter = Expr.filters[ name ];

			if ( filter ) {
				return filter( elem, i, match, array );
			} else if ( name === "contains" ) {
				return (elem.textContent || elem.innerText || Sizzle.getText([ elem ]) || "").indexOf(match[3]) >= 0;
			} else if ( name === "not" ) {
				var not = match[3];

				for ( var j = 0, l = not.length; j < l; j++ ) {
					if ( not[j] === elem ) {
						return false;
					}
				}

				return true;
			} else {
				Sizzle.error( "Syntax error, unrecognized expression: " + name );
			}
		},
		CHILD: function(elem, match){
			var type = match[1], node = elem;
			switch (type) {
				case 'only':
				case 'first':
					while ( (node = node.previousSibling) )	 {
						if ( node.nodeType === 1 ) { 
							return false; 
						}
					}
					if ( type === "first" ) { 
						return true; 
					}
					node = elem;
				case 'last':
					while ( (node = node.nextSibling) )	 {
						if ( node.nodeType === 1 ) { 
							return false; 
						}
					}
					return true;
				case 'nth':
					var first = match[2], last = match[3];

					if ( first === 1 && last === 0 ) {
						return true;
					}
					
					var doneName = match[0],
						parent = elem.parentNode;
	
					if ( parent && (parent.sizcache !== doneName || !elem.nodeIndex) ) {
						var count = 0;
						for ( node = parent.firstChild; node; node = node.nextSibling ) {
							if ( node.nodeType === 1 ) {
								node.nodeIndex = ++count;
							}
						} 
						parent.sizcache = doneName;
					}
					
					var diff = elem.nodeIndex - last;
					if ( first === 0 ) {
						return diff === 0;
					} else {
						return ( diff % first === 0 && diff / first >= 0 );
					}
			}
		},
		ID: function(elem, match){
			return elem.nodeType === 1 && elem.getAttribute("id") === match;
		},
		TAG: function(elem, match){
			return (match === "*" && elem.nodeType === 1) || elem.nodeName.toLowerCase() === match;
		},
		CLASS: function(elem, match){
			return (" " + (elem.className || elem.getAttribute("class")) + " ")
				.indexOf( match ) > -1;
		},
		ATTR: function(elem, match){
			var name = match[1],
				result = Expr.attrHandle[ name ] ?
					Expr.attrHandle[ name ]( elem ) :
					elem[ name ] != null ?
						elem[ name ] :
						elem.getAttribute( name ),
				value = result + "",
				type = match[2],
				check = match[4];

			return result == null ?
				type === "!=" :
				type === "=" ?
				value === check :
				type === "*=" ?
				value.indexOf(check) >= 0 :
				type === "~=" ?
				(" " + value + " ").indexOf(check) >= 0 :
				!check ?
				value && result !== false :
				type === "!=" ?
				value !== check :
				type === "^=" ?
				value.indexOf(check) === 0 :
				type === "$=" ?
				value.substr(value.length - check.length) === check :
				type === "|=" ?
				value === check || value.substr(0, check.length + 1) === check + "-" :
				false;
		},
		POS: function(elem, match, i, array){
			var name = match[2], filter = Expr.setFilters[ name ];

			if ( filter ) {
				return filter( elem, i, match, array );
			}
		}
	}
};

var origPOS = Expr.match.POS,
	fescape = function(all, num){
		return "\\" + (num - 0 + 1);
	};

for ( var type in Expr.match ) {
	Expr.match[ type ] = new RegExp( Expr.match[ type ].source + (/(?![^\[]*\])(?![^\(]*\))/.source) );
	Expr.leftMatch[ type ] = new RegExp( /(^(?:.|\r|\n)*?)/.source + Expr.match[ type ].source.replace(/\\(\d+)/g, fescape) );
}

var makeArray = function(array, results) {
	array = Array.prototype.slice.call( array, 0 );

	if ( results ) {
		results.push.apply( results, array );
		return results;
	}
	
	return array;
};

// Perform a simple check to determine if the browser is capable of
// converting a NodeList to an array using builtin methods.
// Also verifies that the returned array holds DOM nodes
// (which is not the case in the Blackberry browser)
try {
	Array.prototype.slice.call( document.documentElement.childNodes, 0 )[0].nodeType;

// Provide a fallback method if it does not work
} catch(e){
	makeArray = function(array, results) {
		var ret = results || [], i = 0;

		if ( toString.call(array) === "[object Array]" ) {
			Array.prototype.push.apply( ret, array );
		} else {
			if ( typeof array.length === "number" ) {
				for ( var l = array.length; i < l; i++ ) {
					ret.push( array[i] );
				}
			} else {
				for ( ; array[i]; i++ ) {
					ret.push( array[i] );
				}
			}
		}

		return ret;
	};
}

var sortOrder;

if ( document.documentElement.compareDocumentPosition ) {
	sortOrder = function( a, b ) {
		if ( !a.compareDocumentPosition || !b.compareDocumentPosition ) {
			if ( a == b ) {
				hasDuplicate = true;
			}
			return a.compareDocumentPosition ? -1 : 1;
		}

		var ret = a.compareDocumentPosition(b) & 4 ? -1 : a === b ? 0 : 1;
		if ( ret === 0 ) {
			hasDuplicate = true;
		}
		return ret;
	};
} else if ( "sourceIndex" in document.documentElement ) {
	sortOrder = function( a, b ) {
		if ( !a.sourceIndex || !b.sourceIndex ) {
			if ( a == b ) {
				hasDuplicate = true;
			}
			return a.sourceIndex ? -1 : 1;
		}

		var ret = a.sourceIndex - b.sourceIndex;
		if ( ret === 0 ) {
			hasDuplicate = true;
		}
		return ret;
	};
} else if ( document.createRange ) {
	sortOrder = function( a, b ) {
		if ( !a.ownerDocument || !b.ownerDocument ) {
			if ( a == b ) {
				hasDuplicate = true;
			}
			return a.ownerDocument ? -1 : 1;
		}

		var aRange = a.ownerDocument.createRange(), bRange = b.ownerDocument.createRange();
		aRange.setStart(a, 0);
		aRange.setEnd(a, 0);
		bRange.setStart(b, 0);
		bRange.setEnd(b, 0);
		var ret = aRange.compareBoundaryPoints(Range.START_TO_END, bRange);
		if ( ret === 0 ) {
			hasDuplicate = true;
		}
		return ret;
	};
}

// Utility function for retreiving the text value of an array of DOM nodes
Sizzle.getText = function( elems ) {
	var ret = "", elem;

	for ( var i = 0; elems[i]; i++ ) {
		elem = elems[i];

		// Get the text from text nodes and CDATA nodes
		if ( elem.nodeType === 3 || elem.nodeType === 4 ) {
			ret += elem.nodeValue;

		// Traverse everything else, except comment nodes
		} else if ( elem.nodeType !== 8 ) {
			ret += Sizzle.getText( elem.childNodes );
		}
	}

	return ret;
};

// Check to see if the browser returns elements by name when
// querying by getElementById (and provide a workaround)
(function(){
	// We're going to inject a fake input element with a specified name
	var form = document.createElement("div"),
		id = "script" + (new Date()).getTime();
	form.innerHTML = "<a name='" + id + "'/>";

	// Inject it into the root element, check its status, and remove it quickly
	var root = document.documentElement;
	root.insertBefore( form, root.firstChild );

	// The workaround has to do additional checks after a getElementById
	// Which slows things down for other browsers (hence the branching)
	if ( document.getElementById( id ) ) {
		Expr.find.ID = function(match, context, isXML){
			if ( typeof context.getElementById !== "undefined" && !isXML ) {
				var m = context.getElementById(match[1]);
				return m ? m.id === match[1] || typeof m.getAttributeNode !== "undefined" && m.getAttributeNode("id").nodeValue === match[1] ? [m] : undefined : [];
			}
		};

		Expr.filter.ID = function(elem, match){
			var node = typeof elem.getAttributeNode !== "undefined" && elem.getAttributeNode("id");
			return elem.nodeType === 1 && node && node.nodeValue === match;
		};
	}

	root.removeChild( form );
	root = form = null; // release memory in IE
})();

(function(){
	// Check to see if the browser returns only elements
	// when doing getElementsByTagName("*")

	// Create a fake element
	var div = document.createElement("div");
	div.appendChild( document.createComment("") );

	// Make sure no comments are found
	if ( div.getElementsByTagName("*").length > 0 ) {
		Expr.find.TAG = function(match, context){
			var results = context.getElementsByTagName(match[1]);

			// Filter out possible comments
			if ( match[1] === "*" ) {
				var tmp = [];

				for ( var i = 0; results[i]; i++ ) {
					if ( results[i].nodeType === 1 ) {
						tmp.push( results[i] );
					}
				}

				results = tmp;
			}

			return results;
		};
	}

	// Check to see if an attribute returns normalized href attributes
	div.innerHTML = "<a href='#'></a>";
	if ( div.firstChild && typeof div.firstChild.getAttribute !== "undefined" &&
			div.firstChild.getAttribute("href") !== "#" ) {
		Expr.attrHandle.href = function(elem){
			return elem.getAttribute("href", 2);
		};
	}

	div = null; // release memory in IE
})();

if ( document.querySelectorAll ) {
	(function(){
		var oldSizzle = Sizzle, div = document.createElement("div");
		div.innerHTML = "<p class='TEST'></p>";

		// Safari can't handle uppercase or unicode characters when
		// in quirks mode.
		if ( div.querySelectorAll && div.querySelectorAll(".TEST").length === 0 ) {
			return;
		}
	
		Sizzle = function(query, context, extra, seed){
			context = context || document;

			// Only use querySelectorAll on non-XML documents
			// (ID selectors don't work in non-HTML documents)
			if ( !seed && context.nodeType === 9 && !Sizzle.isXML(context) ) {
				try {
					return makeArray( context.querySelectorAll(query), extra );
				} catch(e){}
			}
		
			return oldSizzle(query, context, extra, seed);
		};

		for ( var prop in oldSizzle ) {
			Sizzle[ prop ] = oldSizzle[ prop ];
		}

		div = null; // release memory in IE
	})();
}

(function(){
	var div = document.createElement("div");

	div.innerHTML = "<div class='test e'></div><div class='test'></div>";

	// Opera can't find a second classname (in 9.6)
	// Also, make sure that getElementsByClassName actually exists
	if ( !div.getElementsByClassName || div.getElementsByClassName("e").length === 0 ) {
		return;
	}

	// Safari caches class attributes, doesn't catch changes (in 3.2)
	div.lastChild.className = "e";

	if ( div.getElementsByClassName("e").length === 1 ) {
		return;
	}
	
	Expr.order.splice(1, 0, "CLASS");
	Expr.find.CLASS = function(match, context, isXML) {
		if ( typeof context.getElementsByClassName !== "undefined" && !isXML ) {
			return context.getElementsByClassName(match[1]);
		}
	};

	div = null; // release memory in IE
})();

function dirNodeCheck( dir, cur, doneName, checkSet, nodeCheck, isXML ) {
	for ( var i = 0, l = checkSet.length; i < l; i++ ) {
		var elem = checkSet[i];
		if ( elem ) {
			elem = elem[dir];
			var match = false;

			while ( elem ) {
				if ( elem.sizcache === doneName ) {
					match = checkSet[elem.sizset];
					break;
				}

				if ( elem.nodeType === 1 && !isXML ){
					elem.sizcache = doneName;
					elem.sizset = i;
				}

				if ( elem.nodeName.toLowerCase() === cur ) {
					match = elem;
					break;
				}

				elem = elem[dir];
			}

			checkSet[i] = match;
		}
	}
}

function dirCheck( dir, cur, doneName, checkSet, nodeCheck, isXML ) {
	for ( var i = 0, l = checkSet.length; i < l; i++ ) {
		var elem = checkSet[i];
		if ( elem ) {
			elem = elem[dir];
			var match = false;

			while ( elem ) {
				if ( elem.sizcache === doneName ) {
					match = checkSet[elem.sizset];
					break;
				}

				if ( elem.nodeType === 1 ) {
					if ( !isXML ) {
						elem.sizcache = doneName;
						elem.sizset = i;
					}
					if ( typeof cur !== "string" ) {
						if ( elem === cur ) {
							match = true;
							break;
						}

					} else if ( Sizzle.filter( cur, [elem] ).length > 0 ) {
						match = elem;
						break;
					}
				}

				elem = elem[dir];
			}

			checkSet[i] = match;
		}
	}
}

Sizzle.contains = document.compareDocumentPosition ? function(a, b){
	return !!(a.compareDocumentPosition(b) & 16);
} : function(a, b){
	return a !== b && (a.contains ? a.contains(b) : true);
};

Sizzle.isXML = function(elem){
	// documentElement is verified for cases where it doesn't yet exist
	// (such as loading iframes in IE - #4833) 
	var documentElement = (elem ? elem.ownerDocument || elem : 0).documentElement;
	return documentElement ? documentElement.nodeName !== "HTML" : false;
};

var posProcess = function(selector, context){
	var tmpSet = [], later = "", match,
		root = context.nodeType ? [context] : context;

	// Position selectors must be done after the filter
	// And so must :not(positional) so we move all PSEUDOs to the end
	while ( (match = Expr.match.PSEUDO.exec( selector )) ) {
		later += match[0];
		selector = selector.replace( Expr.match.PSEUDO, "" );
	}

	selector = Expr.relative[selector] ? selector + "*" : selector;

	for ( var i = 0, l = root.length; i < l; i++ ) {
		Sizzle( selector, root[i], tmpSet );
	}

	return Sizzle.filter( later, tmpSet );
};

// Add Sizzle to Glow
// This file is injected into sizzle.js by the ant "deps" target
Glow.provide(function(glow) {
	glow._sizzle = Sizzle;
});

return;


window.Sizzle = Sizzle;

})();
Glow.provide(function(glow) {
	var NodeListProto = glow.NodeList.prototype
	/*
		PrivateVar: ucheck
			Used by unique(), increased by 1 on each use
		*/
		,	ucheck = 1
	/*
		PrivateVar: ucheckPropName
			This is the property name used by unique checks
		*/

	, ucheckPropName = "_unique" + glow.UID;
	/*
		PrivateMethod: unique
			Get an array of nodes without duplicate nodes from an array of nodes.

		Arguments:
			aNodes - (Array|<NodeList>)

		Returns:
			An array of nodes without duplicates.
		*/
		//worth checking if it's an XML document?
		if (glow.env.ie) {
			var unique = function(aNodes) {
				if (aNodes.length == 1) { return aNodes; }

				//remove duplicates
				var r = [],
					ri = 0,
					i = 0;

				for (; aNodes[i]; i++) {
					if (aNodes[i].getAttribute(ucheckPropName) != ucheck && aNodes[i].nodeType == 1) {
						r[ri++] = aNodes[i];
					}
					aNodes[i].setAttribute(ucheckPropName, ucheck);
				}
				for (i=0; aNodes[i]; i++) {
					aNodes[i].removeAttribute(ucheckPropName);
				}
				ucheck++;
				return r;
			}
		} else {
			var unique = function(aNodes) {
				if (aNodes.length == 1) { return aNodes; }

				//remove duplicates
				var r = [],
					ri = 0,
					i = 0;

				for (; aNodes[i]; i++) {
					if (aNodes[i][ucheckPropName] != ucheck && aNodes[i].nodeType == 1) {
						r[ri++] = aNodes[i];
					}
					aNodes[i][ucheckPropName] = ucheck;
				}
				ucheck++;
				return r;
			}
		};
	/**
	@name glow.NodeList#parent
	@function
	@description Gets the unique parent nodes of each node as a new NodeList.
	@param {string | HTMLElement | NodeList} [search] Search value
		If provided, will seek the next parent element until a match is found
	@returns {glow.NodeList}

		Returns a new NodeList containing the parent nodes, with
		duplicates removed

	@example
		// elements which contain links
		var parents = glow.dom.get("a").parent();
	*/
	NodeListProto.parent = function(search) {
		var ret = [],
			ri = 0,
			i = this.length,
			node;
			
		while (i--) {				
			node = this[i];
			if (node.nodeType == 1) {
				if(search){						
					while(node = node.parentNode){											
						if (glow._sizzle.filter(search, [node]).length) {
							ret[ri++] = node;							
							break;
						}							
					}
				}
			
				else if(node = node.parentNode){
						ret[ri++] = node;						
				}

			}

		}
				
		return new glow.NodeList(unique(ret));			
	};
	
	/* Private method for prev() and next() */
	function getNextOrPrev(nodelist, dir, search) {
		var ret = [],
			ri = 0,
			node,
			i = 0,
			length = nodelist.length;

		while (i < length) {			
			node = nodelist[i];			
			if(search){
				while (node = node[dir + 'Sibling']) {					
					if (node.nodeType == 1 && node.nodeName != '!') {						
						if (glow._sizzle.filter(search, [node]).length) {
							ret[ri++] = node;							
							break;
						}					
					}					
				}
			}
			else{
				while (node = node[dir + 'Sibling']) {					
					if (node.nodeType == 1 && node.nodeName != '!') {
							ret[ri++] = node;							
							 break;					
					}					
				}	
			}
		i++;
		}
		return new glow.NodeList(ret);
	}
	
	/**
	@name glow.NodeList#prev
	@function
	@description Gets the previous sibling element for each node in the ElementList.
		If a filter is provided, the previous item that matches the filter is returned, or
		none if no match is found.
	@param {string | HTMLElement | NodeList} [search] Search value
		If provided, will seek the previous sibling element until a match is found
	@returns {glow.ElementList}
		A new ElementList containing the previous sibling elements that match the (optional)
		filter.
	@example
		// gets the element before #myLink (if there is one)
		var next = glow.get("#myLink").prev();
	@example
		// get the previous sibling link element before #skipLink
		glow.get('#skipLink').prev('a')
	*/
	NodeListProto.prev = function(search) {
		return getNextOrPrev(this, 'previous', search);
	};
	
	/**
	@name glow.NodeList#next
	@function
	@description Gets the next sibling element for each node in the ElementList.
		If a filter is provided, the next item that matches the filter is returned, or
		none if no match is found.
	@param {string | HTMLElement | NodeList} [search] Search value
		If provided, will seek the next sibling element until a match is found
	@returns {glow.ElementList}
		A new ElementList containing the next sibling elements that match the (optional)
		filter.
	@example
		// gets the element following #myLink (if there is one)
		var next = glow.get("#myLink").next();
	@example
		// get the next sibling link element after #skipLink
		glow.get('#skipLink').next('a')
	*/
	NodeListProto.next = function(search) {
		return getNextOrPrev(this, 'next', search);	
	};
	
	
	/**
	@name glow.NodeList#get
	@function
	@description Gets decendents of nodes that match a CSS selector.

	@param {String} selector CSS selector

	@returns {glow.NodeList}
		Returns a new NodeList containing matched elements

	@example
		// create a new NodeList
		var myNodeList = glow.dom.create("<div><a href='s.html'>Link</a></div>");

		// get 'a' tags that are decendants of the NodeList nodes
		myNewNodeList = myNodeList.get("a");
	*/
	NodeListProto.get = function(selector) {
		var ret = [],
			i = this.length;

		while (i--) {			
			glow._sizzle(selector, this[i], ret);
			
		}
		// need to remove uniqueSorts because they're slow. Replace with own method for unique.
		return new glow.NodeList(unique(ret));
	};
	
	
	
	/**
	@name glow.NodeList#ancestors
	@function
	@description Gets the unique ancestor nodes of each node as a new NodeList.
	@param {Function|string} [filter] Filter test
		If a string is provided, it is used in a call to {@link glow.ElementList#is ElementList#is}.
		If a function is provided it will be passed 2 arguments, the index of the current item,
		and the ElementList being itterated over.
		Inside the function 'this' refers to the HTMLElement.
		Return true to keep the node, or false to remove it.
	@returns {glow.dom.NodeList}
		Returns NodeList

		@example
		// get ancestor elements for anchor elements 
		var ancestors = glow.dom.get("a").ancestors();
	*/
	NodeListProto.ancestors = function(filter) {
		var ret = [],
			ri = 0,
			i = 0,
			length = this.length,
			node;
					
		while (i < length) {
			node = this[i].parentNode;
					
			while (node && node.nodeType == 1) {							
				ret[ri++] = node;
				node = node.parentNode;
			}								
		i++;
		}
		if(filter){
            ret = new glow.NodeList(ret);
			ret = ret.filter(filter);
		}
		return new glow.NodeList(unique(ret));
	};
	
	/*
		Private method to get the child elements for an html node (used by children())
	*/
		function getChildElms(node) {
			var r = [],
				childNodes = node.childNodes,
				i = 0,
				ri = 0;
			
			for (; childNodes[i]; i++) {
				if (childNodes[i].nodeType == 1 && childNodes[i].nodeName != '!') {
					r[ri++] = childNodes[i];
				}
			}
			return r;
		}
	
	/**
	@name glow.NodeList#children
	@function
	@description Gets the child elements of each node as a new NodeList.

	@returns {glow.dom.NodeList}

		Returns a new NodeList containing all the child nodes
				
	@example
		// get all list items
		var items = glow.dom.get("ul, ol").children();
	*/
	NodeListProto.children = function() {
		var ret = [],
			i = this.length;
				
		while(i--) {
			ret = ret.concat( getChildElms(this[i]) );
		}
		return new glow.NodeList(ret);	
	};
	
	/**
	@name glow.NodeList#contains
	@function
	@description Find if this NodeList contains the given element
		
	@param {string | HTMLELement | NodeList} Single element to check for

	@returns {boolean}
		myElementList.contains(elm)
		// Returns true if an element in myElementList contains elm, or IS elm.
	*/
	NodeListProto.contains = function(elm) {
		var i = 0,
			node = new glow.NodeList(elm)[0],
			length = this.length,
			newNodes,
			toTest;

		// missing some nodes? Return false
		if ( !node || !this.length ) {
			return false;
		}
	
		if (this[0].compareDocumentPosition) { //w3 method
			while (i < length) {
				//break out if the two are teh same
				if(this[i] == node){
					break;
				}
				//check against bitwise to see if node is contained in this
				else if (!(this[i].compareDocumentPosition(node) & 16)) {								
					return false;
				}
			i++;
			}
		}
		else if(node.contains){					
			for (; i < length; i++) {
				if ( !( this[i].contains( node  ) ) ) {
					return false;
				}
			}
		}				
		else { //manual method for last chance corale
			while (i < length) {
				toTest = node;
				while (toTest = toTest.parentNode) {
					if (this[i] == toTest) { break; }
				}
				if (!toTest) {
					return false;
				}
			i++;
			}
		}
			
		return true;
	};
});
Glow.provide(function(glow) {
	var NodeListProto = glow.NodeList.prototype,
		document = window.document,
		undefined;
	
	// create a fragment and insert a set of nodes into it
	function createFragment(nodes) {
		var fragment = document.createDocumentFragment(),
			i = 0,
			node;
		
		while ( node = nodes[i++] ) {
			fragment.appendChild(node);
		}
		
		return fragment;
	}
	
	// generate the #before and #after methods
	// after: 1 for #(insert)after, 0 for #(insert)before
	// insert: 1 for #insert(After|Before), 0 for #(after|before)
	function afterAndBefore(after, insert) {
		return function(elements) {
			var toAddList,
				toAddToList,
				fragmentToAdd,
				nextFragmentToAdd,
				item,
				itemParent;
			
			if (!this.length) { return this; }
			
			// normalise 'elements'
			// if we're dealing with append/prepend then strings are always treated as HTML strings
			if (!insert && typeof elements === 'string') {
				elements = new glow.NodeList( glow.NodeList._strToNodes(elements) );
			}
			else {
				elements = new glow.NodeList(elements);
			}
			
			// set the element we're going to add to, and the elements we're going to add
			if (insert) {
				toAddToList = elements;
				toAddList = new glow.NodeList(this);
			}
			else {
				toAddToList = this;
				toAddList = elements;
			}
			
			nextFragmentToAdd = createFragment(toAddList);
			
			for (var i = 0, leni = toAddToList.length, lasti = leni - 1; i < leni; i++) {
				item = toAddToList[i];
				fragmentToAdd = nextFragmentToAdd;
				
				// we can only append after if the element has a parent right?
				if (itemParent = item.parentNode) {
					if (i !== lasti) { // if not the last item
						nextFragmentToAdd = fragmentToAdd.cloneNode(true);
						insert && toAddList.push(nextFragmentToAdd.childNodes);
					}
					itemParent.insertBefore(fragmentToAdd, after ? item.nextSibling : item);
				}
			}
			
			return insert ? toAddList : toAddToList;
		}
	}
	
	// generate the #append, #appendTo, #prepend and #prependTo methods
	// append: 1 for #append(To), 0 for #prepend(To)
	// to: 1 for #(append|prepend)To, 0 for #(append|prepend)
	function appendAndPrepend(append, to) {
		return function(elements) {
			var toAddList,
				toAddToList,
				fragmentToAdd,
				nextFragmentToAdd,
				item;
			
			if (!this.length) { return this; }
			
			// normalise 'elements'
			// if we're dealing with append/prepend then strings are always treated as HTML strings
			if (!to && typeof elements === 'string') {
				elements = new glow.NodeList( glow.NodeList._strToNodes(elements) );
			}
			else {
				elements = new glow.NodeList(elements);
			}
				
			// set the element we're going to add to, and the elements we're going to add
			if (to) {
				toAddToList = elements;
				toAddList = new glow.NodeList(this);
			}
			else {
				toAddToList = this;
				toAddList = elements;
			}
			
			nextFragmentToAdd = createFragment(toAddList);
			
			for (var i = 0, leni = toAddToList.length, lasti = leni - 1; i < leni; i++) {
				item = toAddToList[i];
				fragmentToAdd = nextFragmentToAdd;
				
				// avoid trying to append to non-elements
				if (item.nodeType === 1) {
					if (i !== lasti) { // if not the last item
						nextFragmentToAdd = fragmentToAdd.cloneNode(true);
						// add the clones to the return element for appendTo / prependTo
						to && toAddList.push(nextFragmentToAdd.childNodes);
					}
					item.insertBefore(fragmentToAdd, append ? null : item.firstChild);
				}
			}
			
			return to ? toAddList : toAddToList;
		}
	}
	
	/**
		@name glow.NodeList#after
		@function
		@description Insert node(s) after each node in this NodeList.
			If there is more than one node in this NodeList, 'nodes'
			will be inserted after the first element and clones will be
			inserted after each subsequent element.
			
		@param {string | HTMLElement | HTMLElement[] | glow.NodeList} nodes Node(s) to insert
			Strings will be treated as HTML strings.
		
		@returns {glow.NodeList} Original NodeList
		
		@example
			// adds a paragraph after each heading
			glow('h1, h2, h3').after('<p>That was a nice heading.</p>');
	*/
	NodeListProto.after = afterAndBefore(1);
	
	/**
		@name glow.NodeList#before
		@function
		@description Insert node(s) before each node in this NodeList.
			If there is more than one node in this NodeList, 'nodes'
			will be inserted before the first element and clones will be
			inserted before each subsequent element.
			
		@param {string | HTMLElement | HTMLElement[] | glow.NodeList} nodes Node(s) to insert
			Strings will be treated as HTML strings.
		
		@returns {glow.NodeList} Original NodeList
		
		@example
			// adds a div before each paragraph
			glow('p').before('<div>Here comes a paragraph!</div>');
	*/
	NodeListProto.before = afterAndBefore(0);
	
	/**
		@name glow.NodeList#append
		@function
		@description Appends node to each node in this NodeList.
			If there is more than one node in this NodeList, then the given nodes
			are appended to the first node and clones are appended to the other
			nodes.
			
		@param {string | HTMLElement | HTMLElement[] | glow.NodeList} nodes Nodes(s) to append
			Strings will be treated as HTML strings.
		
		@returns {glow.NodeList} Original NodeList
		
		@example
			// ends every paragraph with '...'
			glow('p').append('<span>...</span>');
	*/
	NodeListProto.append = appendAndPrepend(1);
	
	/**
		@name glow.NodeList#prepend
		@function
		@description Prepends nodes to each node in this NodeList.
			If there is more than one node in this NodeList, then the given nodes
			are prepended to the first node and clones are prepended to the other
			nodes.
			
		@param {string | HTMLElement | HTMLElement[] | glow.NodeList} nodes Nodes(s) to prepend
			Strings will be treated as HTML strings.
		
		@returns {glow.NodeList} Original NodeList
		
		@example
			// prepends every paragraph with 'Paragraph: '
			glow('p').prepend('<span>Paragraph: </span>');
	*/
	NodeListProto.prepend = appendAndPrepend(0);
	
	/**
		@name glow.NodeList#appendTo
		@function
		@description Appends nodes in this NodeList to given node(s)
			If appending to more than one node, the NodeList is appended
			to the first node and clones are appended to the others.
			
		@param {string | HTMLElement | HTMLElement[] | glow.NodeList} node Node(s) to append to.
			Strings will be treated as CSS selectors or HTML strings.
		
		@returns {glow.NodeList} The appended nodes.
		
		@example
			// appends '...' to every paragraph
			glow('<span>...</span>').appendTo('p');
	*/
	NodeListProto.appendTo = appendAndPrepend(1, 1);

	/**
		@name glow.NodeList#prependTo
		@function
		@description Prepends nodes in this NodeList to given node(s)
			If prepending to more than one node, the NodeList is prepended
			to the first node and clones are prepended to the others.
			
		@param {string | HTMLElement | HTMLElement[] | glow.NodeList} node Node(s) to prepend to
			Strings will be treated as CSS selectors or HTML strings.
		
		@returns {glow.NodeList} The prepended nodes.
		
		@example
			// prepends 'Paragraph: ' to every paragraph
			glow('<span>Paragraph: </span>').prependTo('p');
	*/
	NodeListProto.prependTo = appendAndPrepend(0, 1);
	
	/**
		@name glow.NodeList#insertAfter
		@function
		@description Insert this NodeList after the given nodes
			If inserting after more than one node, the NodeList is inserted
			after the first node and clones are inserted after the others.
			
		@param {string | HTMLElement | HTMLElement[] | glow.NodeList} nodes Node(s) to insert after
			Strings will be treated as CSS selectors.
			
		@returns {glow.NodeList} Inserted nodes.
		
		@example
			// adds a paragraph after each heading
			glow('<p>HAI!</p>').insertAfter('h1, h2, h3');
	*/
	NodeListProto.insertAfter = afterAndBefore(1, 1);
	
	/**
		@name glow.NodeList#insertBefore
		@function
		@description Insert this NodeList before the given nodes
			If inserting before more than one node, the NodeList is inserted
			before the first node and clones are inserted before the others.
			
		@param {string | HTMLElement | HTMLElement[] | glow.NodeList} nodes Node(s) to insert before
			Strings will be treated as CSS selectors.
			
		@returns {glow.NodeList} Inserted nodes.
		
		@example
			// adds a div before each paragraph
			glow('<div>Here comes a paragraph!</div>').insertBefore('p');
	*/
	NodeListProto.insertBefore = afterAndBefore(0, 1);
	
	/**
		@name glow.NodeList#destroy
		@function
		@description Removes each element from the document
			The element, attached listeners & attached data will be
			destroyed to free up memory.
			
			Detroyed elements may not be reused in some browsers.
			
		@returns undefined
		
		@example
			// destroy all links in the document
			glow("a").destroy();
	*/
	var tmpDiv = document.createElement('div');
	
	NodeListProto.destroy = function() {
		var allElements = this.get('*').push(this);
		
		// remove data and listeners
		glow.events.removeAllListeners( allElements.removeData() );
		
		this.appendTo(tmpDiv);
		tmpDiv.innerHTML = '';
	};
	
	/**
		@name glow.NodeList#remove
		@function
		@description Removes each element from the document
			If you no longer need the elements, consider using
			{@link glow.NodeList#destroy destroy}
			
		@returns {glow.NodeList} The removed elements

		@example
			// take all the links out of a document
			glow("a").remove();
	*/
	NodeListProto.remove = function() {
		var parent,
			node,
			i = this.length;
		
		while (i--) {
			node = this[i];
			if (parent = node.parentNode) {
				parent.removeChild(node);
			}
		}
		
		return this;
	};
	
	/**
		@name glow.NodeList#empty
		@function
		@description Removes the nodes' contents

		@returns {glow.NodeList} Original nodes

		@example
			// remove the contents of all textareas
			glow("textarea").empty();
	*/
	// TODO: is this shortcut worth doing?
	NodeListProto.empty = glow.env.ie ?
		// When you clean an element out using innerHTML it destroys its inner text nodes in IE8 and below
		// Here's an alternative method for IE:
		function() {
			var i = this.length, node, child;
			
			while (i--) {
				node = this[i];
				while (child = node.firstChild) {
					node.removeChild(child);
				}
			}
			
			return this;
		} :
		// method for most browsers
		function() {
			var i = this.length;
			
			while (i--) {
				this[i].innerHTML = '';
			}
			
			return this;
		}

	/**
		@name glow.NodeList#replaceWith
		@function
		@description Replace elements with another
		
		@param {string | HTMLElement | HTMLElement[] | glow.NodeList} elements Element(s) to insert into the document
			If there is more than one element in the NodeList, then the given elements
			replace the first element, clones are appended to the other	elements.
			
		@returns {glow.NodeList} The replaced elements
			Call {@link glow.NodeList#destroy destroy} on these if you
			no longer need them.
	*/
	NodeListProto.replaceWith = function(elements) {
		return this.after(elements).remove();
	};
	
	/**
		@name glow.NodeList#wrap
		@function
		@description Wraps the given NodeList with the specified element(s).
			The given NodeList items will always be placed in the first
			child element that contains no further elements.
			
			Each item in a given NodeList will be wrapped individually.
		
		@param {string | HTMLElement | HTMLElement[] | glow.NodeList} wrapper Element to use as a wrapper
			Strings will be treated as HTML strings if they begin with <, else
			they'll be treated as a CSS selector.
		
		@returns {glow.NodeList} The NodeList with new wrapper parents
			
		@example
			// <span id="mySpan">Hello</span>
			glow("#mySpan").wrap("<div><p></p></div>");
			// Makes:
			// <div>
			//     <p>
			//         <span id="mySpan">Hello</span>
			//     </p>
			// </div>
			
	*/
	// get first child element node of an element, otherwise undefined
	function getFirstChildElm(parent) {					
		for (var child = parent.firstChild; child; child = child.nextSibling) {
			if (child.nodeType == 1) {
				return child;
			}			
		}			
		return undefined;			
	}
	
	NodeListProto.wrap = function(wrapper) {
		// normalise input
		wrapper = new glow.NodeList(wrapper);
		
		// escape if the wraper is non-existant or not an element
		if (!wrapper[0] || wrapper[0].nodeType != 1) {
			return this;
		}
		
		var toWrap,
			toWrapTarget,
			firstChildElm;
		
		for (var i = 0, leni = this.length; i<leni; i++) {
			toWrap = this[i];
			// get target element to insert toWrap in
			toWrapTarget = wrapper[0];
			
			while (toWrapTarget) {
				firstChildElm = getFirstChildElm(toWrapTarget);
					
				if (!firstChildElm) {
					break;
				}
				toWrapTarget = firstChildElm;
			}
			
			if (toWrap.parentNode) {						
				wrapper.insertBefore(toWrap);													
			}
			
			// If wrapping multiple nodes, we need to take a clean copy of the wrapping nodes
			if (i != leni-1) {
				wrapper = wrapper.clone();
			}
			
			toWrapTarget.appendChild(toWrap);
		}
		
		return this;
	};
	
	/**
		@name glow.NodeList#unwrap
		@function
		@description Removes the parent of each item in the list
		
		@returns {glow.NodeList} The now unwrapped elements
		
		@example
			// Before: <div><p><span id="mySpan">Hello</span></p></div>
			// unwrap the given element
			glow("#mySpan").unwrap();
			// After: <div><span id="mySpan">Hello</span></div>
	*/
	NodeListProto.unwrap = function() {
		var parentToRemove,
			childNodes,
			// get unique parents
			parentsToRemove = this.parent();
		
		for (var i = 0, leni = parentsToRemove.length; i < leni; i++) {				
			parentToRemove = parentsToRemove.slice(i, i+1);
			// make sure we get all children, including text nodes
			childNodes = new glow.NodeList( parentToRemove[0].childNodes );
			
			// if the item we're removing has no new parent (i.e. is not in document), then we just remove the child and destroy the old parent
			if (!parentToRemove[0].parentNode){
				childNodes.remove();
				parentToRemove.destroy();
			}
			else {
				childNodes.insertBefore(parentToRemove);
				parentToRemove.destroy();							
			}
		}
		return this;
	};
	
	/**
		@name glow.NodeList#clone
		@function
		@description Clones each node in the NodeList, along with data & event listeners
		
		@returns {glow.NodeList}
			Returns a new NodeList containing clones of all the nodes in
			the NodeList
		
		@example
			// get a copy of all heading elements
			var myClones = glow.get("h1, h2, h3, h4, h5, h6").clone();
	*/
	
	NodeListProto.clone = function() {
		var clonedNodeList = this.copy(),
			allCloneElms = clonedNodeList.get('*').push(clonedNodeList),
			allElms = this.get('*').push(this);
		
		// now copy over the data and events for all cloned elements
		glow.events._copyDomEvents(allElms, allCloneElms);
		glow.NodeList._copyData(allElms, allCloneElms);
		
		return clonedNodeList;
	};
	
	
	/**
		@name glow.NodeList#copy
		@function
		@description Copies each node in the NodeList, excluding data & event listeners
		
		@returns {glow.NodeList}
			Returns a new NodeList containing copies of all the nodes in
			the NodeList
		
		@example
			// get a copy of all heading elements
			var myCopies = glow.get("h1, h2, h3, h4, h5, h6").copy();
	*/
	NodeListProto.copy = function() {
		var nodes = [],
			i = this.length,
			clonedNodeList,
			allCloneElms,
			eventIdProp = '__eventId' + glow.UID,
			dataPropName = '_uniqueData' + glow.UID;
		
		while (i--) {
			nodes[i] = this[i].cloneNode(true);
		}
		
		clonedNodeList = new glow.NodeList(nodes);
		
		// IE also clones node properties as attributes
		// we need to get rid of the eventId & dataId
		if (glow.env.ie) {
			allCloneElms = clonedNodeList.get('*').push(nodes);
			i = allCloneElms.length;
			while (i--) {
				allCloneElms[i][dataPropName] = allCloneElms[i][eventIdProp] = undefined;
			}
		}
		
		return clonedNodeList;
	};
	
	/**
		@name glow.NodeList#html
		@function
		@description Gets / sets HTML content
			Either gets content of the first element, or sets the content
			for all elements in the list
			
		@param {String} [htmlString] String to set as the HTML of elements
			If omitted, the html for the first element in the list is
			returned.
		
		@returns {glow.NodeList | string}
			Returns the original NodeList when setting,
			or the HTML content when getting.
			
		@example
			// get the html in #footer
			var footerContents = glow("#footer").html();
			
		@example
			// set a new footer
			glow("#footer").html("<strong>Hello World!</strong>");
	*/
	NodeListProto.html = function(htmlString) {
		// getting
		if (!arguments.length) {
			return this[0] ? this[0].innerHTML : '';
		}
		
		// setting
		var i = this.length,
			node;
		
		// normalise the string
		htmlString = htmlString === undefined? '' : String(htmlString);
		
		while (i--) {
			node = this[i];
			if (node.nodeType == 1) {
				try {
					// this has a habit of failing in IE for some elements
					node.innerHTML = htmlString;
				}
				catch (e) {
					new glow.NodeList(node).empty().append(htmlString);
				}
			}
		}
		
		return this;
	};
	
	/**
		@name glow.NodeList#text
		@function
		@description Gets / set the text content
			Either gets content of the first element, or sets the content
			for all elements in the list
		
		@param {String} [text] String to set as the text of elements
			If omitted, the test for the first element in the list is
			returned.
		
		@returns {glow.NodeList | String}
			Returns the original NodeList when setting,
			or the text content when getting.

		@example
			// set text
			var div = glow("<div></div>").text("Fun & games!");
			// <div>Func &amp; games!</div>
			
		@example
			// get text
			var mainHeading = glow('#mainHeading').text();
	*/
	NodeListProto.text = function(textString) {
		var firstNode = this[0],
			i = this.length,
			node;
		
		// getting
		if (!arguments.length) {
			// get the text by checking a load of properties in priority order
			return firstNode ?
				firstNode.textContent ||
				firstNode.innerText ||
				firstNode.nodeValue || '' // nodeValue for comment & text nodes
				: '';
		}
		
		// setting
		// normalise the string
		textString = textString ? String(textString): '';
		
		this.empty();
		while (i--) {
			node = this[i];
			if (node.nodeType == 1) {
				node.appendChild( document.createTextNode(textString) );
			}
			else {
				node.nodeValue = textString;
			}
		}
		
		return this;
	};
});
Glow.provide(function(glow) {
	var NodeList = glow.NodeList,
		NodeListProto = NodeList.prototype,
		win = window,
		document = win.document,	
		getComputedStyle = document.defaultView && document.defaultView.getComputedStyle,
		// regex for toStyleProp
		dashAlphaRe = /-(\w)/g,
		// regex for getCssValue
		isNumberButNotPx = /^-?[\d\.]+(?!px)[%a-z]+$/i,
		ieOpacityRe = /alpha\(opacity=([^\)]+)\)/,
		// regex for #css
		hasUnits = /width|height|top$|bottom$|left$|right$|spacing$|indent$|font-size/;
	
	// replace function for toStyleProp
	function toStylePropReplace(match, p1) {
		return p1.toUpperCase();
	}
	
	/**
		@private
		@function
		@description Converts a css property name into its javascript name.
			Such as "background-color" to "backgroundColor".
		@param {string} prop CSS Property name.
		@returns {string}
	*/
	function toStyleProp(prop) {
		if (prop == 'float') {
			return glow.env.ie ? 'styleFloat' : 'cssFloat';
		}
		return prop.replace(dashAlphaRe, toStylePropReplace);
	}
	
	/**
		@private
		@function
		@description Get a total value of multiple CSS properties
		@param {HTMLElement} elm
		@param {string[]} props CSS properties to get the total value of
		@returns {number}
	*/
	function getTotalCssValue(elm, props) {
		var total = 0,
			i = props.length;
			
		while (i--) {
			total += parseFloatFunc(
				getCssValue( elm, props[i] )
			) || 0;
		}
		
		return total;
	}
	
	/**
		@private
		@function
		@description Get a computed css property
		@param {HTMLElement} elm
		@param {string} prop CSS property to get the value of
		@returns {string}
	*/
	function getCssValue(elm, prop) {
		var defaultView = elm.ownerDocument.defaultView,
			computedStyle,
			r,
			currentStyle,
			oldDisplay,
			match;
		
		if (getComputedStyle) { // the W3 way
			computedStyle = defaultView.getComputedStyle(elm, null);
			
			// http://bugs.webkit.org/show_bug.cgi?id=13343
			// Webkit fails to get margin-right for rendered elements.
			// margin-right is measured from the right of the element to the right of the parent
			if (glow.env.webkit && prop === 'margin-right') {
				oldDisplay = elm.style.display;
				elm.style.display = 'none';
				r = computedStyle[prop];
				elm.style.display = oldDisplay;
			}
			else {
				r = computedStyle.getPropertyValue(prop);
			}
		}
		else if (currentStyle = elm.currentStyle) { // the IE<9 way
			if (prop === 'opacity') { // opacity, the IE way
				match = ieOpacityRe.exec(currentStyle.filter);
				return match ? String(parseInt(match[1], 10) / 100) || '1' : '1';
			}
			// catch border-*-width. IE gets this wrong if the border style is none
			else if (
				prop.indexOf('border') === 0 &&
				prop.slice(-5) === 'width' &&
				getCssValue(elm, 'border-style') === 'none') {
				
				return '0px';
			}
			
			r = currentStyle[ toStyleProp(prop) ];
			
			// font-size gives us incorrect values when put through getPixelValue, avoid
			if (isNumberButNotPx.test(r) && prop != 'font-size') {
				r = getPixelValue( elm, r, prop.indexOf('height') >= 0 || prop.indexOf('top') >= 0 ) + 'px';
			}
		}
		
		// post-process return value
		if (prop === 'opacity') {
			r = r || '1';
		}
		else if (prop.indexOf('color') != -1) { //deal with colour values
			r = NodeList._parseColor(r).toString();
		}
		
		return r;
	}
	
	// vars used in _parseColor
	var mathRound = Math.round,
		parseIntFunc = parseInt,
		parseFloatFunc = parseFloat,
		htmlColorNames = {
			black:   0x000000,
			silver:  0xc0c0c0,
			gray:    0x808080,
			white:   0xffffff,
			maroon:  0x800000,
			red:     0xff0000,
			purple:  0x800080,
			fuchsia: 0xff00ff,
			green:   0x008000,
			lime:    0x00ff00,
			olive:   0x808000,
			yellow:  0xffff00,
			navy:    0x000080,
			blue:    0x0000ff,
			teal:    0x008080,
			aqua:    0x00ffff,
			orange:  0xffa500
		},
		// match a string like rgba(10%, 10%, 10%, 0.5) where the % and alpha parts are optional
		colorRegex = /^rgba?\(([\d\.]+)(%?),\s*([\d\.]+)(%?),\s*([\d\.]+)(%?)(?:,\s*([\d\.]+))?/i,
		transColorRegex = /^(transparent|rgba\(0, ?0, ?0, ?0\))$/,
		wordCharRegex = /\w/g;
	
	/**
		@name glow.NodeList._parseColor
		@private
		@function
		@description Convert a CSS colour string into a normalised format
		@returns {string} String in format rgb(0, 0, 0)
			Returned string also has r, g & b number properties
	*/
	NodeList._parseColor = function (val) {
		if ( transColorRegex.test(val) ) {
			return 'rgba(0, 0, 0, 0)';
		}
		
		var match, //tmp regex match holder
			r, g, b, a, //final colour vals
			hex; //tmp hex holder

		if ( match = colorRegex.exec(val) ) { //rgb() format, cater for percentages
			r = match[2] ? mathRound( parseFloatFunc(match[1]) * 2.55 ) : parseIntFunc(match[1]);
			g = match[4] ? mathRound( parseFloatFunc(match[3]) * 2.55 ) : parseIntFunc(match[3]);
			b = match[6] ? mathRound( parseFloatFunc(match[5]) * 2.55 ) : parseIntFunc(match[5]);
			a = parseFloatFunc( match[7] || '1' );
		} else {
			if (typeof val == 'number') {
				hex = val;
			}
			else if (val.charAt(0) == '#') {
				if (val.length === 4) { //deal with #fff shortcut
					val = val.replace(wordCharRegex, '$&$&');
				}
				hex = parseIntFunc(val.slice(1), 16);
			}
			else {
				hex = htmlColorNames[val];
			}

			r = (hex) >> 16;
			g = (hex & 0x00ff00) >> 8;
			b = (hex & 0x0000ff);
			a = 1;
		}

		val = new String('rgba(' + r + ', ' + g + ', ' + b + ', ' + a + ')');
		val.r = r;
		val.g = g;
		val.b = b;
		val.a = a;
		return val;
	}
	
	// vars for generateWidthAndHeight
	var horizontalBorderPadding = [
			'border-left-width',
			'border-right-width',
			'padding-left',
			'padding-right'
		],
		verticalBorderPadding = [
			'border-top-width',
			'border-bottom-width',
			'padding-top',
			'padding-bottom'
		];
	
	/**
		@private
		@function
		@description Get width or height of an element width/height.
		@param {HTMLElement} elm Element to measure.
		@param {string} 'Width' or 'Height'.
	*/
	function getDimension(elm, cssProp) {
		// exit if there's no element, or it isn't an Element, window or document
		if ( !elm || elm.nodeType === 3 || elm.nodeType === 8 ) {
			return 0;
		}
		
		var r,
			document = elm.ownerDocument || elm.document || elm,
			docElm = document.documentElement,
			docBody = document.body,
			docElmOrBody = glow.env.standardsMode ? docElm : docBody,
			isWidth = (cssProp == 'Width'),
			cssBorderPadding;

		if (elm.window) { // is window
			r = glow.env.webkit ? (isWidth ? docBody.clientWidth : elm.innerHeight) :
				/* else */        docElmOrBody['client' + cssProp];
		}
		else if (elm.getElementById) { // is document
			// we previously checked offsetWidth & clientWidth here
			// but they returned values too large in IE6
			r = Math.max(
				docBody['scroll' + cssProp],
				docElm['scroll' + cssProp]
			)
		}
		else {
			// get an array of css borders & padding
			cssBorderPadding = isWidth ? horizontalBorderPadding : verticalBorderPadding;
			r = elm['offset' + cssProp] - getTotalCssValue(elm, cssBorderPadding);
		}
		return r;
	}
	
	/**
		@private
		@function
		@description Converts a relative value into an absolute pixel value.
			Only works in IE with Dimension value (not stuff like relative font-size).
			Based on some Dean Edwards' code
		
		@param {HTMLElement} element Used to calculate relative values
		@param {string} value Relative value
		@param {boolean} useYAxis Calulate relative values to the y axis rather than x
		@returns number
	*/
	function getPixelValue(element, value, useYAxis) {
		// Remember the original values
		var axisPos = useYAxis ? 'top' : 'left',
			axisPosUpper = useYAxis ? 'Top' : 'Left',
			elmStyle = element.style,
			positionVal = elmStyle[axisPos],
			runtimePositionVal = element.runtimeStyle[axisPos],
			r;
			
		// copy to the runtime type to prevent changes to the display
		element.runtimeStyle[axisPos] = element.currentStyle[axisPos];
			// set value to left / top
		elmStyle[axisPos] = value;
		// get the pixel value
		r = elmStyle['pixel' + axisPosUpper];
			
		// revert values
		elmStyle[axisPos] = positionVal;
		element.runtimeStyle[axisPos] = runtimePositionVal;
			
		return r;
	}
	
	/**
	@name glow.NodeList#css
	@function
	@description Get / set a CSS property value
		
	@param {string | Object} property The CSS property name, or object of property-value pairs to set
		
	@param {string | number} [value] The value to apply
		Number values will be treated as 'px' unless the CSS property
		accepts a unitless value.
		
		If value is omitted, the value for the given property will be returned
			
	@returns {glow.NodeList | string} Returns the NodeList when setting value, or the CSS value when getting values.
		CSS values are strings. For instance, "height" will return
		"25px" for an element 25 pixels high. You can use
		parseInt to convert these values.
		
	@example
		// get value from first node
		glow('#subNav').css('display');
		
	@example
		// set left padding to 10px on all nodes
		glow('#subNav li').css('padding-left', '2em');
		
	@example
		// where appropriate, px is assumed when no unit is passed
		glow('#mainPromo').css('margin-top', 300);
		
	@example
		// set multiple CSS values at once
		// NOTE: Property names containing a hyphen such as font-weight must be quoted
		glow('#myDiv').css({
			'font-weight': 'bold',
			'padding'	 : '10px',
			'color'		 : '#00cc99'
		});
	*/
	NodeListProto.css = function(prop, val) {
		var thisStyle,
			i = this.length,
			styleProp,
			style,
			firstItem = this[0];

		if (prop.constructor === Object) { // set multiple values
			for (style in prop) {
				this.css( style, prop[style] );
			}
			return this;
		}
		else if (val !== undefined) { //set one CSS value
			styleProp = toStyleProp(prop);
			while (i--) {
				if (this[i].nodeType === 1) {
					thisStyle = this[i].style;
						
					if ( !isNaN(val) && hasUnits.test(prop) ) {
						val += 'px';
					}
					
					if (prop === 'opacity' && glow.env.ie) {
						val = parseFloatFunc(val);
						//in IE the element needs hasLayout for opacity to work
						thisStyle.zoom = '1';
						thisStyle.filter = (val !== 1) ?
							'alpha(opacity=' + mathRound(val * 100) + ')' :
							'';
					}
					else {
						thisStyle[styleProp] = val;
					}
				}
			}
			return this;
		}
		else { //getting stuff
			if (prop === 'width' || prop === 'height') {
				return this[prop]() + 'px';
			}
			return (firstItem && firstItem.nodeType === 1) ? getCssValue(firstItem, prop) : '';
		}	
	};
	
	/**
	@name glow.NodeList#height
	@function
	@description Gets / set element height
		Return value does not include the padding or border of the element in
		browsers supporting the correct box model.
			
		You can use this to easily get the height of the document or
		window, see example below.
		
	@param {Number} [height] New height in pixels for each element in the list
		If ommited, the height of the first element is returned
		
	@returns {glow.NodeList | number}
		Height of first element, or original NodeList when setting heights.
		
	@example
		// get the height of #myDiv
		glow("#myDiv").height();
		
	@example
		// set the height of list items in #myList to 200 pixels
		glow("#myList > li").height(200);
		
	@example
		// get the height of the document
		glow(document).height();
		
	@example
		// get the height of the window
		glow(win).height();
	*/
	NodeListProto.height = function(height) {
		if (height === undefined) {
			return getDimension(this[0], 'Height');
		}
		return this.css('height', height);	
	};
	
	/**
	@name glow.NodeList#width
	@function
	@description Gets / set element width
		Return value does not include the padding or border of the element in
		browsers supporting the correct box model.
			
		You can use this to easily get the width of the document or
		window, see example below.
		
	@param {Number} [width] New width in pixels for each element in the list
		If ommited, the width of the first element is returned
		
	@returns {glow.NodeList | number}
		width of first element, or original NodeList when setting widths.
		
	@example
		// get the width of #myDiv
		glow("#myDiv").width();
		
	@example
		// set the width of list items in #myList to 200 pixels
		glow("#myList > li").width(200);
		
	@example
		// get the width of the document
		glow(document).width();
		
	@example
		// get the width of the window
		glow(window).width();
	*/
	NodeListProto.width = function(width) {
		if (width === undefined) {
			return getDimension(this[0], 'Width');
		}
		return this.css('width', width);
	};
	
	/**
	@name glow.NodeList#scrollLeft
	@function
	@description Gets/sets the number of pixels the element has scrolled horizontally
		To get/set the scroll position of the window, use this method on
		a nodelist containing the window object.
			
	@param {Number} [val] New left scroll position
		Omit this to get the current scroll position
			
	@returns {glow.NodeList | number}
		Current scrollLeft value, or NodeList when setting scroll position.
			
	@example
		// get the scroll left value of #myDiv
		var scrollPos = glow('#myDiv').scrollLeft();
		// scrollPos is a number, eg: 45

	@example
		// set the scroll left value of #myDiv to 20
		glow('#myDiv').scrollLeft(20);

	@example
		// get the scrollLeft of the window
		glow(window).scrollLeft();
		// scrollPos is a number, eg: 45
	*/
	NodeListProto.scrollLeft = function(val) {
		return scrollOffset(this, true, val);	
	};
	
	/**
	@name glow.NodeList#scrollTop
	@function
	@description Gets/sets the number of pixels the element has scrolled vertically
		To get/set the scroll position of the window, use this method on
		a nodelist containing the window object.
		
	@param {Number} [val] New top scroll position
		Omit this to get the current scroll position
			
	@returns {glow.NodeList | number}
		Current scrollTop value, or NodeList when setting scroll position.

	@example
		// get the scroll top value of #myDiv
		var scrollPos = glow("#myDiv").scrollTop();
		// scrollPos is a number, eg: 45

	@example
		// set the scroll top value of #myDiv to 20
		glow("#myDiv").scrollTop(20);

	@example
		// get the scrollTop of the window
		glow(window).scrollTop();
		// scrollPos is a number, eg: 45
	*/
	NodeListProto.scrollTop = function(val) {
		return scrollOffset(this, false, val);	
	};
	/**
	@name glow.dom-getScrollOffset
	@private
	@description Get the scrollTop / scrollLeft of a particular element
	@param {Element} elm Element (or window object) to get the scroll position of
	@param {Boolean} isLeft True if we're dealing with left scrolling, otherwise top
	*/
	function getScrollOffset(elm, isLeft) {
		var r,			
			scrollProp = 'scroll' + (isLeft ? 'Left' : 'Top');
			
		// are we dealing with the window object or the document object?
		if (elm.window) {
			// get the scroll of the documentElement or the pageX/Yoffset
			// - some browsers use one but not the other
			r = elm.document.documentElement[scrollProp]
				|| (isLeft ? elm.pageXOffset : elm.pageYOffset)
				|| 0;
		} else {
			r = elm[scrollProp];
		}
		return r;
	}
		
	/**
	@name glow.dom-setScrollOffset
	@private
	@description Set the scrollTop / scrollLeft of a particular element
	@param {Element} elm Element (or window object) to get the scroll position of
	@param {Boolean} isLeft True if we're dealing with left scrolling, otherwise top
	@param {Number} newVal New scroll value
	*/
	function setScrollOffset(elm, isLeft, newVal) {
	// are we dealing with the window object or the document object?
		if (elm.window) {
			// we need to get whichever value we're not setting
			elm.scrollTo(
				isLeft  ? newVal : getScrollOffset(elm, true),
				!isLeft ? newVal : getScrollOffset(elm, false)
			);
		} else {
			elm['scroll' + (isLeft ? 'Left' : 'Top')] = newVal;
		}
	}
	
	/**
	@name glow.dom-scrollOffset
	@private
	@description Set/get the scrollTop / scrollLeft of a NodeList
	@param {glow.dom.NodeList} nodeList Elements to get / set the position of
	@param {Boolean} isLeft True if we're dealing with left scrolling, otherwise top
	@param {Number} [val] Val to set (if not provided, we'll get the value)
	@returns NodeList for sets, Number for gets
	*/
	function scrollOffset(nodeList, isLeft, val) {
		var i = nodeList.length;
			
		if (val !== undefined) {
			while (i--) {
				setScrollOffset(nodeList[i], isLeft, val);
			}
			return nodeList;
		} else {
			return getScrollOffset(nodeList[0], isLeft);
		}
	}
	/**
	@name glow.NodeList#hide
	@function
	@description Hides all items in the NodeList.
		
	@returns {glow.NodeList}
		
	@example
		// Hides all list items within #myList
		glow("#myList li").hide();
	*/
	NodeListProto.hide = function() {
		return this.css('display', 'none').css('visibility', 'hidden');	
	};
	
	/**
	@name glow.NodeList#show
	@function
	@description Shows all hidden items in the NodeList.
		
	@returns {glow.NodeList}
		
	@example
		// Show element with ID myDiv
		glow("#myDiv").show();
			
	@example
		// Show all list items within #myList
		glow("#myList li").show();
	*/
	NodeListProto.show = function() {
		var i = this.length,
			currItem,
			itemStyle;
			
		while (i--) {
			/* Create a NodeList for the current item */
			currItem = new glow.NodeList(this[i]);
			itemStyle = currItem[0].style;
			if (currItem.css('display') == 'none') {
				itemStyle.display = '';
				itemStyle.visibility = 'visible';
				/* If display is still none, set to block */
				if (currItem.css('display') == 'none') {
					itemStyle.display = 'block';
				}
			}	
		}
		return this;	
	};

	/**
	@name glow.NodeList#offset
	@function
	@description Gets the offset from the top left of the document.
		If the NodeList contains multiple items, the offset of the
		first item is returned.
			
	@returns {Object}
		Returns an object with "top" & "left" properties in pixels
			
	@example
		glow("#myDiv").offset().top
	*/
	NodeListProto.offset = function() {
		if ( !this[0] || this[0].nodeType !== 1) {
			return {top: 0, left: 0};
		}
		
		// http://weblogs.asp.net/bleroy/archive/2008/01/29/getting-absolute-coordinates-from-a-dom-element.aspx - great bit of research, most bugfixes identified here (and also jquery trac)
		var elm = this[0],
			doc = elm.ownerDocument,
			docElm = doc.documentElement,
			window = doc.defaultView || doc.parentWindow,
			docScrollPos = {
				x: getScrollOffset(window, true),
				y: getScrollOffset(window, false)
			};

		//this is simple(r) if we can use 'getBoundingClientRect'
		// Sorry but the sooper dooper simple(r) way is not accurate in Safari 4
		if (!glow.env.webkit && elm.getBoundingClientRect) {
			var rect = elm.getBoundingClientRect();
			
			return {
				top: Math.floor(rect.top)
					/*
					 getBoundingClientRect is realive to top left of
					 the viewport, so we need to sort out scrolling offset
					*/
					+ docScrollPos.y
					/*
					IE adds the html element's border to the value. We can
					deduct this value using client(Top|Left). However, if
					the user has done html{border:0} clientTop will still
					report a 2px border in IE quirksmode so offset will be off by 2.
					Hopefully this is an edge case but we may have to revisit this
					in future
					*/
					- docElm.clientTop,

				left: Math.floor(rect.left) //see above for docs on all this stuff
					+ docScrollPos.x
					- docElm.clientLeft
			};
		}
		else { //damnit, let's go the long way around
			var top = elm.offsetTop,
				left = elm.offsetLeft,
				originalElm = elm,
				nodeNameLower,
				docBody = document.body,
				//does the parent chain contain a position:fixed element
				involvesFixedElement = false,
				offsetParentBeforeBody = elm;

			//add up all the offset positions
			while (elm = elm.offsetParent) {
				left += elm.offsetLeft;
				top += elm.offsetTop;

				//if css position is fixed, we need to add in the scroll offset too, catch it here
				if (getCssValue(elm, 'position') == 'fixed') {
					involvesFixedElement = true;
				}

				//gecko & webkit (safari 3) don't add on the border for positioned items
				if (glow.env.gecko || glow.env.webkit > 500) {
					left += parseInt(getCssValue(elm, 'border-left-width')) || 0;
					top  += parseInt(getCssValue(elm, 'border-top-width'))  || 0;
				}
				
				//we need the offset parent (before body) later
				if (elm.nodeName.toLowerCase() != 'body') {
					offsetParentBeforeBody = elm;
				}
			}

			//deduct all the scroll offsets
			elm = originalElm;
			
			while ((elm = elm.parentNode) && (elm != docBody) && (elm != docElm)) {
				left -= elm.scrollLeft;
				top -= elm.scrollTop;

				//FIXES
				//gecko doesn't add the border of contained elements to the offset (overflow!=visible)
				if (glow.env.gecko && getCssValue(elm, 'overflow') != 'visible') {
					left += parseInt(getCssValue(elm, 'border-left-width'));
					top += parseInt(getCssValue(elm, 'border-top-width'));
				}
			}

			//if we found a fixed position element we need to add the scroll offsets
			if (involvesFixedElement) {
				left += docScrollPos.x;
				top += docScrollPos.y;
			}

			//FIXES
			// Gecko - non-absolutely positioned elements that are direct children of body get the body offset counted twice
			if (
				(glow.env.gecko && getCssValue(offsetParentBeforeBody, 'position') != 'absolute')
			) {
				left -= docBody.offsetLeft;
				top -= docBody.offsetTop;
			}

			return {left:left, top:top};
		}
	};
	
	/**
	@name glow.NodeList#position
	@function
	@description Get the top & left position of an element relative to its positioned parent
		This is useful if you want to make a position:static element position:absolute
		and retain the original position of the element
			
	@returns {Object}
		An object with 'top' and 'left' number properties
		
	@example
		// get the top distance from the positioned parent
		glow("#elm").position().top
	*/
	NodeListProto.position = function() {
		var positionedParent = new glow.NodeList( getPositionedParent(this[0]) ),
			hasPositionedParent = !!positionedParent[0],
					
			// element margins to deduct
			marginLeft = parseInt( this.css('margin-left') ) || 0,
			marginTop  = parseInt( this.css('margin-top')  ) || 0,
					
			// offset parent borders to deduct, set to zero if there's no positioned parent
			positionedParentBorderLeft = ( hasPositionedParent && parseInt( positionedParent.css('border-left-width') ) ) || 0,
			positionedParentBorderTop  = ( hasPositionedParent && parseInt( positionedParent.css('border-top-width')  ) ) || 0,
					
			// element offsets
		elOffset = this.offset(),
		positionedParentOffset = hasPositionedParent ? positionedParent.offset() : {top: 0, left: 0};
				
		return {
			left: elOffset.left - positionedParentOffset.left - marginLeft - positionedParentBorderLeft,
			top:  elOffset.top  - positionedParentOffset.top  - marginTop  - positionedParentBorderTop
		}	
	};
	/*
		Get the 'real' positioned parent for an element, otherwise return null.
	*/
	function getPositionedParent(elm) {
		var offsetParent = elm.offsetParent,
		docElm = document.documentElement;
			
		// get the real positioned parent
		// IE places elements with hasLayout in the offsetParent chain even if they're position:static
		// Also, <body> and <html> can appear in the offsetParent chain, but we don't want to return them if they're position:static
		while (offsetParent && new glow.NodeList(offsetParent).css('position') == 'static') {	
			offsetParent = offsetParent.offsetParent;
		}
			
		// sometimes the <html> element doesn't appear in the offsetParent chain, even if it has position:relative
		if (!offsetParent && new glow.NodeList(docElm).css('position') != 'static') {
			offsetParent = docElm;
		}
			
		return offsetParent || null;
	}
});
Glow.provide(function(glow) {
	var NodeListProto = glow.NodeList.prototype,
		document = window.document,
		undefined,
		keyEventNames = ' keypress keydown keyup ';
	
	/**
		@name glow.NodeList#on
		@function
		@description Listen for an event.
		   This will listen for a particular event on each dom node
		   in the NodeList.
		   
		   If you're listening to many children of a particular item,
		   you may get better performance from {@link glow.NodeList#delegate}.
		
		@param {String} eventName Name of the event to listen for.
		   This can be any regular DOM event ('click', 'mouseover' etc) or
		   a special event of NodeList.
		   
		@param {Function} callback Function to call when the event fires.
		   The callback is passed a single event object. The type of this
		   object is {@link glow.DomEvent} unless otherwise stated.
		   
		@param {Object} [thisVal] Value of 'this' within the callback.
		   By default, this is the dom node being listened to.
		
		@returns this
		
		@example
		   glow.get('#testLink').on('click', function(domEvent) {
			   // do stuff
			   
			   // if you want to cancel the default action (following the link)...
			   return false;
		   });
	*/
	NodeListProto.on = function(eventName, callback, thisVal) {
		var isKeyEvent = (keyEventNames.indexOf(' ' + eventName + ' ') > -1);
		
		// add standard glow listeners
		glow.events.addListeners(this, eventName, callback, thisVal);
		
		// add the bridge functions if needed
		if (isKeyEvent) {
			glow.events._addKeyListener(this);
		}
		else { // assume it's a DOM event
			glow.events._addDomEventListener(this, eventName);
		}
		
		return this;
	}
	
	/**
		@name glow.NodeList#detach
		@function
		@description detach a listener from elements
		   This will detach the listener from each dom node in the NodeList.
		
		@param {String} eventName Name of the event to detach the listener from
		   
		@param {Function} callback Listener callback to detach
		
		@returns this
		
		@example
			function clickListener(domEvent) {
				// ...
			}
			
			// adding listeners
			glow.get('a').on('click', clickListener);
			
			// removing listeners
			glow.get('a').detach('click', clickListener);
	*/
	NodeListProto.detach = function(eventName, callback) {
		var isKeyEvent = (keyEventNames.indexOf(' ' + eventName + ' ') > -1);
		
		// remove standard glow listeners
		glow.events.removeListeners(this, eventName, callback);
		
		// remove the bridge functions if needed
		if (isKeyEvent) {
			glow.events._removeKeyListener(this);
		}
		else { // assume it's a DOM event
			glow.events._removeDomEventListener(this, eventName);
		}
		
		return this;
	}
	
	/**
		@name glow.NodeList#delegate
		@function
		@description Listen for an event occurring on child elements matching a selector.
			'delegate' will catch events which occur on matching items created after
			the listener was added. 
		
		@param {String} eventName Name of the event to listen for.
			This can be any regular DOM event ('click', 'mouseover' etc) or
			a special event of NodeList.
		
		@param {String} selector CSS selector of child elements to listen for events on
			For example, if you were wanting to hear events from links, this
			would be 'a'.
		
		@param {Function} callback Function to call when the event fires.
			The callback is passed a single event object. The type of this
			object is {@link glow.DomEvent} unless otherwise stated.
		
		@param {Object} [thisVal] Value of 'this' within the callback.
			By default, this is the dom node matched by 'selector'.
		
		@returns this
		
		@example
			// Using 'on' to catch clicks on links in a list
			glow.get('#nav a').on('click', function() {
				// do stuff
			});
			
			// The above adds a listener to each link, any links created later
			// will not have this listener, so we won't hear about them.
			
			// Using 'delegate' to catch clicks on links in a list
			glow.get('#nav').delegate('click', 'a', function() {
				// do stuff
			});
			
			// The above only adds one listener to #nav which tracks clicks
			// to any links within. This includes elements created after 'delegate'
			// was called.
		
		@example
			// Using delegate to change class names on table rows so :hover
			// behaviour can be emulated in IE6
			glow.get('#contactData').delegate('mouseover', 'tr', function() {
				glow.get(this).addClass('hover');
			});
			
			glow.get('#contactData').delegate('mouseout', 'tr', function() {
				glow.get(this).removeClass('hover');
			});
	*/
	NodeListProto.delegate = function(eventName, selector, callback, thisVal) {
		var isKeyEvent = (keyEventNames.indexOf(' ' + eventName + ' ') > -1);
		
		// add standard glow listeners
		glow.events.addListeners(this, eventName + '/' + selector, callback, thisVal);
		
		// register delegates
		glow.events._registerDelegate(this, eventName, selector);
		
		// add the bridge functions if needed
		if (isKeyEvent) {
			glow.events._addKeyListener(this);
		}
		else { // assume it's a DOM event
			glow.events._addDomEventListener(this, eventName);
		}
		
		return this;
	}
	
	/**
		@name glow.NodeList#detachDelegate
		@function
		@description detach a delegated listener from elements
		   This will detach the listener from each dom node in the NodeList.
		
		@param {String} eventName Name of the event to detach the listener from
		
		@param {String} selector CSS selector of child elements the listener is listening to
		
		@param {Function} callback Listener callback to detach
		
		@returns this
		
		@example
			function clickListener(domEvent) {
				// ...
			}
			
			// adding listeners
			glow.get('#nav').delegate('click', 'a', clickListener);
			
			// removing listeners
			glow.get('#nav').detachDelegate('click', 'a', clickListener);
	*/
	NodeListProto.detachDelegate = function(eventName, selector, callback, thisVal) {
		var isKeyEvent = (keyEventNames.indexOf(' ' + eventName + ' ') > -1);
		
		// remove standard glow listeners
		glow.events.removeListeners(this, eventName + '/' + selector, callback);
		
		// unregister delegates
		glow.events._unregisterDelegate(this, eventName, selector);
		
		// remove the bridge functions if needed
		if (isKeyEvent) {
			glow.events._removeKeyListener(this);
		}
		else { // assume it's a DOM event
			glow.events._removeDomEventListener(this, eventName);
		}
		
		return this;
	}
	
	/**
		@name glow.NodeList#fire
		@function
		@param {String} eventName Name of the event to fire
		@param {glow.events.Event} [event] Event object to pass into listeners.
		   You can provide a simple object of key / value pairs which will
		   be added as properties of a glow.events.Event instance.
		
		@description Fire an event on dom nodes within the NodeList
		   Note, this will only trigger event listeners to be called, it won't
		   for example, move the mouse or click a link on the page.
		
		@returns glow.events.Event
		
		@example
		   glow.get('#testLink').on('click', function() {
			   alert('Link clicked!');
		   });
		   
		   // The following causes 'Link clicked!' to be alerted, but doesn't
		   // cause the browser to follow the link
		   glow.get('#testLink').fire('click');
	*/
	NodeListProto.fire = function(eventName, event) {
		return glow.events.fire(this, eventName, event);
	}
	
	/**
		@name glow.NodeList#event:mouseenter
		@event
		@description Fires when the mouse enters the element specifically, does not bubble
		
		@param {glow.events.DomEvent} event Event Object
	*/
	
	/**
		@name glow.NodeList#event:mouseleave
		@event
		@description Fires when the mouse leaves the element specifically, does not bubble
		
		@param {glow.events.DomEvent} event Event Object
	*/
	
	/**
		@name glow.NodeList#event:keydown
		@event
		@description Fires when the user presses a key
			Only fires if the element has focus, listen for this event on
			the document to catch all keydowns.
			
			This event related to the user pressing a key on the keyboard,
			if you're more concerned about the character entered, see the
			{@link glow.NodeList#event:keypress keypress} event.
			
			keydown will only fire once, when the user presses the key.
			
			The order of events is keydown, keypress*, keyup. keypress may
			fire many times if the user holds the key down.
		
		@param {glow.events.KeyboardEvent} event Event Object
	*/
	
	/**
		@name glow.NodeList#event:keypress
		@event
		@description Fires when a key's command executes.
			For instance, if you hold down a key, it's action will occur many
			times. This event will fire on each action.
			
			This event is useful when you want to react to keyboard repeating, or
			to detect when a character is entered into a field.
			
			The order of events is keydown, keypress*, keyup. keypress may
			fire many times if the user holds the key down.
		
		@param {glow.events.KeyboardEvent} event Event Object
	*/
	
	/**
		@name glow.NodeList#event:keyup
		@event
		@description Fires when the user releases a key
			Only fires if the element has focus, listen for this event on
			the document to catch all keyups.
			
			This event related to the user pressing a key on the keyboard,
			if you're more concerned about the character entered, see the
			{@link glow.NodeList#event:keypress keypress} event.
			
			The order of events is keydown, keypress*, keyup. keypress may
			fire many times if the user holds the key down.
		
		@param {glow.events.KeyboardEvent} event Event Object
	*/
});
Glow.provide(function(glow) {
	var NodeList = glow.NodeList,
		NodeListProto = NodeList.prototype,
		undefined,
		parseFloat = window.parseFloat,
		// used to detect which CSS properties require units
		requiresUnitsRe = /width|height|top$|bottom$|left$|right$|spacing$|indent$|fontSize/i,
		// which simple CSS values cannot be negative
		noNegativeValsRe = /width|height|padding|opacity/,
		getUnit = /\D+$/,
		usesYAxis = /height|top/;
	
	// TODO: get this from appearence.js
	function toStyleProp(prop) {
		if (prop == 'float') {
			return glow.env.ie ? 'styleFloat' : 'cssFloat';
		}
		return prop.replace(/-(\w)/g, function(match, p1) {
			return p1.toUpperCase();
		});
	}
	
	/**
		@private
		@function
		@param {nodelist} element
		@param {string} toUnit (em|%|pt...)
		@param {string} axis (x|y)
		@description Converts a css unit.
			We need to know the axis for calculating relative values, since they're
			relative to the width / height of the parent element depending
			on the situation.
	*/
	var testElement = glow('<div style="position:absolute;visibility:hidden;border:0;margin:0;padding:0"></div>');
	
	function convertCssUnit(element, value, toUnit, axis) {
		var elmStyle = testElement[0].style,
			axisProp = (axis === 'x') ? 'width' : 'height',
			startPixelValue,
			toUnitPixelValue;
		
		startPixelValue = testElement.css(axisProp, value).insertAfter(element)[axisProp]();
		// using 10 of the unit then dividing by 10 to increase accuracy
		toUnitPixelValue = testElement.css(axisProp, 10 + toUnit)[axisProp]() / 10;
		testElement.remove();
		return startPixelValue / toUnitPixelValue;
	}
	
	/**
		@private
		@function
		@description Animate a colour value
	*/
	function animateColor(anim, stylePropName, from, to) {
		to = NodeList._parseColor(to);
		to = [to.r, to.g, to.b];
		from = NodeList._parseColor(from);
		from = [from.r, from.g, from.b];
		
		anim.prop(stylePropName, {
			// we only need a template if we have units
			template: 'rgb(?,?,?)',
			from: from,
			to: to,
			round: true,
			min: 0,
			max: 255
		});
	}
	
	/**
		@private
		@function
		@description Animate opacity in IE's 'special' way
	*/
	function animateIeOpacity(elm, anim, from, to) {
		to   = parseFloat(to)   * 100;
		from = parseFloat(from) * 100;
		
		// give the element 'hasLayout'
		elm.style.zoom = 1;
		
		anim.prop('filter', {
			// we only need a template if we have units
			template: 'alpha(opacity=?)',
			from: from,
			to: to,
			allowNegative: false
		});
	}
	
	/**
		@private
		@function
		@description Scroll positions
	*/
	function animateScroll(elm, anim, from, to, scrollTopOrLeft) {
		var diff;
		
		to   = parseFloat(to);
		from = parseFloat(from);
		elm = glow(elm);
		
		// auto-get start value if there isn't one
		if ( isNaN(from) ) {
			from = elm[scrollTopOrLeft]();
		}
		
		diff = to - from;
		
		anim.on('frame', function() {
			elm[scrollTopOrLeft]( diff * this.value + from );
		});
	}
	
	/**
		@private
		@function
		@description Animate simple values
			This is a set of space-separated numbers (42) or numbers + unit (42em)
			
			Units can be mixed
	*/
	function animateValues(element, anim, stylePropName, from, to) {
		var toUnit,
			fromUnit,
			round = [],
			template = '',
			requiresUnits = requiresUnitsRe.test(stylePropName),
			minZero = noNegativeValsRe.test(stylePropName);
		
		from = String(from).split(' ');
		to = String(to).split(' ');
		
		for (var i = 0, leni = to.length; i < leni; i++) {
			toUnit   = ( getUnit.exec( to[i] )   || [''] )[0];
			fromUnit = ( getUnit.exec( from[i] ) || [''] )[0];
			
			// create initial units if required
			if (requiresUnits) {
				toUnit = toUnit || 'px';
				fromUnit = fromUnit || 'px';
			}
			
			round[i] = (toUnit === 'px');
			
			// make the 'from' unit the same as the 'to' unit
			if (toUnit !== fromUnit) {
				from = convertCssUnit( element, from, toUnit, usesYAxis.test(stylePropName) ? 'y' : 'x' );
			}
			
			template += ' ?' + toUnit;
			from[i] = parseFloat( from[i] );
			to[i]   = parseFloat( to[i] );
		}
		
		anim.prop(stylePropName, {
			template: template,
			from: from,
			to: to,
			round: round,
			min: minZero ? 0 : undefined
		});
	}
	
	/**
		@private
		@function
		@description Makes an animtion adjust CSS values over time
	*/
	function addCssAnim(nodeList, anim, properties) {
		var to, from, i,
			property,
			propertyIsArray,
			stylePropName;
		
		for (var propName in properties) {
			property = properties[propName];
			propertyIsArray = property.push;
			stylePropName = toStyleProp(propName);
			to = propertyIsArray ? property[1] : property;
			i = nodeList.length;
			
			// do this for each nodelist item
			while (i--) {
				// deal with special values, scrollTop and scrollLeft which aren't really CSS
				// This is the only animation that can work on the window object too
				if ( propName.indexOf('scroll') === 0 && (nodeList[i].scrollTo || nodeList[i].scrollTop !== undefined) ) {
					animateScroll(nodeList[i], anim, propertyIsArray ? property[0] : undefined, to, propName);
					continue;
				}
				
				// skip non-element nodes
				if ( nodeList[i].nodeType !== 1 ) { continue; }
				
				// set new target
				anim.target( nodeList[i].style );
				
				from = propertyIsArray ? property[0] : nodeList.item(i).css(propName);
				
				// deal with colour values
				if ( propName.indexOf('color') !== -1 ) {
					animateColor(anim, stylePropName, from, to);
				}
				// nice special case for IE
				else if (glow.env.ie && stylePropName === 'opacity') {
					animateIeOpacity(nodeList[i], anim, from, to);
				}
				// assume we're dealing with simple numbers, or numbers + unit
				// eg "5px", "5px 2em", "10px 5px 1em 4px"
				else {
					animateValues(nodeList[i], anim, stylePropName, from, to);
				}
			}
		}
	}
	
	/**
		@name glow.NodeList#anim
		@function
		@description Animate properties of elements
			All elements in the NodeList are animated
			
			All CSS values which are simple numbers (with optional unit)
			are supported. Eg: width, margin-top, left
			
			All CSS values which are space-separated values are supported
			(eg background-position, margin, padding), although a 'from'
			value must be provided for short-hand properties like 'margin'.
			
			All CSS colour values are supported. Eg: color, background-color.
			
			'scrollLeft' and 'scrollTop' can be animated for elements and
			the window object.
			
			Other properties, including CSS properties with limited support, can
			be animated using {@link glow.anim.Anim#prop}.
		
		@param {number} duration Length of the animation in seconds.
		@param {Object} properties Properties to animate.
			This is an object where the key is the CSS property and the value
			is the value to animate to.
			
			The value can also be an array, where the first item is the value to
			animate from, and the second is the value to animate to.
			
			Numerical values will be treated as 'px' if the property requires units.
		
		@param {Object} [opts] Options object
		@param {function|string} [opts.tween='easeBoth'] The motion of the animation.
			Strings are treated as properties of {@link glow.tweens}, although
			a tween function can be provided.
		@param {boolean} [opts.destroyOnComplete=true] Destroy the animation once it completes (unless it loops).
			This will free any DOM references the animation may have created. Once
			the animation is destroyed, it cannot be started again.
		@param {boolean} [opts.loop=true] Loop the animation.
		@param {boolean} [opts.startNow=true] Start the animation straight away?
			Animations can be started by calling {@link glow.anim.Anim#start}
		
		@returns {glow.anim.Anim}
		
		@example
			// change the nav's background colour to white and the top position
			// to 20px over a duration of 3 seconds
			glow('#nav').anim(3, {
				'background-color': '#fff',
				'top': 20
			});
			
		@example
			// Fade an element out and alert 'done' when complete
			glow('#nav').anim(3, {
				'opacity': 0
			}).on('complete', function() {
				alert('done!');
			});
			
		@example
			// Scroll the window to the top
			glow(window).anim(2, {
				scrollTop: 0
			});
		
		@see {@link glow.NodeList#queueAnim} - Queue an animation to run after the current anim
		@see {@link glow.NodeList#fadeIn} - Shortcut to fade elements in
		@see {@link glow.NodeList#fadeOut} - Shortcut to fade elements out
		@see {@link glow.NodeList#fadeToggle} - Shortcut to toggle the fade of an element
		@see {@link glow.NodeList#slideOpen} - Shortcut to slide an element open
		@see {@link glow.NodeList#slideShut} - Shortcut to slide an element shut
		@see {@link glow.NodeList#slideToggle} - Shortcut to toggle an element open / shut

	*/
	NodeListProto.anim = function(duration, properties, opts) {
		/*!debug*/
			if (arguments.length < 2 || arguments.length > 3) {
				glow.debug.warn('[wrong count] glow.NodeList#anim expects 2 or 3 arguments, not ' + arguments.length + '.');
			}
			if (typeof duration !== 'number') {
				glow.debug.warn('[wrong type] glow.NodeList#anim expects number as "duration" argument, not ' + typeof duration + '.');
			}
			if (typeof properties !== 'object') {
				glow.debug.warn('[wrong type] glow.NodeList#anim expects object as "properties" argument, not ' + typeof properties + '.');
			}
			if (opts !== undefined && typeof opts !== 'object') {
				glow.debug.warn('[wrong type] glow.NodeList#anim expects object as "opts" argument, not ' + typeof opts + '.');
			}
		/*gubed!*/
		
		opts = opts || {};
		
		var anim = new glow.anim.Anim(duration, opts);
		
		addCssAnim(this, anim, properties);
		
		// auto start
		!(opts.startNow === false) && anim.start();
		return anim;
	};
	
	/**
		@private
		@function
		@description Used as a listener for an animations's stop event.
			'this' is a nodelist of the animating item
			
			Set in queueAnim
	*/
	function queueAnimStop() {
		this.removeData('glow_lastQueuedAnim').removeData('glow_currentAnim');
	}
	
	/**
		@name glow.NodeList#queueAnim
		@function
		@description Queue an animation to run after the current animation
			All elements in the NodeList are animated
		
			This supports the same CSS properties as {@link glow.NodeList#anim},
			but the animation is not started until the previous animation (added
			via {@link glow.NodeList#queueAnim queueAnim})
			on that element ends.
			
			If there are no queued animations on the element, the animation starts
			straight away.
		
		@param {number} duration Length of the animation in seconds.
		@param {Object} Properties to animate.
			This is an object where the key is the CSS property and the value
			is the value to animate to.
			
			The value can also be an array, where the first item is the value to
			animate from, and the second is the value to animate to.
			
			Numerical values will be treated as 'px' if the property requires units.
		
		@param {Object} [opts] Options object
		@param {function|string} [opts.tween='easeBoth'] The motion of the animation.
			Strings are treated as properties of {@link glow.tweens}, although
			a tween function can be provided.
		@param {boolean} [opts.destroyOnComplete=true] Destroy the animation once it completes (unless it loops).
			This will free any DOM references the animation may have created. Once
			the animation is destroyed, it cannot be started again.
		
		@returns {glow.NodeList}
		
		@example
			// change a nav item's background colour from white to yellow
			// when the mouse is over it, and back again when the mouse
			// exits.
			glow('#nav').delegate('mouseenter', 'li', function() {
				glow(this).queueAnim(0.5, {
					'background-color': 'yellow'
				});
			}).delegate('mouseleave', 'li', function() {
				glow(this).queueAnim(0.5, {
					'background-color': 'white'
				});
			});
			
		@example
			// adding listeners to a queued anim
			glow('#elementToAnimate').queueAnim(0.5, {
				height: 0
			}).lastQueuedAnim().on('complete', function() {
				alert('Animation complete!');
			});
			
		@example
			// stopping and clearing current animation queue.
			// The next animation created via queueAnim will start
			// immediately
			glow('#elementToAnimate').curentAnim().stop();
		
		@see {@link glow.NodeList#fadeIn} - Shortcut to fade elements in
		@see {@link glow.NodeList#fadeOut} - Shortcut to fade elements out
		@see {@link glow.NodeList#fadeToggle} - Shortcut to toggle the fade of an element
		@see {@link glow.NodeList#slideOpen} - Shortcut to slide an element open
		@see {@link glow.NodeList#slideShut} - Shortcut to slide an element shut
		@see {@link glow.NodeList#slideToggle} - Shortcut to toggle an element open / shut

	*/
	NodeListProto.queueAnim = function(duration, properties, opts) {
		/*!debug*/
			if (arguments.length < 2 || arguments.length > 3) {
				glow.debug.warn('[wrong count] glow.NodeList#queueAnim expects 2 or 3 arguments, not ' + arguments.length + '.');
			}
			if (typeof duration !== 'number') {
				glow.debug.warn('[wrong type] glow.NodeList#queueAnim expects number as "duration" argument, not ' + typeof duration + '.');
			}
			if (typeof properties !== 'object') {
				glow.debug.warn('[wrong type] glow.NodeList#queueAnim expects object as "properties" argument, not ' + typeof properties + '.');
			}
			if (opts !== undefined && typeof opts !== 'object') {
				glow.debug.warn('[wrong type] glow.NodeList#queueAnim expects object as "opts" argument, not ' + typeof opts + '.');
			}
		/*gubed!*/
		
		opts = opts || {};
		
		var i = this.length,
			item,
			lastQueuedAnim,
			anim,
			startNextAnim;
		
		// we don't want animations starting now
		opts.startNow = false;
		
		while (i--) {
			item = this.item(i);
			if (item[0].nodeType !== 1) { continue; }
			lastQueuedAnim = item.data('glow_lastQueuedAnim');
			// add a listener to 'stop', to clear the queue
			anim = new glow.anim.Anim(duration, opts).on('stop', queueAnimStop, item);
			item.data('glow_lastQueuedAnim', anim);
			
			// closure some properties
			(function(item, properties, anim) {
				startNextAnim = function() {
					addCssAnim(item, anim, properties);
					anim.start();
					item.data('glow_currentAnim', anim);
				}
			})(item, properties, anim);
			
			// do we start the anim now, or after the next one?
			if (lastQueuedAnim) {
				lastQueuedAnim.on('complete', startNextAnim);
			}
			else {
				startNextAnim();
			}
		}
		
		return this;
	};
	
	/**
		@name glow.NodeList#currentAnim
		@function
		@description Get the currently playing animation added via {@link glow.NodeList#queueAnim queueAnim} for this element
			If no animation is currently playing, an empty animation is returned.
			This means you don't need to check to see if the item is defined before
			calling methods on it.
			
			This method acts on the first item in the NodeList.
		
		@returns {glow.anim.Anim}
			
		@example
			// stopping and clearing current animation queue.
			// The next animation created via queueAnim will start
			// immediately
			glow('#elementToAnimate').curentAnim().stop();
		
		@example
			// Is the element animating as part of queueAnim?
			glow('#elementToAnimate').curentAnim().playing; // true/false
	*/
	NodeListProto.currentAnim = function() {
		/*!debug*/
			if (arguments.length !== 0) {
				glow.debug.warn('[wrong count] glow.NodeList#currentAnim expects 0 arguments, not ' + arguments.length + '.');
			}
		/*gubed!*/
		return this.data('glow_currentAnim') || new glow.anim.Anim(0);
	}
	
	/**
		@name glow.NodeList#lastQueuedAnim
		@function
		@description Get the last animation added via {@link glow.NodeList#queueAnim queueAnim} for this element
			If no animation has been added, an empty animation is returned.
			This means you don't need to check to see if the item is defined before
			calling methods on it.
			
			This method acts on the first item in the NodeList.
		
		@returns {glow.anim.Anim}
	*/
	NodeListProto.lastQueuedAnim = function() {
		/*!debug*/
			if (arguments.length !== 0) {
				glow.debug.warn('[wrong count] glow.NodeList#lastQueuedAnim expects 0 arguments, not ' + arguments.length + '.');
			}
		/*gubed!*/
		return this.data('glow_lastQueuedAnim') || new glow.anim.Anim(0);
	}
	
	/**
		@private
		@function
		@description This function generates the various anim shortcut functions
	*/
	function animShortcut(animName, animReverseName, animPropsFunc, defaultTween, onComplete, additionalFunc) {
		return function(duration, opts) {
			/*!debug*/
				if (arguments.length > 2) {
					glow.debug.warn('[wrong count] glow.NodeList animation shortcuts expect 0, 1 or 2 arguments, not ' + arguments.length + '.');
				}
				if (duration !== undefined && typeof duration !== 'number') {
					glow.debug.warn('[wrong type] glow.NodeList animation shortcuts expect number as "duration" argument, not ' + typeof duration + '.');
				}
				if (opts !== undefined && typeof opts !== 'object') {
					glow.debug.warn('[wrong type] glow.NodeList animation shortcuts expect object as "opts" argument, not ' + typeof opts + '.');
				}
			/*gubed!*/
			
			opts = opts || {};
			
			var item,
				reverseAnim,
				currentAnim,
				calcDuration,
				anim,
				i = this.length;
				
			opts.tween = opts.tween || defaultTween;
			
			if (duration === undefined) {
				duration = 1;
			}
			
			calcDuration = duration;
			
			while (i--) {
				item = this.item(i);
				currentAnim = item.data('glow_' + animName);
				// if this isn't an element ,or we're already animating it, skip
				if ( item[0].nodeType !== 1 || (currentAnim && currentAnim.playing) ) { continue; }
				
				// if there's a reverse anim happening & it's playing, get rid
				reverseAnim = item.data('glow_' + animReverseName);
				if (reverseAnim && reverseAnim.playing) {
					// reduce the duration if we're not fading out as much
					calcDuration = duration * (reverseAnim.position / reverseAnim.duration);
					
					reverseAnim.stop().destroy();
				}
				
				item.data('glow_' + animName,
					anim = item.anim( calcDuration, animPropsFunc(item), opts ).on('complete', onComplete, item)
				);
				
				additionalFunc && additionalFunc(anim, item, opts);
			}
			
			return this;
		}
	};
	
	/**
		@name glow.NodeList#fadeIn
		@function
		@description Fade elements in
			If the element is currently fading out, the fadeOut animation will be automatically stopped.
		
		@param {number} [duration=1] Duration in seconds
		@param {Object} [opts] Options object
		@param {function|string} [opts.tween='easeOut'] The motion of the animation.
			Strings are treated as properties of {@link glow.tweens}, although
			a tween function can be provided.
			
		@returns {glow.NodeList}
		
		@example
			// make a tooltip fade in & out
			var tooltip = glow('#emailTooltip');
			
			glow('#emailInput').on('focus', function() {
				tooltip.fadeIn();
			}).on('blur', function() {
				tooltip.fadeOut();
			});
	*/
	NodeListProto.fadeIn = animShortcut('fadeIn', 'fadeOut', function(item) {
		item.css('display', 'block');
		return {opacity: 1};
	}, 'easeOut', function() {
		// on complete
		// we remove the filter from IE to bring back cleartype
		if (glow.env.ie) {
			this[0].style.filter = '';
		}
	});
	
	/**
		@name glow.NodeList#fadeOut
		@function
		@description Fade elements out
			If the element is currently fading in, the fadeIn animation will be automatically stopped.
		
		@param {number} [duration=1] Duration in seconds
		@param {Object} [opts] Options object
		@param {function|string} [opts.tween='easeIn'] The motion of the animation.
			Strings are treated as properties of {@link glow.tweens}, although
			a tween function can be provided.
			
		@returns {glow.NodeList}
		
		@example
			// make a tooltip fade in & out
			var tooltip = glow('#emailTooltip');
			
			glow('#emailInput').on('focus', function() {
				tooltip.fadeIn();
			}).on('blur', function() {
				tooltip.fadeOut();
			});
	*/
	NodeListProto.fadeOut = animShortcut('fadeOut', 'fadeIn', function() {
		return {opacity:0}
	}, 'easeIn', function() {
		this.css('display', 'none');
	});
	
	/**
		@name glow.NodeList#fadeToggle
		@function
		@description Fade elements in/out
			If the element is currently fading in/out, the fadeIn/fadeOut animation
			will be automatically stopped.
			
			// Implementation note: (delete me later)
			If the element has an opactity of 0, then fade in, otherwise fade out.
			UNLESS there's fadeOut animation currently happening on this element,
			then fade in.
			
		@param {number} [duration=1] Duration in seconds
		@param {Object} [opts] Options object
		@param {function|string} [opts.tween] The motion of the animation.
			Strings are treated as properties of {@link glow.tweens}, although
			a tween function can be provided.
			
			By default, 'easeIn' is used for fading out, and 'easeOut' is
			used for fading in.
			
		@returns {glow.NodeList}
		
		@example
			// make a tooltip fade in & out
			var tooltip = glow('#emailTooltip');
			
			glow('#toggleTooltip').on('click', function() {
				tooltip.fadeToggle();
			});
	*/
	NodeListProto.fadeToggle = function(duration, opts) {
		var i = this.length,
			item,
			fadeOutAnim;
		
		while (i--) {
			item = this.item(i);
			if (item[0].nodeType === 1) {
				// if the element has an opacity of 0, or is currently fading out
				if ( item.css('opacity') === '0' || ((fadeOutAnim = item.data('glow_fadeOut')) && fadeOutAnim.playing) ) {
					item.fadeIn(duration, opts);
				}
				else {
					item.fadeOut(duration, opts);
				}
			}
		}
		
		return this;
	};
	
	/**
		@name glow.NodeList#slideOpen
		@function
		@description Slide elements open
			This animates an element's height from its current height to its
			full auto-height size.
			
			If the element is currently sliding shut, the slideShut animation
			will be automatically stopped.
		
		@param {number} [duration=1] Duration in seconds
		@param {Object} [opts] Options object
			@param {function|string} [opts.tween='easeBoth'] The motion of the animation.
				Strings are treated as properties of {@link glow.tweens}, although
				a tween function can be provided.
			@param {boolean} [opts.lockToBottom=false] Lock the bottom of the content to the bottom of the element.
				This means the bottom of the content is shown first, rather than the top.
			
		@returns {glow.NodeList}
		
		@example
			var menuContent = glow('#menu div.content');
			
			glow('#menu').on('mouseenter', function() {
				menuContent.slideOpen();
			}).on('mouseleave', function() {
				menuContent.slideShut();
			});
		
		@example
			glow('#furtherInfoHeading').on('click', function() {
				glow('#furtherInfoContent').slideOpen();
			});
			
		@example
			// add content onto an element, and slide to reveal the new content
			glow('<div>' + newContent + '</div>').appendTo('#content').height(0).slideOpen();
			
	*/
	NodeListProto.slideOpen = animShortcut('slideOpen', 'slideShut', function(item) {		
		var currentHeight = item.css('height'),
			fullHeight;
		
		if ( item.css('overflow') === 'visible' ) {
			item.css('overflow', 'hidden');
		}
		
		item.css('height', 'auto');
		fullHeight = item.height();
		item.css('height', currentHeight);
		return {height: fullHeight}
	}, 'easeBoth', function() {
		this.css('height', 'auto').scrollTop(0);
	}, lockToBottom);
	
	/**
		@name glow.NodeList#slideShut
		@function
		@description Slide elements shut
			This animates an element's height from its current height to zero.
			
			If the element is currently sliding open, the slideOpen animation
			will be automatically stopped.
		
		@param {number} [duration=1] Duration in seconds
		@param {Object} [opts] Options object
			@param {function|string} [opts.tween='easeBoth'] The motion of the animation.
				Strings are treated as properties of {@link glow.tweens}, although
				a tween function can be provided.
			@param {boolean} [opts.lockToBottom=false] Lock the bottom of the content to the bottom of the element.
				This means the top of the content is hidden first, rather than the bottom.
			
		@returns {glow.NodeList}
		
		@example
			var menuContent = glow('#menu div.content');
			
			glow('#menu').on('mouseenter', function() {
				menuContent.slideOpen();
			}).on('mouseleave', function() {
				menuContent.slideShut();
			});
	*/
	NodeListProto.slideShut = animShortcut('slideShut', 'slideOpen', function(item) {
		if ( item.css('overflow') === 'visible' ) {
			item.css('overflow', 'hidden');
		}
		return {height: 0}
	}, 'easeBoth', function() {}, lockToBottom);
	
	/**
		@private
		@function
		@description Add frame listener to lock content to the bottom of an item.
		@param {glow.anim.Anim} anim Anim to alter
		@param {glow.NodeList} element Element being animated
		@param {Object} opts Options from slide[Open|Shut|Toggle]
	*/
	function lockToBottom(anim, element, opts) {
		var node = element[0],
			scrollHeight = node.scrollHeight;
		
		if (opts.lockToBottom) {
			anim.on('frame', function() {
				element.scrollTop( scrollHeight - node.offsetHeight );
			});
		}
	}
	
	/**
		@name glow.NodeList#slideToggle
		@function
		@description Slide elements open/shut
			If the element is currently sliding open/shut, the slideOpen/slideShut animation
			will be automatically stopped.
			
			// Implementation note: (delete me later)
			If the element has a height of 0, then slide open, otherwise slide shut.
			UNLESS there's slideShut animation currently happening on this element,
			then slide open.
		
		@param {number} [duration=1] Duration in seconds
		@param {Object} [opts] Options object
			@param {function|string} [opts.tween='easeBoth'] The motion of the animation.
				Strings are treated as properties of {@link glow.tweens}, although
				a tween function can be provided.
			@param {boolean} [opts.lockToBottom=false] Lock the bottom of the content to the bottom of the element.
				This means the top of the content is hidden first & shown last.
			
		@returns {glow.NodeList}
		
		@example
			var menuContent = glow('#menuContent');
			
			glow('#toggleMenu').on('click', function() {
				menuContent.slideToggle();
			});
	*/
	NodeListProto.slideToggle = function(duration, opts) {
		var i = this.length,
			item,
			slideShutAnim;
		
		while (i--) {
			item = this.item(i);
			if (item[0].nodeType === 1) {
				// if the element has an height of 0, or is currently sliding shut
				if ( item.height() === 0 || ((slideShutAnim = item.data('glow_slideShut')) && slideShutAnim.playing) ) {
					item.slideOpen(duration, opts);
				}
				else {
					item.slideShut(duration, opts);
				}
			}
		}
		
		return this;
	};
});
/**
	@name glow.net
	@namespace
	@description Methods for getting data & resources from other locations. Sometimes referred to as AJAX.
*/
Glow.provide(function(glow) {
	var net = {},
		undefined,
		emptyFunc = function(){};
	
	/**
		@private
		@function
		@description Create XhrRequest factory methods
		
		@param {string} method HTTP method
		@returns {function} Factory method
	*/
	function createXhrFactory(method) {
		return function(url, data, opts) {
			// only put & post use the data param
			if (method === 'POST' || method === 'PUT') {
				opts = opts || {};
				opts.data = data;
			}
			else {
				opts = data;
			}
			
			return new net.XhrRequest(method, url, opts);
		}
	}
	
	/**
		@name glow.net.get
		@function
		@description Makes an HTTP GET request to a given url.
			This is a shortcut to creating an instance of {@link glow.net.XhrRequest}.
	 
		@param {string} url Url to make the request to.
			This can be a relative path. You cannot make requests for files on
			other domains (including sub-domains). For cross-domain requests, see
			{@link glow.dom.getJsonp} and {@link glow.dom.crossDomainGet}.
		@param {Object} [opts] Options.
			These options are the same as the constructor options for {@link glow.net.XhrRequest}.
	 
		@returns {glow.net.XhrRequest}
	 
		@example
			glow.net.get('myFile.html').on('load', function(response){
				alert( 'Got file:' + response.text() );
			}).on('error', function(response){
				alert( 'Something went wrong:' + response.text() );
			});
			
	*/
	net.get = createXhrFactory('GET');
	
	/**
		@name glow.net.post
		@function
		@description Makes an HTTP POST request to a given url
			This is a shortcut to creating an instance of {@link glow.net.XhrRequest}.
		 
		@param {string} url Url to make the request to.
			This can be a relative path. You cannot make requests for files on
			other domains (including sub-domains). For cross-domain requests, see
			{@link glow.dom.getJsonp} and {@link glow.dom.crossDomainGet}.
		@param {Object|String} data Data to send.
			This can be either a JSON-style object or a urlEncoded string.
		@param {Object} [opts] Options.
			These options are the same as the constructor options for {@link glow.net.XhrRequest}.
		
		@returns {glow.net.XhrRequest}
	 
		@example
			glow.net.post('myFile.html', {
				key: 'value',
				otherkey: ['value1', 'value2']
			}).on('load', function(response) {
				alert( 'Got file:' + response.text() );
			});
	*/
	net.post = createXhrFactory('POST');
	
	/**
		@name glow.net.put
		@function
		@description Makes an HTTP PUT request to a given url
			This is a shortcut to creating an instance of {@link glow.net.XhrRequest}.
		 
		@param {string} url Url to make the request to.
			This can be a relative path. You cannot make requests for files on
			other domains (including sub-domains). For cross-domain requests, see
			{@link glow.dom.getJsonp} and {@link glow.dom.crossDomainGet}.
		@param {Object|String} data Data to send.
			This can be either a JSON-style object or a urlEncoded string.
		@param {Object} [opts] Options.
			These options are the same as the constructor options for {@link glow.net.XhrRequest}.
 
		@returns {glow.net.XhrRequest}
 
		@example
			glow.net.put('myFile.html', {
				key: 'value',
				otherkey: ['value1', 'value2']
			}).on('load', function(response) {
				// handle response
			});
	*/
	net.put = createXhrFactory('PUT');
	
	/**
		@name glow.net.del
		@function
		@description Makes an HTTP DELETE request to a given url.
			This is a shortcut to creating an instance of {@link glow.net.XhrRequest}.
		 
		@param {string} url Url to make the request to.
			This can be a relative path. You cannot make requests for files on
			other domains (including sub-domains). For cross-domain requests, see
			{@link glow.dom.getJsonp} and {@link glow.dom.crossDomainGet}.
		@param {Object} [opts] Options.
			These options are the same as the constructor options for {@link glow.net.XhrRequest}.
 
		@returns {glow.net.XhrRequest}
 
		@example
			glow.net.del('myFile.html').on('load', function(response) {
				// handle response
			});
	*/
	
	net.del = createXhrFactory('DELETE');		
		
	// export
	glow.net = net;
});
Glow.provide(function(glow) {
	var undefined,
		XhrRequestProto,
		events = glow.events,
		removeAllListeners = events.removeAllListeners;
	
	/**
		@private
		@function
		@description Creates an XMLHttpRequest transport
		@returns XMLHttpRequest
	*/
	var xmlHTTPRequest = window.ActiveXObject ?
		function() {
			return new ActiveXObject('Microsoft.XMLHTTP');
		} :
		function() {
			return new XMLHttpRequest();
		};
		
	/**
		@private
		@function
		@description Apply option object defaults.
		
		@param {object} opts Options object to apply defaults to.
		@param {string} method HTTP method.
		
		@returns {object} New opts object with defaults applied.
	*/
	function applyOptsDefaults(opts, method) {
		opts = glow.util.apply({
			headers: {}
		}, opts);
		
		var headers = opts.headers;
		
		// convert data to string
		if (typeof opts.data === 'object') {
			opts.data = glow.util.encodeUrl(opts.data);
		}
		
		// add requested with header if one hasn't been added
		if ( !headers['X-Requested-With'] ) {
			headers['X-Requested-With'] = 'XMLHttpRequest';
		}
		
		if (method !== 'GET' && !headers["Content-Type"]) {
			headers["Content-Type"] = 'application/x-www-form-urlencoded;';
		}
		
		return opts;
	}
	
	/**
		@name glow.net.XhrRequest
		@class
		@param {string} method The HTTP method to use for the request.
			Methods are case sensitive in some browsers.
		@param {string} url Url to make the request to.
			This can be a relative path. You cannot make requests for files on
			other domains (including sub-domains). For cross-domain requests, see
			{@link glow.dom.getJsonp} and {@link glow.dom.crossDomainGet}.
		@param {Object} [opts] Options object
			@param {Object} [opts.headers] A hash of headers to send along with the request.
				eg `{'Accept-Language': 'en-gb'}`
			@param {boolean} [opts.cacheBust=false] Prevent the browser returning a cached response.
				If true, a value is added to the query string to ensure a fresh version of the
	 			file is being fetched.
			@param {number} [opts.timeout] Time to allow for the request in seconds.
				No timeout is set by default. Once the time is reached, the error
				event will fire with a '408' status code.
			@param {boolean} [opts.forceXml=false] Treat the response as XML.
				This will allow you to use {@link glow.net.XhrResponse#xml response.xml()}
				even if the response has a non-XML mime type.
			@param {Object|string} [opts.data] Data to send.
				This can be either a JSON-style object or a urlEncoded string.
				
		@description Create an XHR request.
			Most common requests can be made using shortcuts methods in {@link glow.net},
			such as {@link glow.net.get}.
		
		@example
			new glow.net.XhrRequest('DELETE', 'whatever.php', {
				timeout: 10
			}).on('load', function(response) {
				alert( response.text() );
			});
	*/
	function XhrRequest(method, url, opts) {
		this._opts = opts = applyOptsDefaults(opts, method);
		
		var request = this,
			nativeRequest = request.nativeRequest = xmlHTTPRequest(), //request object
			i;

		// add the cacheBust to the url
		if (opts.cacheBust) {
			url = url + (url.indexOf('?') === -1 ? '?' : '&') + 'cachebuster=' + new Date().valueOf();
		}

		request.complete = false;
		
		//open needs to go first to maintain cross-browser support for readystates
		nativeRequest.open(method, url, true);
 
		//add custom headers
		for (i in opts.headers) {
			nativeRequest.setRequestHeader( i, opts.headers[i] );
		}
		
		// force the reponse to be treated as xml
		// IE doesn't support overrideMineType, we need to deal with that in {@link glow.net.XhrResponse#xml}
		if (opts.forceXml && nativeRequest.overrideMimeType) {
			nativeRequest.overrideMimeType('application/xml');
		}
		
		//sort out the timeout if there is one
		if (opts.timeout) {			 
			request._timeout = setTimeout(function() {
				var response = new glow.net.XhrResponse(request, true);
				request.abort().fire('error', response);
			}, opts.timeout * 1000);
		}
		
		nativeRequest.onreadystatechange = function() {
			if (nativeRequest.readyState === 4) {
				var response = new glow.net.XhrResponse(request);
				
				//clear the timeout
				clearTimeout(request._timeout);
				
				//set as completed
				request.completed = true;
				request.fire(response.successful ? 'load' : 'error', response);
				
				// prevent parent scopes leaking (cross-page) in IE
				nativeRequest.onreadystatechange = new Function();
				removeAllListeners(request);
			}
		};
		
		// make sure it doesn't complete before listeners are attached
		setTimeout(function() {
			nativeRequest.send(opts.data || null);
		}, 0);
	}
	glow.util.extend(XhrRequest, events.Target);
	XhrRequestProto = XhrRequest.prototype;
	
	/**
		@name glow.net.XhrRequest#_timeout
		@private
		@description setTimeout ID
		@type number
	*/
	
	/**
		@name glow.net.XhrRequest#complete
		@description Boolean indicating whether the request has completed
		@example
			// request.complete with an asynchronous call
			var request = glow.net.get(
				"myFile.html").on('load', 
				function(response){
					alert(request.complete); // returns true
				})
				
					
		@type boolean
	*/
	
	/**
		@name glow.net.XhrRequest#nativeRequest
		@description The request object from the browser.
			This may not have the same properties and methods across user agents.
			Also, this will be undefined if the request originated from getJsonp.
			
		@type Object
	*/
	
	/**
		@name glow.net.XhrRequest#abort
		@function
		@description Aborts a request
			The load & error events will not fire.
		@example
			var request = glow.net.get('myFile.html').on('load', function(response) {
				//handle response
			}).on('abort', function() {
				alert('Something bad happened. The request was aborted.');
			});
			
			request.abort(); // alerts "Something bad happened.  The request was aborted"
		@returns this
	*/
	XhrRequestProto.abort = function() {
		if ( !this.completed && !this.fire('abort').defaultPrevented() ) {
			clearTimeout(this._timeout);
			this.nativeRequest.onreadystatechange = new Function();
			removeAllListeners(this);
		}
		return this;
	};
		 
	/**
		@name glow.net.XhrRequest#event:load
		@event
		@param {glow.net.XhrResponse} response
		@description Fired when the request is sucessful
			This will be fired when request returns with an HTTP code of 2xx. 
	*/
 
	/**
		@name glow.net.XhrRequest#event:abort
		@event
		@param {glow.events.Event} event Event Object
		@description Fired when the request is aborted
			If you cancel the default (eg, by returning false) the request
			will continue.
	*/
 
	/**
		@name glow.net.XhrRequest#event:error
		@event
		@param {glow.net.XhrResponse} response
		@description Fired when the request is unsucessful
			This will be fired when request returns with an HTTP code which
			isn't 2xx or the request times out.
	*/
	
	glow.net.XhrRequest = XhrRequest;
});
Glow.provide(function(glow) {
	var XhrResponseProto,
		util = glow.util;
	
	/**
		@name glow.net.XhrResponse
		@class
		@extends glow.events.Event
		@description The event object for {@link glow.net.XhrRequest}'s 'load' & 'error' events.
		@glowPrivateConstructor There is no direct constructor.
	*/
	
	/*
		These params are hidden as we don't want users to try and create instances of this...
	 
		@param {glow.net.XhrRequest} [request] Original request object
		@param {Boolean} [timedOut=false] Set to true if the response timed out
		
	*/
	function XhrResponse(request, timedOut) {
		var nativeResponse = this.nativeResponse = request.nativeRequest;
		this._request = request;
		
		//IE reports status as 1223 rather than 204, for laffs
		this.status = timedOut ? 408 :
			nativeResponse.status == 1223 ? 204 : nativeResponse.status;

		this.timedOut = !!timedOut;
			
		this.successful = (this.status >= 200 && this.status < 300) ||
			//from cache
			this.status == 304 ||
			//watch our for requests from file://
			(this.status == 0 && nativeResponse.responseText);
	}

	util.extend(XhrResponse, glow.events.Event);
	XhrResponseProto = XhrResponse.prototype;
	
	/**
		@name glow.net.XhrResponse#_request
		@private
		@description Original request object
		@type glow.net.XhrRequest
	*/
	
	/**
		@name glow.net.XhrResponse#nativeResponse
		@description The response object from the browser.
			This may not have the same properties and methods across user agents.
		@type XMLHttpRequest
	*/
	
	/**
		@name glow.net.XhrResponse#status
		@description HTTP status code of the response
		@type number
	*/

	/**
		@name glow.net.XhrResponse#timedOut
		@description Boolean indicating if the requests time out was reached.
		@type boolean
	*/
	
	/**
		@name glow.net.XhrResponse#successful
		@description  Boolean indicating if the request returned successfully.
		@type boolean
	*/
	
	/**
		@name glow.net.XhrResponse#text
		@function
		@description Gets the body of the response as plain text
		@returns {string}
	*/

	XhrResponseProto.text = function() {
		return this.nativeResponse.responseText;
	};
	
	/**
		@name glow.net.XhrResponse#xml
		@function
		@description Gets the body of the response as xml
		@returns {XML} 
	*/
	
	XhrResponseProto.xml = function() {
		var nativeResponse = this.nativeResponse,
			contentType = this.header("Content-Type");
		
		if (
			// IE 6 & 7 fail to recognise Content-Types ending +xml (eg application/rss+xml)
			// Files from the filesystem don't have a content type, but could be xml files, parse them to be safe
			glow.env.ie && (
				contentType.slice(-4) === '+xml' ||
				contentType === '' ||
				this._request._opts.forceXml
			)
		) {
			var doc = new ActiveXObject("Microsoft.XMLDOM");
			doc.loadXML( nativeResponse.responseText );
			return doc;
		}
		else {
			return nativeResponse.responseXML;
		}				
	};
	
	/**
		@name glow.net.XhrResponse#json
		@function
		@description Gets the body of the response as a JSON object.
		
		@param {boolean} [safeMode=false]
			If true, the response will be parsed using a string parser which
			will filter out non-JSON javascript, this will be slower but
			recommended if you do not trust the data source.
	
		@returns {object}
	*/
	XhrResponseProto.json = function(safe) {
		return util.decodeJson(this.text(), {safeMode:safe});
	};
	
	/**
		@name glow.net.XhrResponse#nodeList
		@function
		@description Gets the body of the response as a {@link glow.NodeList}.
	
		@returns {glow.NodeList}
	*/
	XhrResponseProto.nodeList = function(safe) {
		return glow(
			glow.NodeList._strToNodes( this.text() )
		);
	};

	/**
		@name glow.net.XhrResponse#header
		@function
		@description Gets a header from the response.
	
		@param {string} name Header name
		@returns {string} Header value
	
		@example var contentType = myResponse.header("Content-Type");
	*/
	
	XhrResponseProto.header = function(name) {
		return this.nativeResponse.getResponseHeader(name);
	};

	/**
		@name glow.net.XhrResponse#statusText
		@function
		@description Gets the meaning of {@link glow.net.XhrResponse#status status}.
	
		@returns {string}
	*/
	XhrResponseProto.statusText = function() {
		return this.timedOut ? "Request Timeout" : this.nativeResponse.statusText;
	};
	
	glow.net.XhrResponse = XhrResponse;
});
Glow.provide(function(glow) {
	var undefined,
		JsonpRequestProto,
		net = glow.net,
		emptyFunc = function(){},
		events = glow.events,
		// Script elements that have been added via {@link glow.net.jsonp jsonp}, keyed by callback name
		scriptElements = {},
		scriptElementsLen = 0,
		callbackPrefix = 'c',
		// Name of the global object used to store jsonp callbacks
		globalObjectName = '_' + glow.UID + 'jsonp',
		head = glow('head'),
		// a reference to the global object holding the callbacks
		globalObject;
	
	/**
		@private
		@function
		@description Handle jsonp load.
		@param {glow.net.JsonpRequest} request
		@param {Object[]} args Arguments object passed to the callback from the jsonp source
	*/
	function jsonpLoad(request, args) {
		// we have to call listeners manually as we don't provide a real event object. A bit of a hack.
		var loadListeners = events._getListeners(request).load,
			i;
			
		if (loadListeners) {
			loadListeners = loadListeners.slice(0);
			i = loadListeners.length;
			while (i--) {
				loadListeners[i][0].apply( loadListeners[i][1], args );
			}
		}
		//set as completed
		request.completed = true;

		cleanUp(request);
	}
	
	/**
		@private
		@function
		@description Clean up to avoid memory leaks
		@param {glow.net.JsonpRequest} request
		@param {boolean} [leaveEmptyFunc] Replace global callback with blank function.
			If false, the global callback will be set to undefined, which is better for memory,
			but in some cases the callback may later be called (like a timed out request) so an
			empty function needs to be used to avoid errors.
	*/
	function cleanUp(request, leaveEmptyFunc) {
		var callbackName = request._callbackName;
		
		clearTimeout(request._timeout);
		globalObject[callbackName] = leaveEmptyFunc ? emptyFunc : undefined;
		glow( scriptElements[callbackName] ).destroy();
		scriptElements[callbackName] = undefined;
	}
	
	/**
		@name glow.net.JsonpRequest
		@class
		@description A JSONP request.
			Although instance of this can be created manually, using
			{@link glow.net.jsonp} is preferred. 
	*/
	// the params for this are the same as {@link glow.net.jsonp}.
	function JsonpRequest(url, opts) {
		opts = opts || {};
		
		var newIndex = scriptElements.length,
			//script element that gets inserted on the page
			//generated name of the callback
			callbackName = this._callbackName = callbackPrefix + (scriptElementsLen++),
			// script element to add to the page
			script = scriptElements[callbackName] = document.createElement('script'),
			request = this,
			timeout = opts.timeout,
			charset = opts.charset;
		
		// add the callback name to the url
		url = glow.util.interpolate(url, {
			callback: globalObjectName + '.' + callbackName
		});
		
		// create the global object if it doesn't exist already
		globalObject || ( globalObject = window[globalObjectName] = {} );
		
		// create our callback
		globalObject[callbackName] = function() {
			jsonpLoad(request, arguments);
		};
		
		// set charset
		charset && (script.charset = charset);
		
		if (opts.timeout) {
			request._timeout = setTimeout(function() {
				request.abort().fire('error');
			}, timeout * 1000);
		}
			
		script.src = url;
		
		//add script to page
		head.prepend(script);
		
		script = undefined;
	}
	
	glow.util.extend(JsonpRequest, events.Target);
	JsonpRequestProto = JsonpRequest.prototype;
	
	/**
		@name glow.net.JsonpRequest#_callbackName
		@private
		@description The name of the callback, used as a property name in globalObject and scriptElements
	*/
	
	/**
		@name glow.net.JsonpRequest#_timeout
		@private
		@description timeout ID
		@type number
	*/
	
	/**
		@name glow.net.JsonpRequest#complete
		@description Boolean indicating whether the request has completed
		@type boolean
	*/
	JsonpRequestProto.complete = false;
	
	/**
		@name glow.net.JsonpRequest#abort
		@function
		@description Abort the request.
			The script file may still load, but the 'load' event will not fire.
		@returns this
	*/
	JsonpRequestProto.abort = function() {
		this.fire('abort');
		cleanUp(this, true);
		return this;
	};
	
	/**
		@name glow.net.JsonpRequest#event:load
		@event
		@description Fired when the request is sucessful.
			The parameters to this event are whatever the datasource provides.
			
		@example
			glow.net.jsonp('http://twitter.com/statuses/user_timeline/15390783.json?callback={callback}')
				.on('load', function(data) {
					alert(data);
				});
	*/
 
	/**
		@name glow.net.JsonpRequest#event:abort
		@event
		@param {glow.events.Event} event Event Object
		@description Fired when the request is aborted.
	*/
 
	/**
		@name glow.net.JsonpRequest#event:error
		@event
		@param {glow.events.Event} event Event Object
		@description Fired when the request times out.
	*/
	
	/**
		@name glow.net.jsonp
		@function
		@description Fetch JSON via JSONP.
			This can be used cross domain, but should only be used with trusted
			sources as any javascript included in the script will be executed.
			
			This method only works if the server allows you to specify a callback
			name for JSON data. Not all JSON sources support this, check the API of the
			data source to ensure you're using the correct querystring parameter
			to set the callback name.
	 
		@param {string} url Url of the script.
			Set the callback name via the querystring to `{callback}`, Glow will
			replace this with another value and manage the callback internally.
			
			Check the API of your data source for the correct parameter name.
			Eg, in Flickr it's `jsoncallback={callback}`, in Twitter it's
			`callback={callback}`.
		@param {object} [opts]
			@param {number} [opts.timeout] Time to allow for the request in seconds.
			@param {string} [opts.charset] Charset attribute value for the script.
	 
		@returns {glow.net.JsonpRequest}
	 
		@example
			glow.net.jsonp('http://twitter.com/statuses/user_timeline/15390783.json?callback={callback}', {
				timeout: 5
			}).on('load', function(data) {
				alert(data);
			}).on('error', function() {
				alert('Request timeout');
			});
	*/
	net.jsonp = function(url, opts) {
		return new glow.net.JsonpRequest(url, opts);
	};
	
	glow.net.JsonpRequest = JsonpRequest;
});
Glow.provide(function(glow) {
	var undefined,
		ResourceRequestProto,
		ResourceResponseProto,
		net = glow.net;
	
	/**
		@private
		@function
		@description Normalise urls param.
			Normalise ResourceRequest's urls parameter to an object with 'css', 'js' and 'img' properties.
	*/
	function normaliseUrlsParam(urls) {
		var r = {
				js: [],
				css: [],
				img: []
			},
			url;
		
		if (typeof urls === 'object' && !urls.push) {
			r = glow.util.apply(r, urls);
		}
		else {
			// convert urls to an array if need be
			typeof urls === 'string' && ( urls = [urls] );
			
			// forwards loop, maintain order
			for (var i = 0, len = urls.length; i < len; i++) {
				url = urls[i];
				if ( url.slice(-4) === '.css' ) {
					r.css[r.css.length] = url;
				}
				else if ( url.slice(-3) === '.js' ) {
					r.js[r.js.length] = url;
				}
				else {
					r.img[r.img.length] = url;
				}
			}
		}
		
		return r;
	}
	
	/**
		@name glow.net.ResourceRequest
		@class
		@description Request made via {@link glow.net.getResources}
		@glowPrivateConstructor There is no direct constructor.
	*/
	function ResourceRequest(urls) {
		urls = normaliseUrlsParam(urls);
		
		var request = this,
			js = urls.js,
			css = urls.css,
			img = urls.img,
			jsLen = js.length,
			cssLen = css.length,
			imgLen = img.length,
			i;
		
		request.totalResources = jsLen + cssLen + imgLen;
		
		// ensure events don't fire until they're added
		setTimeout(function() {
			// guess it makes sense to load CSS, js then images (the browser will queue the requests)
			for (i = 0; i < cssLen; i++) {
				loadCss( request, css[i] );
			}
			for (i = 0; i < jsLen; i++) {
				loadJs( request, js[i] );
			}
			for (i = 0; i < imgLen; i++) {
				loadImg( request, img[i] );
			}
		}, 0);
		
	}
	
	glow.util.extend(ResourceRequest, glow.events.Target);
	ResourceRequestProto = ResourceRequest.prototype;
	
	/**
		@name glow.net.ResourceRequest#totalResources
		@type number
		@description Total number of resources requested.
	*/
	ResourceRequestProto.totalResources = 0;
	
	/**
		@name glow.net.ResourceRequest#totalLoaded
		@type number
		@description Total number of resources successfully loaded.
	*/
	ResourceRequestProto.totalLoaded = 0;
	
	/**
		@private
		@function
		@description Update a request after a resource loads.
		
		@param {glow.net.ResourceRequest} request.
		@param {string} url Url of the requested resource.
		@param {glow.NodeList} resource The element used to load the resource.
		@param {string} type 'js', 'css' or 'img'
	*/
	function progress(request, url, resource, type) {
		
		var totalLoaded = ++request.totalLoaded;
		
		request.fire('progress', {
			resource: resource,
			url: url,
			type: type
		});
		
		if (totalLoaded === request.totalResources) {
			request.fire('load');
		}
	}
	
	/**
		@private
		@function
		@description Start loading an image
		
		@param {glow.net.ResourceRequest} request
		@param {string} imgUrl
	*/
	function loadImg(request, imgUrl) {
		var img = new Image;
		// keep the url in its original format
		glow(img).data('srcUrl', imgUrl).on('load', imgLoaded, request);
		img.src = imgUrl;
	}
	
	/**
		@private
		@function
		@description Process a loaded image.
			'this' is the ResourceRequest
	*/
	function imgLoaded(event) {
		var img = glow(event.attachedTo);
		progress( this, img.data('srcUrl'), img, 'img' );
	}
	
	/**
		@private
		@function
		@description Start loading a script
		
		@param {glow.net.ResourceRequest} request
		@param {string} scriptUrl
	*/
	function loadJs(request, scriptUrl){
		var script = glow( document.createElement('script') )
			.data('srcUrl', scriptUrl)
			.prependTo('head');
		
		// two methods, one for IE (readystatechange) and the other for others
		script.on('readystatechange', jsLoaded, request).on('load', jsLoaded, request);
		
		script[0].src = scriptUrl;
	}
	
	/**
		@private
		@function
		@description Process a loaded script.
			'this' is the ResourceRequest
	*/
	function jsLoaded(event) {
		var script = glow(event.attachedTo),
			scriptElm = script[0],
			readyState = scriptElm.readyState;
		
		if ( !readyState || readyState === 'loaded' || readyState === 'complete' ) {
			// remove events to prevent double-firing
			script.detach('readystatechange', jsLoaded).detach('load', jsLoaded);
			progress( this, script.data('srcUrl'), script, 'js' );
		}
	}
	
	/**
		@private
		@function
		@description Start loading a CSS file
		
		@param {glow.net.ResourceRequest} request
		@param {string} cssUrl
	*/
	// This technique was found in http://code.google.com/p/ajaxsoft/source/browse/trunk/xLazyLoader
	function loadCss(request, cssUrl){
		var currentHostname,
			urlHostname,
			link = glow('<link rel="stylesheet" type="text/css" media="all" href="' + cssUrl + '" />').data('srcUrl', cssUrl);
		
		// we have to do something special for Gecko browsers when the css is from another domain
		if ( glow.env.gecko && /^(?:https?\:|\/\/)/.test(cssUrl) ) {
			currentHostname = location.hostname.replace('www.', '');
			urlHostname = cssUrl.replace(/https?:\/\/|www\.|:.*/g, '').replace(/\/.*/g, '');
			
			if (currentHostname !== urlHostname) {
				// ack, we have to cheat
				setTimeout(function() {
					cssLoaded.call(request, {
						attachedTo: link
					});
				}, 500);
			}
		}
		else {
			// two methods, one for IE (readystatechange), and one for opera
			link.on('readystatechange', cssLoaded, request).on('load', cssLoaded, request);
			// ...and one more for Moz & webkit
			(function pollCssRules() {
				try {
					link[0].sheet.cssRules;
					// we'll error before the next line if CSS hasn't loaded
					cssLoaded.call(request, {
						attachedTo: link
					});
				}
				catch (e) {
					if ( !link.data('loaded') ) {
						setTimeout(pollCssRules, 20);
					}
				};
			})();
		}
		
		//link[0].href = cssUrl;
		link.prependTo('head');
	}
	
	/**
		@private
		@function
		@description Process a loaded stylesheet.
			'this' is the ResourceRequest
	*/
	function cssLoaded(event) {
		var link = glow(event.attachedTo),
			linkElm = link[0],
			readyState = linkElm.readyState;
		
		if ( !readyState || readyState === 'loaded' || readyState === 'complete' ) {
			// just incase there's a timeout still waiting
			if ( link.data('loaded') ) {
				return;
			}
			link.data('loaded', true);
			
			// remove events to prevent double-firing
			link.detach('readystatechange', cssLoaded).detach('load', cssLoaded);
			progress( this, link.data('srcUrl'), link, 'css' );
		}
	}
	
	
	/**
		@name glow.net.ResourceRequest#event:load
		@event
		@param {glow.events.Event} event Event Object
		@description Fired when all the requested items have completed.
	*/
 
	/**
		@name glow.net.ResourceRequest#event:progress
		@event
		@description Fired when a single resource loads.
		
		@param {glow.events.Event} event Event Object
			@param {string} event.url Url of the loaded resource.
			@param {glow.NodeList} event.resource The element used to load the resource.
				This will be a `<script>`, `<link>`, or `<img>` element.
			@param {string} event.type 'js', 'css' or 'img'.
	*/
	
	/**
		@name glow.net.getResources
		@function
		@description Load scripts, images & CSS.
			Files can be loaded from other domains.
			
			Note: Due to a cross-browser restriction, 'load' may fire before
			CSS files from another domain are fully loaded in Gecko browsers.
	 
		@param {string[]|string|Object} url
			Url(s) to load. Urls ending in ".css" are assumed to be CSS files,
			Urls ending in ".js" are assumed to be JavaScript. All other files
			will be treated as images.
			
			You can provide an object in the form `{js: [], css: [], img: []}` to
			be explicit about how to treat each file.
			
		@returns {glow.net.ResourceRequest}
	 
		@example
			// load a single CSS file with a callback specified
			glow.net.getResources('/styles/custom.css').on('load', function() {
				// CSS has now loaded
			});
			
		@example
			// load a single CSS file with a callback specified
			glow.net.getResources([
				'/imgs/whatever.png',
				'/style/screen.css',
			]).on('load', function() {
				// CSS & image now loaded
			});
			
		@example
			// load multiple files by specifying and array
			glow.net.getResources({
				js: ['http://www.server.com/script', 'http://www.server.com/anotherScript'],
				img: ['http://www.server.com/product4/thumb']
			}).on('progress', function(event) {
				// update a progress meter
			}).on('load', function(response){
				// files now loaded
			});
	*/
	net.getResources = function(urls, opts) {
		/*!debug*/
			if (arguments.length < 1 && arguments.length > 2) {
				glow.debug.warn('[wrong count] glow.net.getResources expects 1 or 2 arguments, not ' + arguments.length + '.');
			}
		/*gubed!*/
		return new glow.net.ResourceRequest(urls, opts);
	};
	
	glow.net.ResourceRequest = ResourceRequest;	
});
Glow.provide(function(glow) {
	var undefined,
		CrossDomainRequestProto,
		CrossDomainResponseProto,
		net = glow.net,
		// We borrow some methods from XhrRequest later
		XhrResponseProto = net.XhrRequest.prototype,
		Target = glow.events.Target;

	/**
		@name glow.net.CrossDomainRequest
		@class
		@description Cross-domain request via window.name
			A request made via a form submission in a hidden iframe, with the
			result being communicated via the name attribute of the iframe's window.
			
			The URL that's requested should respond with a blank HTML page containing JavaScript
			that assigns the result to window.name as a string:
 
			`<script type="text/javascript">window.name = 'result string';</script>`
			
			Instances of this are returned by shortcut methods {@link glow.net.crossDomainGet}
			and {@link glow.net.crossDomainPost}
	
		@param {string} method The HTTP method to use for the request.
			Only 'POST' and 'GET' are considered cross-browser.
		@param {string} url The URL to request.
		@param {Object} [opts]
			@param {Object|string} [opts.data] Data to send.
				This can be either a JSON-style object or a urlEncoded string.
			@param {number} [opts.timeout] Time to allow for the request in seconds.
				No timeout is set by default.
			@param {string} [opts.blankUrl='/favicon.ico']
				The path of a page on same domain as the caller, ideally a page likely
				to be in the user's cache.
	*/
	function CrossDomainRequest(method, url, opts) {
		var request = this,
			timeout;
		
		request._opts = opts = glow.util.apply({
			data: {},
			blankUrl: '/favicon.ico'
		}, opts);
		
		// convert data to object
		if (typeof opts.data === 'string') {
			opts.data = glow.util.decodeUrl(opts.data);
		}
		
		// set timeout for the request
		timeout = opts.timeout;
		if (timeout) {
			request._timeout = setTimeout(function () {				
				request.fire('error');					
				cleanup(request);
			}, timeout * 1000);
		}
		
		addIframe(request);
		buildAndSubmitForm(request, method, url);
	}
	glow.util.extend(CrossDomainRequest, Target);
	CrossDomainRequestProto = CrossDomainRequest.prototype;
	
	/**
		@name glow.net.CrossDomainRequest#_opts
		@private
		@type Object
		@description Options object with defaults applied
	*/
	
	/**
		@name glow.net.CrossDomainRequest#_iframe
		@private
		@type glow.NodeList
		@description Iframe used to send the data.
	*/
	
	/**
		@name glow.net.CrossDomainRequest#_timeout
		@private
		@type number
		@description setTimeout id for request timeout
	*/

	/**
		@private
		@function
		@description Add a hidden iframe for posting the request
		@param {glow.net.CrossDomainRequest} request
	*/
	function addIframe(request) {
		var iframe = request._iframe = glow(
			'<iframe style="visibility: hidden; position: absolute; height: 0;"></iframe>'
		).appendTo(document.body);
	};
		
	/**
		@private
		@function
		@description Add a form to the iframe & submit it
		
		@param {glow.net.CrossDomainRequest} request
		@param {string} method The HTTP method to use for the request.
			Only 'POST' and 'GET' are considered cross-browser.
		@param {string} url The URL to request.
	*/
	function buildAndSubmitForm(request, method, url) {
		var iframe = request._iframe,
			win = iframe[0].contentWindow,
			doc = win.document,
			form,
			data = request._opts.data;

		// IE needs an empty document to be written to written to the iframe
		if (glow.env.ie) {
			doc.open();
			doc.write('<html><body></body></html>');
			doc.close();
		}
		
		// create form
		form = doc.createElement('form');
		form.action = url;
		form.method = method;

		doc.body.appendChild(form);
		
		// build form elements
		for (var i in data) {
			if ( !data.hasOwnProperty(i) ) { continue; }
			
			if (data[i] instanceof Array) {
				for (var j = 0, jLen = data[i].length; j < jLen; j++) {
					addHiddenInput( form, i, this.data[i][j] );
				}
			}
			else {
				addHiddenInput( form, i, this.data[i] );
			}
		}
		
		// submit - the setTimeout makes the function run in the context of the form
		win.setTimeout(function () {
			form.submit();
		}, 0);
		
		// listen for form submitting
		iframe.on('load', handleResponse, request);
	}

	/**
		@private
		@function
		@description Add a hidden input to a form for a piece of data.
		
		@param {HTMLFormElement} form
		@param {string} name Input name
		@param {string} value Input value
	*/
	function addHiddenInput(form, name, value) {
		var input = form.ownerDocument.createElement('input');
		input.type = 'hidden';
		input.name = name;
		input.value = value;
		form.appendChild(input);
	}

	/**
		@private
		@function
		@description Callback for load event in the hidden iframe.
			`this` is the request.
	*/
	function handleResponse() {		
		var err,
			href,
			win = this._iframe[0].contentWindow;
		
		try {
			href = win.location.href;
		}
		catch (e) {
			err = e;
		}
		
		if (href !== 'about:blank' || err) {
			clearTimeout(this._timeout);
			this._iframe.detach('load', handleResponse).on('load', readHandler, this);
			
			win.location = window.location.protocol + '//' + window.location.host + this._opts.blankUrl;
		}
	}

	/**
		@private
		@function
		@description Callback for load event of blank page in same origin.
			`this` is the request.
	*/
	function readHandler() {
		this.fire( 'load', new CrossDomainResponse(this._iframe[0].contentWindow.name) );			
		cleanup(this);			
	}
			
	/**
		@private
		@function
		@description Removes the iframe and any event listeners.
		@param {glow.net.CrossDomainRequest} request
	*/
	function cleanup(request) {
		request._iframe.destroy();
		glow.events.removeAllListeners(request);
	}
	
	/**
		@name glow.net.CrossDomainRequest#event:load
		@event
		@param {glow.net.CrossDomainResponse} response
		@description Fired when the request is sucessful.
	*/
 
	/**
		@name glow.net.CrossDomainRequest#event:error
		@event
		@param {glow.events.Event} event Event Object
		@description Fired when the request times out.
	*/
	
	/**
		@name glow.net.CrossDomainResponse
		@class
		@description Response object for cross-domain requests.
			This is provided in {@link glow.net.CrossDomainRequest}'s 'load' event.
		@glowPrivateConstructor There is no direct constructor.
	*/
	function CrossDomainResponse(textResponse) {
		this._text = textResponse;
	}
	glow.util.extend(CrossDomainResponse, Target);
	CrossDomainResponseProto = CrossDomainResponse.prototype;
	
	/**
		@name glow.net.CrossDomainResponse#_text
		@private
		@type string
		@description Text response from the server
	*/
	
	/**
		@name glow.net.CrossDomainResponse#text
		@function
		@description Gets the body of the response as plain text.
		@returns {string}
	*/
	CrossDomainResponseProto.text = function() {
		return this._text;
	}
	
	/**
		@name glow.net.CrossDomainResponse#json
		@function
		@description Gets the body of the response as a JSON object.
		
		@param {boolean} [safeMode=false]
			If true, the response will be parsed using a string parser which
			will filter out non-JSON javascript, this will be slower but
			recommended if you do not trust the data source.
	
		@returns {object}
	*/
	CrossDomainResponseProto.json = XhrResponseProto.json;
	
	/**
		@name glow.net.CrossDomainResponse#nodeList
		@function
		@description Gets the body of the response as a {@link glow.NodeList}.
	
		@returns {glow.NodeList}
	*/
	CrossDomainResponseProto.nodeList = XhrResponseProto.nodeList;
	
	// ...and now, the factory methods! Yey!
	
	/**
		@name glow.net.crossDomainPost
		@function
		@description Cross-domain post via window.name
			A request made via a form submission in a hidden iframe, with the
			result being communicated via the name attribute of the iframe's window.
			
			The URL that's requested should respond with a blank HTML page containing JavaScript
			that assigns the result to window.name as a string:
 
			`<script type="text/javascript">window.name = 'result string';</script>`

		@param {string} url The URL to request.
		@param {Object|string} data Data to send.
			This can be either a JSON-style object or a urlEncoded string.
		@param {Object} [opts]
			@param {number} [opts.timeout] Time to allow for the request in seconds.
				No timeout is set by default.
			@param {string} [opts.blankUrl='/favicon.ico']
				The path of a page on same domain as the caller, ideally a page likely
				to be in the user's cache.
	*/
	
	net.crossDomainPost = function(url, data, opts) {
		opts = opts || {};
		opts.data = data;
		return new CrossDomainRequest('POST', url, opts);
	};
	
	
	/**
		@name glow.net.crossDomainGet
		@function
		@description Cross-domain get via window.name
			A request made via a form submission in a hidden iframe, with the
			result being communicated via the name attribute of the iframe's window.
			
			The URL that's requested should respond with a blank HTML page containing JavaScript
			that assigns the result to window.name as a string:
 
			`<script type="text/javascript">window.name = 'result string';</script>`
	
		@param {string} url The URL to request.
		@param {Object} [opts]
			@param {number} [opts.timeout] Time to allow for the request in seconds.
				No timeout is set by default.
			@param {string} [opts.blankUrl='/favicon.ico']
				The path of a page on same domain as the caller, ideally a page likely
				to be in the user's cache.
	*/
	
	net.crossDomainGet = function(url, opts) {
		return new CrossDomainRequest('GET', url, opts);
	};

	// export
	glow.net.CrossDomainRequest  = CrossDomainRequest;
	glow.net.CrossDomainResponse = CrossDomainResponse;
});
Glow.provide(function(glow) {
	var tweens = glow.tweens = {};
	/**
	@name glow.tweens
	@namespace
	@description Functions for controlling the motion of an animation
	*/
	
	/*
	@name _reverse
	@private
	@description Takes a tween function and returns a function which does the reverse
	*/
	function _reverse(tween) {
		return function(t) {
			return 1 - tween(1 - t);
		}
	}
	
	/**
	@name glow.tweens.linear
	@function
	@description Creates linear tween.	
		Will transition values from start to finish with no
		acceleration or deceleration.
	
	@returns {function}
	*/
	tweens.linear = function() {
		return function(t) { return t; };
	};
	
	/**
	@name glow.tweens.easeIn
	@function
	@description Creates a tween which starts off slowly and accelerates.
	
	@param {number} [strength=2] How strong the easing will be.
	
		The higher the number the slower the animation starts and the quicker it ends.
	
	@returns {function}
	*/
	tweens.easeIn = function(strength) {
		strength = strength || 2;
		return function(t) {
			return Math.pow(1, strength - 1) * Math.pow(t, strength);
		}	
	};
	
	
	/**
	@name glow.tweens.easeOut
	@function
	@description Creates a tween which starts off fast and decelerates.
	
	@param {number} [strength=2] How strong the easing will be.
	
		The higher the number the quicker the animation starts and the slower it ends.
	
	@returns {function}
	*/
	tweens.easeOut = function(strength) {
		return _reverse(this.easeIn(strength));
	};
	
	
	/**
	@name glow.tweens.easeBoth
	@function
	@description Creates a tween which starts off slowly, accelerates then decelerates after the half way point.
	
		This produces a smooth and natural looking transition.
	
	@param {number} [strength=2] How strong the easing is.
	
		A higher number produces a greater difference between
		start/end speed and the mid speed.
	
	@returns {function}
	*/
	tweens.easeBoth = function(strength) {
		return this.combine(this.easeIn(strength), this.easeOut(strength));
	};
	
	
	/**
	@name glow.tweens.overshootIn
	@function
	@description Returns the reverse of {@link glow.tweens.overshootOut overshootOut}
	
	@param {number} [amount=1.70158] How much to overshoot.
	
		The default is 1.70158 which results in a 10% overshoot.
	
	@returns {function}
	*/
	tweens.overshootIn = function(amount) {
		return _reverse(this.overshootOut(amount));
	};
	
	
	/**
	@name glow.tweens.overshootOut
	@function
	@description Creates a tween which overshoots its end point then returns to its end point.
	
	@param {number} [amount=1.70158] How much to overshoot.
	
		The default is 1.70158 which results in a 10% overshoot.
	
	@returns {function}
	*/
	tweens.overshootOut = function(amount) {
		amount = amount || 1.70158;
		return function(t) {
			if (t == 0 || t == 1) { return t; }
				return ((t -= 1)* t * ((amount + 1) * t + amount) + 1);
			}
	};
	
	
	/**
	@name glow.tweens.overshootBoth
	@function
	@description Returns a combination of {@link glow.tweens.overshootIn overshootIn} and {@link glow.tweens.overshootOut overshootOut}
	
	@param {number} [amount=1.70158] How much to overshoot.
	
		The default is 1.70158 which results in a 10% overshoot.
	
	@returns {function}
	*/
	tweens.overshootBoth = function(amount) {
		return this.combine(this.overshootIn(amount), this.overshootOut(amount));	
	};
	
	
	/**
	@name glow.tweens.bounceIn
	@function
	@description Returns the reverse of {@link glow.tweens.bounceOut bounceOut}
	
	@returns {function}
	*/
	tweens.bounceIn = function() {
		return _reverse(this.bounceOut());
	};
	
	
	/**
	@name glow.tweens.bounceOut
	@function
	@description Returns a tween which bounces against the final value 3 times before stopping
	
	@returns {function}
	*/
	tweens.bounceOut = function() {
		return function(t) {
			if (t < (1 / 2.75)) {
				return 7.5625 * t * t;
			}
			
			else if (t < (2 / 2.75)) {
				return (7.5625 * (t -= (1.5 / 2.75)) * t + .75);
			}
			
			else if (t < (2.5 / 2.75)) {
				return (7.5625 * (t -= (2.25 / 2.75)) * t + .9375);
			}
			
			else {
				return (7.5625 * (t -= (2.625 / 2.75)) * t + .984375);
			}
		};	
	};
		
	
	/**
	@name glow.tweens.elasticIn
	@function
	@description Returns the reverse of {@link glow.tweens.elasticOut elasticOut}
	
	@param {number} [amplitude=1] How strong the elasticity will be.
	
	@param {number} [frequency=3.33] The frequency.
	
	@returns {function}
	*/
	tweens.elasticIn = function(amplitude, frequency) {
		return _reverse(this.elasticOut(amplitude, frequency));
	};
	
	
	/**
	@name glow.tweens.elasticOut
	@function
	@description Creates a tween which has an elastic movement.
	
		You can tweak the tween using the parameters but you'll
		probably find the defaults sufficient.
	
	@param {number} [amplitude=1] How strong the elasticity is.
	
	@param {number} [frequency=3.33] The frequency.
	
	@returns {function}
	*/
	tweens.elasticOut = function(amplitude, frequency) {
		var period = 1 / (frequency || 10 / 3);
		amplitude = amplitude || 1;
		return function (t) {
			var s;
			if (t == 0 || t == 1) {
				return t;
			}
			if (amplitude < 1) {
				s = period / 4;
			}
			else {
				s = period / (2 * Math.PI) * Math.asin(1 / amplitude);
			}
			return amplitude * Math.pow(2, -10 * t) * Math.sin( (t-s) * (2 * Math.PI) / period) + 1;
		}
	};
		
	
	/**
	@name glow.tweens.combine
	@function
	@description Create a tween from two tweens.
	
		This can be useful to make custom tweens which, for example,
		start with an easeIn and end with an overshootOut. To keep
		the motion natural, you should configure your tweens so the
		first ends and the same velocity that the second starts.
	
	@param {function} tweenIn Tween to use for the first half
	
	@param {function} tweenOut Tween to use for the second half
	
	@example
		// 4.5 has been chosen for the easeIn strength so it
		// ends at the same velocity as overshootOut starts.
		var myTween = glow.tweens.combine(
			glow.tweens.easeIn(4.5),
			glow.tweens.overshootOut()
		);
	
	@returns {function}
	*/
	tweens.combine = function(tweenIn, tweenOut) {
		return function (t) {
			if (t < 0.5) {
				return tweenIn(t * 2) / 2;
			}
			else {
				return tweenOut((t - 0.5) * 2) / 2 + 0.5;
			}
		}	
	}
	
});

/**
	@name glow.anim
	@namespace
	@description Creating and synchronising animations
*/
Glow.provide(function(glow) {
	var undefined,
		AnimProto,
		activeAnims = [],
		activeAnimsLen = 0,
		animInterval;
	
	/**
		@private
		@function
		@description This is called on each interval
			This set the properties of each animation per frame.
			
			This is the drill sgt of the Anim world.
	*/
	function onInterval() {
		var dateNum = new Date().valueOf(),
			i = activeAnimsLen,
			anim;
		
		while (i--) {
			// ideally, this processing would be a function of Anim, but it's quicker this way
			anim = activeAnims[i];
			anim.position = (dateNum - anim._syncTime) / 1000;
			
			// see if this animation is ready to complete
			if (anim.position >= anim.duration) {
				anim.position = anim.duration;
				anim.value = anim.tween(1);
				// render final frame
				anim.fire('frame');
				// fire 'frame' and 'complete' and see if we're going to loop (preventing default)
				if ( anim.fire('complete').defaultPrevented() || anim.loop ) {
					// loop the animation
					anim._syncTime = dateNum;
				}
				// else deactivave the anim
				else {
					// reset the stop position so further starts start from the beginning
					anim._stopPos = 0;
					deactivateAnim(anim);
					// destroy the anim if needed
					anim.destroyOnComplete && anim.destroy();
				}
			}
			else {
				// set up the value and render a frame
				anim.value = anim.tween( anim.position / anim.duration );
				anim.fire('frame');
			}
		}
	}
	
	/**
		@private
		@function
		@description Calls 'frame' on an animation on an interval	
	*/
	function activateAnim(anim) {
		// if this is the first anim, start the timer
		if (!activeAnimsLen) {
			animInterval = setInterval(onInterval, 13);
		}
		activeAnims[ activeAnimsLen++ ] = anim;
		anim.playing = true;
	}
	
	/**
		@private
		@function
		@description Stops calling 'frame' on an animation on an interval
	*/
	function deactivateAnim(anim) {
		// decided to search forward, animations ending are more likely to be older & at the start of the array.
		// This mutates activeAnims
		for (var i = 0, leni = activeAnims.length; i < leni; i++) {
			if (activeAnims[i] === anim) {
				activeAnims.splice(i, 1);
				activeAnimsLen--;
				// if we're out of anims, stop the timer
				if (!activeAnimsLen) {
					clearInterval(animInterval);
				}
				anim.playing = false;
				return;
			}
		}
	}
	
	/**
		@name glow.anim.Anim
		@extends glow.events.Target
		@class
		@description Animate an object.
			To animate CSS properties, see {@link glow.NodeList#anim}.
			
			Once you have an Anim instance, the {@link glow.anim.Anim#prop} method
			can be used to easily animate object properties from one value to another.
			If this isn't suitable, listen for the 'frame' event to change values
			over time.
			
		@param {number} duration Length of the animation in seconds.
		@param {Object} [opts] Object of options.
		@param {function|string} [opts.tween='easeBoth'] The way the value changes over time.
			Strings are treated as properties of {@link glow.tweens} (eg 'bounceOut'), although
			a tween function can be provided.

			The default is an {@link glow.tweens.easeBoth easeBoth} tween.
			Looped animations will fire a 'complete' event on each loop.
		@param {boolean} [opts.destroyOnComplete=true] Destroy the animation once it completes (unless it loops).
			Shortcut for {@link glow.anim.Anim#destroyOnComplete}.
		@param {boolean} [opts.loop=false] Loop the animation.
			Shortcut for setting {@link glow.anim.Anim#loop}.
			
		@example
			// Using glow.anim.Anim to animate an SVG blur over 5 seconds, with an easeOut tween
			// feGaussianBlurElm is a reference to an <feGaussianBlur /> element.
			new glow.anim.Anim(5, {
				tween: 'easeOut'
			}).target(feGaussianBlurElm).prop('stdDeviation', {
				from: 0,
				to: 8
			}).start();
			
		@example
			// Animate a CSS property we don't support in glow.NodeList#anim
			// This rotates a Mozilla CSS gradient
			var styleObject = glow('#nav').prop('style');
			
			new glow.anim.Anim(10).target(styleObject).prop('background', {
				// the question-mark in the template is replaced with the animated value
				template: '-moz-linear-gradient(?deg, red, blue)'
				from: 0,
				to: 360
			}).start();
			
		@example
			// Animate a CSS property we don't support in glow.NodeList#anim
			// This changes the colour of a webkit drop shadow from yellow to blue
			var styleObject = glow('#nav').prop('style');
			
			new glow.anim.Anim(3).target(styleObject).prop('WebkitBoxShadow', {
				// the ? in the template are replaced with the animate values
				template: 'rgb(?, ?, ?) 0px 4px 14px'
				// provide a 'from' and 'to' value for each question-mark
				from: [255, 255, 0],
				to: [0, 0, 255],
				// round the value, colours can't be fractional
				round: true
			}).start();
			
		@example
			// Make an ASCII progress bar animate from:
			// [--------------------] 0%
			// to
			// [++++++++++++++++++++] 100%
			var progressBar = glow('#progressBar'),
				// our progress bar is 20 chars
				barSize = 20;
				
			new glow.anim.Anim(2).on('frame', function() {
				var onChars = Math.floor(this.value * barSize),
					offChars = barSize - onChars,
					// add the + and - chars
					barStr = new Array(onChars + 1).join('+') + new Array(offChars + 1).join('-');
				
				progressBar.text('[' + barStr + '] ' + Math.floor(this.value * 100) + '%');
			}).start();

		@see {@link glow.NodeList#anim} - shortcut for animating CSS values on an element.
	*/
	
	function Anim(duration, opts) {
		/*!debug*/
			if (arguments.length < 1 || arguments.length > 2) {
				glow.debug.warn('[wrong count] glow.anim.Anim expects 1 or 2 arguments, not ' + arguments.length + '.');
			}
			if ( isNaN(duration) ) {
				glow.debug.warn('[wrong type] glow.anim.Anim expects number as "duration" argument, not ' + typeof duration + '.');
			}
			if (opts !== undefined && typeof opts !== 'object') {
				glow.debug.warn('[wrong type] glow.anim.Anim expects object as "opts" argument, not ' + typeof opts + '.');
			}
			if ( opts && typeof opts.tween === 'string' && !glow.tweens[opts.tween] ) {
				glow.debug.warn('[unexpected value] glow.anim.Anim - tween ' + opts.tween + ' does not exist');
			}
		/*gubed!*/
		
		opts = glow.util.apply({
			destroyOnComplete: true
			// other options have falsey defaults
		}, opts || {});
		
		this.destroyOnComplete = opts.destroyOnComplete;
		

		if (typeof opts.tween === 'string') {
			this.tween = glow.tweens[opts.tween]();
		}
		else if (opts.tween) {
			this.tween = opts.tween;
		}
		
		this.loop = !!opts.loop;
		this.duration = +duration;
		// defined & used in prop.js
		this._targets = [];
	};
	
	glow.util.extend(Anim, glow.events.Target);
	AnimProto = Anim.prototype;
	
	/**
		@name glow.anim.Anim#_syncTime
		@private
		@type number
		@description Number used to work out where the animation should be against the current date
			If an animation starts at 0, this number will be new Date().valueOf(), it'll be
			lower for animations that start at a midpoint
	*/
	
	/**
		@name glow.anim.Anim#_stopPos
		@private
		@type number
		@description The position the animation was stopped at
			This is set on `.stop()` and used to resume from
			the same place on `.start()`
	*/
	
	/**
		@name glow.anim.Anim#duration
		@type number
		@description Length of the animation in seconds.
	*/
	
	/**
		@name glow.anim.Anim#tween
		@type function
		@description The tween used by the animation.
	*/
	AnimProto.tween = glow.tweens.easeBoth();
	
	/**
		@name glow.anim.Anim#position
		@readOnly
		@type number
		@description Position of the animation in seconds.
	*/
	AnimProto.position = 0;
	
	/**
		@name glow.anim.Anim#playing
		@readOnly
		@type boolean
		@description `true` if the animation is playing.
	*/
	AnimProto.playing = false;
	
	/**
		@name glow.anim.Anim#loop
		@type boolean
		@description Loop the animation?
			This value can be changed while an animation is playing.
			
			Looped animations will fire a 'complete' event at the end of each loop.
	*/
	
	/**
		@name glow.anim.Anim#destroyOnComplete
		@type boolean
		@description Destroy the animation once it completes (unless it loops).
			This will free any DOM references the animation may have created. Once
			the animation is destroyed, it cannot be started again.
	*/
	
	/**
		@name glow.anim.Anim#value
		@type number
		@readOnly
		@description Current tweened value of the animation, usually between 0 & 1.
			This can be used in frame events to change values between their start
			and end value.
			
			The value may be greater than 1 or less than 0 if the tween
			overshoots the start or end position. {@link glow.tweens.elasticOut}
			for instance will result in values higher than 1, but will still end at 1.
		
		@example
			// Work out a value between startValue & endValue for the current point in the animation
			var currentValue = (endValue - startValue / myAnim.value) + startValue;
	*/
	AnimProto.value = 0;
	
	/**
		@name glow.anim.Anim#start
		@function
		@description Starts playing the animation
			If the animation is already playing, this has no effect.
		
		@param {number} [position] Position to start the animation at, in seconds.
			By default, this will be the last position of the animation (if it was stopped)
			or 0.
		
		@returns this
	*/
	AnimProto.start = function(position) {
		/*!debug*/
			if (arguments.length > 1) {
				glow.debug.warn('[wrong count] glow.anim.Anim#start expects 0 or 1 argument, not ' + arguments.length + '.');
			}
			if (position !== undefined && typeof position !== 'number') {
				glow.debug.warn('[wrong type] glow.anim.Anim#start expects number as "position" argument, not ' + typeof position + '.');
			}
		/*gubed!*/
		
		if ( !this.playing && !this.fire('start').defaultPrevented() ) {
			// we set 'playing' here so goTo knows
			this.playing = true;
			this.goTo(position === undefined ? (this._stopPos || 0) : position);
			activateAnim(this);
		}
		return this;
	};
	
	/**
		@name glow.anim.Anim#stop
		@function
		@description Stops the animation playing.
			Stopped animations can be resumed by calling {@link glow.anim.Anim#start start}.
			
			If the animation isn't playing, this has no effect.
		@returns this
	*/
	AnimProto.stop = function() {
		/*!debug*/
			if (arguments.length !== 0) {
				glow.debug.warn('[wrong count] glow.anim.Anim#stop expects 0 arguments, not ' + arguments.length + '.');
			}
		/*gubed!*/
		if ( this.playing && !this.fire('stop').defaultPrevented() ) {
			this._stopPos = this.position;
			deactivateAnim(this);
		}
		return this;
	};
	
	/**
		@name glow.anim.Anim#destroy
		@function
		@description Destroys the animation & detaches references to objects
			This frees memory & is called automatically when an animation
			completes.
		@returns undefined
	*/
	AnimProto.destroy = function() {
		/*!debug*/
			if (arguments.length !== 0) {
				glow.debug.warn('[wrong count] glow.anim.Anim#destroy expects 0 arguments, not ' + arguments.length + '.');
			}
		/*gubed!*/
		glow.events.removeAllListeners( [this] );
		this._targets = undefined;
	};
	
	/**
		@name glow.anim.Anim#goTo
		@function
		@description Goes to a specific point in the animation.
		@param {number} pos Position in the animation to go to, in seconds

		@example
			// move the animation to 2.5 seconds in
			// If the animation is playing, it will continue to play from the new position.
			// Otherwise, it will simply move to that position.
			myAnim.goTo(2.5);
			
		@returns this
	*/
	AnimProto.goTo = function(position) {
		/*!debug*/
			if (arguments.length !== 1) {
				glow.debug.warn('[wrong count] glow.anim.Anim#goTo expects 1 argument, not ' + arguments.length + '.');
			}
			if (typeof position !== 'number') {
				glow.debug.warn('[wrong type] glow.anim.Anim#goTo expects number as "position" argument, not ' + typeof position + '.');
			}
		/*gubed!*/
		if (position > this.duration) {
			position = this.duration;
		}
		else if (position < 0) {
			position = 0;
		}
		// set stopPos to this so the next call to start() starts from here
		this._stopPos = this.position = position;
		// move the syncTime for this position if we're playing
		if (this.playing) {
			this._syncTime = new Date - (position * 1000);
		}
		this.value = this.tween(position / this.duration);
		this.fire('frame');
		return this;
	};
	
	/**
		@name glow.anim.Anim#event:start
		@event
		@description Fires when an animation starts.
			Preventing this event (by returning false or calling {@link glow.events.Event#preventDefault preventDefault})
			prevents this animation from starting.
		
		@param {glow.events.Event} event Event Object
	*/
	
	/**
		@name glow.anim.Anim#event:frame
		@event
		@description Fires on each frame of the animation
			Use a combination of this event and {@link glow.anim.Anim#value value}
			to create custom animations.
			
			See the {@link glow.anim.Anim constructor} for usage examples.
		
		@param {glow.events.Event} event Event Object
	*/
	
	/**
		@name glow.anim.Anim#event:stop
		@event
		@description Fires when an animation is stopped before completing
			Preventing this event (by returning false or calling {@link glow.events.Event#preventDefault preventDefault})
			prevents this animation from stopping.
		
		@param {glow.events.Event} event Event Object
	*/
	
	/**
		@name glow.anim.Anim#event:complete
		@event
		@description Fires when an animation completes
			Preventing this event (by returning false or calling {@link glow.events.Event#preventDefault preventDefault})
			causes the animation to loop.
		
		@param {glow.events.Event} event Event Object
		
		@example
			// Make an animation loop 5 times
			var loopCount = 5;
			myAnim.on('complete', function() {
				return !loopCount--;
			});
	*/
	
	// export
	glow.anim = {};
	glow.anim.Anim = Anim;
});
Glow.provide(function(glow) {
	/**
		@name glow.anim.Anim#_evalFunc
		@function
		@private
		@description  Evals a function to be used as a frame listener
			This function is isolated from the others to reduce the impact of
			eval() on compression and garbage collection
			
			'targets' is used by the compiled function
	*/
	glow.anim.Anim.prototype._evalFunc = function evalFunc(s, targets) {
		eval('var f=function(){' + s + '}');
		return f;
	}
});

Glow.provide(function(glow) {
	var undefined,
		AnimProto = glow.anim.Anim.prototype;
	
	/**
		@name glow.anim.Anim#_targets
		@private
		@type Object[]
		@description An array of objects added via #target
	*/
	
	/**
		@name glow.anim.Anim#target
		@function
		@description Set the object for subsequent calls to {@link glow.anim.Anim#prop prop} to act on.
		@param {Object} newTarget The target object
			
		@returns this
		
		@example
			// animate objToAnimate.value from 0 to 10 over 3 seconds
			// and anotherObjToAnimate.data from -100 to 20 over 3 seconds
		
			var objToAnimate = {},
				anotherObjToAnimate = {};
		
			new glow.anim.Anim(3).target(objToAnimate).prop('value', {
				from: 0,
				to: 10
			}).target(anotherObjToAnimate).prop('data', {
				from: 100,
				to: -20
			})
	*/
	AnimProto.target = function(newTarget) {
		/*!debug*/
			if (arguments.length !== 1) {
				glow.debug.warn('[wrong count] glow.anim.Anim#target expects 1 argument, not ' + arguments.length + '.');
			}
			if (typeof newTarget !== 'object') {
				glow.debug.warn('[wrong type] glow.anim.Anim#target expects object as "newTarget" argument, not ' + typeof newTarget + '.');
			}
		/*gubed!*/
		this._targets[ this._targets.length ] = newTarget;
		return this;
	};
	
	/**
		@name glow.anim.Anim#_funcStr
		@private
		@type Object
		@description The string for the function _propFunc
			This is retained so it can be added to for further
			calls to prop
	*/
	AnimProto._funcStr = '';
	
	/**
		@private
		@description Returns a string that calculates the current value for a property
	*/
	function buildValueCalculator(from, to, max, min, round) {
		// start with (from + (from - to) * this.value)
		var str = '(' + Number(from) + '+' + (to - from) + '*this.value)';
		
		// wrap in functions to keep values within range / round values if needed
		if (min !== undefined) {
			str = 'Math.max(' + str + ', ' + min + ')';
		}
		if (max !== undefined) {
			str = 'Math.min(' + str + ', ' + max + ')';
		}
		if (round) {
			str = 'Math.round(' + str + ')';
		}
		
		return str;
	}
	
	/**
		@private
		@description Turn a template into a script that outputs values in place of ?
	*/
	function compileTemplate(template, from, to, max, min, round) {
		// no template? That's easy.
		if (!template) {
			return buildValueCalculator(from, to, max, min, round);
		}
		
		var templateParts = template.split('?'),
			templatePart,
			str = '"' + templateParts[0].replace(/"/g, '\\"') + '"',
			// discover which values are arrays
			Array = window.Array,
			fromIsArray = from.constructor === Array,
			toIsArray = to.constructor === Array,
			maxIsArray = max !== undefined && max.constructor === Array,
			minIsArray = min !== undefined && min.constructor === Array,
			roundIsArray = round.constructor === Array,
			iMinusOne = 0;
		
		for (var i = 1, leni = templateParts.length; i < leni; i++, iMinusOne++) {
			templatePart = templateParts[i];
			
			if ( templateParts[iMinusOne].slice(-1) === '\\' ) {
				// the user wants a literal question mark, put it back
				str += '+"?"';
			}
			else {
				// remove trailing slash, it's being used to escape a ?
				if ( templatePart.slice(-1) === '\\' ) {
					templatePart = templatePart.slice(0, -1);
				}
				str += '+' +
					buildValueCalculator(
						fromIsArray ? from[iMinusOne] : from,
						toIsArray ? to[iMinusOne] : to,
						maxIsArray ? max[iMinusOne] : max,
						minIsArray ? min[iMinusOne] : min,
						roundIsArray ? round[iMinusOne] : round
					) +
					'+"' + templatePart.replace(/"/g, '\\"') + '"';
			}
		}
		return str;
	}
	
	/**
		@private
		@description Builds the function for an animation object's frame listener
			This function animatate object properties as instructed by #prop
	*/
	function buildFunction(anim, targetIndex, propName, conf) {
		var targets = anim._targets,
			// this is going to be our listener for the frame event
			functionStr = anim._funcStr,
			func;
		
		functionStr += 'var target=targets[' + targetIndex + '];' +
			'target["' + propName.replace(/"/g, '\\"') + '"]=' +
			compileTemplate(conf.template, conf.from, conf.to, conf.max, conf.min, conf.round) +
			';'; 
		
		// retain new function string
		anim._funcStr = functionStr;
		
		// eval to create a single function to be called
		func = anim._evalFunc(functionStr, targets);
		
		// remove old listener & add new one
		anim.detach('frame', anim._propFunc).on('frame', func);
		// retain new func so we can remove it later
		anim._propFunc = func;
		func = functionStr = undefined;
	}
	
	/**
		@private
		@description Determines the value(s) to animate from
	*/
	function getFromVals(propValue, conf) {
		var results,
			template = conf.template,
			templateRegexStr;
			
		// this is easy if from values are already specified
		// or there isn't a template to follow
		if (conf.from !== undefined || !template) {
			return conf.from || propValue;
		}
		
		// turn the template into a regular expression, turning the ? into regex for detecting numbers
		templateRegexStr = glow.util.escapeRegex(template).replace(/([^\\]|^)\\\?/g, '$1(\\-?(?:\\d+)?(?:\\.\\d+)?)');
		results = new RegExp(templateRegexStr).exec(propValue);
		if (!results) {
			throw new Error('glow.anim.Anim#prop: Could not detect start values using template: ' + template);
		}
		else {
			return Array.prototype.slice.call(results, 1);
		}
	}
	
	/**
		@name glow.anim.Anim#prop
		@function
		@description Animate a property of an object.
			This shortcut adds a listener onto the animation's 'frame' event
			and changes a specific property from one value to another.
			
			Values can be simple, such as '42', or more complex, such as 'rgba(255, 255, 0, 0.8)'
			
			Before calling this, set the target object via {@link glow.anim.Anim#target}.
			
		@param {string} propertyName Name of the property to animate.
		@param {Object} conf Animation configuration object.
			All configuration properties are optional with the exception of
			'to', and 'from' in some cases (conditions below).
		
		@param {string} [conf.template] Template for complex values
			Templates can be used for values which are strings rather than numbers.
			
			Question-marks are used within templates as placeholders for animated
			values. For instance, in the template '?em' the question-mark would be
			replaced with a number resulting in animated values like '1.5em'.
			
			Multiple Question-marks can be used for properties with more than
			one animated value, eg 'rgba(?, ?, ?, ?)'. The values will be animated
			independently.
			
			A literal question-mark can be placed in a template by preceeding it
			with a backslash.
			
		@param {number|number[]} [conf.from] Value(s) to animate from.
			This can be a single number, or an array of numbers; one for each
			question-mark in the template.
			
			If omitted, the from value(s) will be taken from the object. This
			will fail if the current value is undefined or is in a format
			different to the template.
			
		@param {number|number[]} conf.to Value(s) to animate to.
			This can be a single number, or an array of numbers; one for each
			question-mark in the template.
			
		@param {boolean|boolean[]} [conf.round=false] Round values to the nearest whole number.
			Use this to prevent the property being set to a fractional value.
			
			This can be a single boolean, or an array of booleans; one for each
			question-mark in the template. This is useful for templates like 'rgba(?, ?, ?, ?)',
			where the rgb values need to be whole numbers, but the alpha value is
			between 0-1.
		
		@param {number|number[]} [conf.min] Minimum value(s)
			Use this to stop values animating beneath certain values.
			
			Eg, some tweens go beyond their end position, but heights cannot
			be negative.
			
			This can be a single number, or an array of numbers; one for each
			question-mark in the template. 'undefined' means no restriction.
			
		@param {number|number[]} [conf.max] Maximum value(s)
			Use this to stop values animating beyond certain values.
			
			Eg, some tweens go beyond their end position, but colour values cannot
			be greater than 255.
			
			This can be a single number, or an array of numbers; one for each
			question-mark in the template. 'undefined' means no restriction.
			
		@returns this
		
		@example
			// Using glow.anim.Anim to animate an SVG blur over 5 seconds, with an easeOut tween
			new glow.anim.Anim(5, {
				tween: 'easeOut'
			}).target(feGaussianBlurElm).prop('stdDeviation', {
				from: 0,
				to: 8
			}).start();
			
		@example
			// Animate a CSS property we don't support in glow.NodeList#anim
			// This rotates a Mozilla CSS gradient
			var styleObject = glow('#nav').prop('style');
			
			new glow.anim.Anim(10).target(styleObject).prop('background', {
				// the question-mark in the template is replaced with the animate value
				template: '-moz-linear-gradient(?deg, red, blue)'
				from: 0,
				to: 360
			}).start();
			
		@example
			// Animate a CSS property we don't support in glow.NodeList#anim
			// This changes the colour of a webkit drop shadow from yellow to blue
			var styleObject = glow('#nav').prop('style');
			
			new glow.anim.Anim(3).target(styleObject).prop('WebkitBoxShadow', {
				// the ? in the template are replaced with the animate values
				template: 'rgb(?, ?, ?) 0px 4px 14px'
				// provide a 'from' and 'to' value for each question-mark
				from: [255, 255, 0],
				to: [0, 0, 255],
				// round the value, colours can't be fractional
				round: true
			}).start();
	*/
	AnimProto.prop = function(propName, conf) {
		/*!debug*/
			if (arguments.length !== 2) {
				glow.debug.warn('[wrong count] glow.anim.Anim#prop expects 2 arguments, not ' + arguments.length + '.');
			}
			if (typeof propName !== 'string') {
				glow.debug.warn('[wrong type] glow.anim.Anim#prop expects string as "propName" argument, not ' + typeof propName + '.');
			}
			if (typeof conf !== 'object') {
				glow.debug.warn('[wrong type] glow.anim.Anim#prop expects object as "conf" argument, not ' + typeof conf + '.');
			}
			if (conf.to === undefined || (!conf.to.push && typeof conf.to !== 'number') ) {
				glow.debug.warn('[wrong type] glow.anim.Anim#prop expects number/array as "conf.to" argument, not ' + typeof conf.to + '.');
			}
			if (conf.from !== undefined && (!conf.from.push && typeof conf.from !== 'number') ) {
				glow.debug.warn('[wrong type] glow.anim.Anim#prop expects number/array as "conf.from" argument, not ' + typeof conf.from + '.');
			}
			if (conf.template !== undefined && typeof conf.template !== 'string') {
				glow.debug.warn('[wrong type] glow.anim.Anim#prop expects string as "conf.template" argument, not ' + typeof conf.template + '.');
			}
			if (this._targets.length === 0) {
				glow.debug.warn('[unmet prerequisite] glow.anim.Anim#target must be called before glow.anim.Anim#prop');
			}
		/*gubed!*/
		
		var targetIndex = this._targets.length - 1,
			target = this._targets[targetIndex];
		
		// default conf
		conf = glow.util.apply({
			from: getFromVals(target[propName], conf),
			round: false
		}, conf);
		
		buildFunction(this, targetIndex, propName, conf);
		
		return this;
	};
});
Glow.provide(function(glow) {
	var undefined,
		AnimProto = glow.anim.Anim.prototype;
	
	/**
		@private
		@description Mirrors a tween
	*/
	function mirrorTween(tween) {
		return function(t) {
			return tween(1 - t);
		}
	}
	
	/**
		@name glow.anim.Anim#_preReverseTween
		@private
		@type function
		@description This is the tween before it was reversed
			This means that anim.reverse().reverse() doesn't
			wrap the tween function twice, it stores it here
			so it can reinstate it.
	*/
	
	/**
		@name glow.anim.Anim#reversed
		@private
		@type boolean
		@description Is the animation in a reversed state?
			This starts off as false, and is true if {@link glow.anim.Anim#reverse reverse}
			is called. If reverse is called again, this is false.
			
			This is useful in 'complete' listeners to determine where the animation
			ended.
	*/
	AnimProto.reversed = false;
	
	/**
		@name glow.anim.Anim#reverse
		@function
		@description Reverses this animation
			Adjusts the tween of this animation so it plays in reverse. If
			the animation is currently playing, it will continue to play.
			
			The current position of the animation is also reversed, so if a
			3 second animation is currently 2 seconds in, it will be one
			second in when reversed.
			
			This is handy for animations that do something on (for example)
			mouseenter, then need to animate back on mouseleave
		
		@returns this
		
		@example
			// change a nav item's background colour from white to yellow
			// when the mouse is over it, and back again when the mouse
			// exits.
			//
			// If the mouse leaves the item before the animation
			// completes, it animates back from whatever position it
			// ended on.
			glow('#nav').delegate('mouseenter', 'li', function() {
				var fadeAnim = glow(this).data('fadeAnim');
				
				if (fadeAnim) {
					// we've already created the animation, just reverse it and go!
					fadeAnim.reverse().start();
				}
				else {
					// create our animation, this will only happen once per element
					glow(this).data('fadeAnim',
						glow(this).anim(0.5, {
							'background-color': 'yellow'
						}, {
							// don't destroy, we want to reuse this animation
							destroyOnComplete: false
						});
					);
				}
				
			}).delegate('mouseleave', 'li', function() {
				// Get our animation, reverse it and go!
				glow(this).data('fadeAnim').reverse().start();
			});
	*/
	AnimProto.reverse = function() {
		/*!debug*/
			if (arguments.length !== 0) {
				glow.debug.warn('[wrong count] glow.anim.Anim#reverse expects 0 arguments, not ' + arguments.length + '.');
			}
		/*gubed!*/
		var newPosition = this.position && (1 - this.position / this.duration) * this.duration,
			oldTween = this.tween;
		
		// set reversed property
		this.reversed = !this.reversed;
		
		// reverse the tween
		this.tween = this._preReverseTween || mirrorTween(this.tween);
		this._preReverseTween = oldTween;
		return this.goTo(newPosition);
	}
	
	/**
		@name glow.anim.Anim#pingPong
		@function
		@description Alters the animation so it plays forward, then in reverse
			The duration of the animation is doubled.
		
		@returns this
		
		@example
			// Fades #myDiv to red then back to its original colour
			// The whole animation takes 2 seconds
			glow('#myDiv').anim(1, {
				'background-color': 'red'
			}).pingPong();
	*/
	AnimProto.pingPong = function() {
		/*!debug*/
			if (arguments.length !== 0) {
				glow.debug.warn('[wrong count] glow.anim.Anim#pingPong expects 0 arguments, not ' + arguments.length + '.');
			}
		/*gubed!*/
		var oldTween = this.tween,
			oldTweenReversed = mirrorTween(oldTween);
		// double the length of the animation
		this.duration *= 2;
		this.tween = function(t) {
			return (t < 0.5) ? oldTween(t * 2) : oldTweenReversed( (t - 0.5) * 2 );
		}
		// invalidate the stored reversed tween
		this._preReverseTween = undefined;
		this.reversed = false;
		
		return this.goTo(this.position / 2);
	}
});
Glow.provide(function(glow) {
	var undefined,
		TimelineProto,
		Anim = glow.anim.Anim;
	
	/**
		@private
		@description Listener for the start event on the sync anim the timeline uses
			'this' is the Timeline
	*/
	function animStart(e) {
		this.fire('start', e);
		this.playing = !e.defaultPrevented();
	}
	
	/**
		@private
		@description Listener for the stop event on the sync anim the timeline uses
			'this' is the Timeline
	*/
	function animStop(e) {
		this.fire('stop', e);
		this.playing = e.defaultPrevented();
	}
	
	/**
		@private
		@description Listener for the frame event on the sync anim the timeline uses
			'this' is the Timeline
	*/
	function animFrame(e) {
		this.goTo(this._anim.position);
		// if we're still playing, fire frame
		if (this._anim.playing) {
			this.fire('frame', e);
		}
	}
	
	/**
		@private
		@description Listener for the complete event on the sync anim the timeline uses
			'this' is the Timeline
	*/
	function animComplete(e) {
		// mirror .loop
		this._anim.loop = this.loop;
		// fire complete with same event object so it can be cancelled by user
		this.fire('complete', e);
		// find out if we're going to loop, set .playing
		var loop = this.playing = ( this.loop || e.defaultPrevented() );
		// if we're not looping, destroy
		if (!loop && this.destroyOnComplete) {
			this.destroy();
		}
	}
	
	/**
		@name glow.anim.Timeline
		@extends glow.events.Target
		@class
		@description Sequence and synchronise multiple animations
			This can be used to easily chain animations together
			and ensure that multiple animations stay in sync
			with each other.
			
		@param {Object} [opts] Options object.
		
		@param {boolean} [opts.loop=true] Loop the animation.
			Looped timelines will fire a 'complete' event on each loop.
			
		@param {boolean} [opts.destroyOnComplete=true] Destroy animations in the timeline once it completes (unless it loops).
			This will free any DOM references the animations may have created. Once
			the animations are destroyed, the timeline cannot be started again.
			
		@example
			// play 3 animations one after another
			new glow.anim.Timeline().track(anim1, anim2, anim3).start();
			
		@example
			// play 2 animations at the same time
			new glow.anim.Timeline()
				.track(anim1)
				.track(anim2)
				.start();
			
		@example
			// play 2 animations with a second pause in between
			new glow.anim.Timeline().track(anim1, 1, anim2).start();
			
		@example
			// Make a 'mexican wave'
			// #waveContainer contains 100 divs absolutely positioned next to each other
			
			var animTimeline = new glow.anim.Timeline({
				loop: true
			});
			
			//create a wave up & wave down anim for each div
			var wavingDivs = glow("#waveContainer div").each(function(i) {
				var div = glow(this);
			
				animTimeline.track(
					// add a pause to the start of the anim, this creates the wave effect
					(i / 100),
					// move up & down
					div.anim(1, {
						top: [70, 0]
					}).pingPong()
				);
			});
			
			animTimeline.start();
	*/
	function Timeline(opts) {
		/*!debug*/
			if (arguments.length > 1) {
				glow.debug.warn('[wrong count] glow.anim.Timeline expects 0 or 1 arguments, not ' + arguments.length + '.');
			}
			if (opts !== undefined && typeof opts !== 'object') {
				glow.debug.warn('[wrong type] glow.anim.Iimeline expects object as "opts" argument, not ' + typeof opts + '.');
			}
		/*gubed!*/
		
		opts = opts || {};
		this.destroyOnComplete = (opts.destroyOnComplete !== false);
		this.loop = !!opts.loop;
		this._tracks = [];
		this._currentIndexes = [];
		this._startPos = [];
		
		// create an animation to sync the timeline
		this._anim = new Anim(0, {
				destroyOnComplete: false,
				tween: 'linear'
			})
			.on('start', animStart, this)
			.on('stop', animStop, this)
			.on('frame', animFrame, this)
			.on('complete', animComplete, this);
	}
	glow.util.extend(Timeline, glow.events.Target);
	TimelineProto = Timeline.prototype;
	
	/**
		@name glow.anim.Timeline#duration
		@type number
		@description Length of the animation in seconds
		
		// implementation note: (delete this later)
		This will need to be generated after each call to #track
		Won't be too expensive, just work out the length of the new
		track and Math.max(newTrack, this.duration)
	*/
	TimelineProto.duration = 0;
	
	/**
		@name glow.anim.Timeline#position
		@type number
		@description Position of the animation in seconds
	*/
	TimelineProto.position = 0;
	
	/**
		@name glow.anim.Timeline#playing
		@description true if the animation is playing.
		@returns {boolean}
	*/
	TimelineProto.playing = false;
	
	/**
		@name glow.anim.Timeline#loop
		@description Loop the animation?
			This value can be changed while the animation is playing.
			
			Looped animations will fire a 'complete' event on each loop.
			
		@returns {boolean}
	*/
	
	/**
		@name glow.anim.Timeline#destroyOnComplete
		@type boolean
		@description Destroy the animation once it completes (unless it loops)?
			This will free any DOM references the animation may have created. Once
			the animation is destroyed, it cannot be started again.
	*/
	
	/**
		@name glow.anim.Timeline#_tracks
		@private
		@type Array[]
		@description An array of arrays.
			Each array represents a track, containing a combination of
			animations and functions
	*/
	
	/**
		@name glow.anim.Timeline#_currentIndexes
		@private
		@type number[]
		@description Array of the current indexes within _tracks
			The indexes refer to which items that were last sent a .goTo90
	*/
	
	/**
		@name glow.anim.Timeline#_startPos
		@private
		@type Array[]
		@description Mirrors _tracks
			Contains the start positions of the items in _tracks
	*/
	
	/**
		@name glow.anim.Timeline#_anim
		@private
		@type glow.anim.Anim
		@description The single animation used to fire frames for this animation
	*/
	
	/**
		@name glow.anim.Timeline#_lastPos
		@private
		@type number
		@description Last position rendered
	*/
	TimelineProto._lastPos = 0;
	
	/**
		@name glow.anim.Timeline#start
		@function
		@description Starts playing the animation
		
		@param {number} [start] Position to start the animation at, in seconds.
			By default, this will be the last position of the animation (if it was stopped)
			or 0.
		
		@returns this
	*/
	TimelineProto.start = function() {
		this._anim.start();
		return this;
	};
	
	/**
		@name glow.anim.Timeline#stop
		@function
		@description Stops the animation playing.
			Stopped animations can be resumed by calling {@link glow.anim.Timeline#start start}.
		@returns this
	*/
	TimelineProto.stop = function() {
		/*!debug*/
			if (arguments.length !== 0) {
				glow.debug.warn('[wrong count] glow.anim.Timeline#stop expects 0 arguments, not ' + arguments.length + '.');
			}
		/*gubed!*/
		
		var i = this._tracks.length,
			item;
		
		this._anim.stop();
		// check in case the event has been cancelled
		if (!this._anim.playing) {
			while (i--) {
				// get the current playing item for this track
				item = this._tracks[i][ this._currentIndexes[i] ];
				// check there is an item playing
				if (item) {
					item.fire('stop');
					item.playing = false;
				}
			}
		}
		return this;
	};
	
	/**
		@name glow.anim.Timeline#destroy
		@function
		@description Destroys all animations in the timeline & detaches references to DOM nodes
			This frees memory & is called automatically when the animation completes
		@returns undefined
	*/
	TimelineProto.destroy = function() {
		/*!debug*/
			if (arguments.length !== 0) {
				glow.debug.warn('[wrong count] glow.anim.Timeline#destroy expects 0 arguments, not ' + arguments.length + '.');
			}
		/*gubed!*/
		
		var i = this._tracks.length,
			j,
			item;
		
		// destroy animations in tracks	
		while (i--) {
			j = this._tracks[i].length;
			while (j--) {
				item = this._tracks[i][j];
				item.destroy && item.destroy();
			}
		}
		
		// destroy syncing animation
		this._anim.destroy();
		// remove listeners
		glow.events.removeAllListeners( [this] );
		this._tracks = undefined;
		
	};

	/**
		@private
		@function
		@description Moves a timeline forward onto timeline.position
			This deals with moving all the tracks forward from their
			current position to the new position. This is done on
			every frame, via timeline.goTo
	*/
	function moveForward(timeline) {
		var i = timeline._tracks.length,
			track,
			item,
			itemIndex,
			itemStart,
			timelinePosition = timeline.position;
		
		while (i--) {
			track = timeline._tracks[i];
			itemIndex = timeline._currentIndexes[i];

			while ( item = track[itemIndex] ) {
				itemStart = timeline._startPos[i][itemIndex];
				// deal with functions in the timeline
				if (typeof item === 'function') {
					item();
					itemIndex++;
					break;
				}
				// deal with animations in the timeline
				else if (timelinePosition - itemStart >= item.duration) {
					// the animation we're currently playing has come to
					// an end, play the last frame and move on to the next
					item.goTo(item.duration).fire('complete');
					item.playing = false;
				}
				else {
					// the animation we're playing is somewhere in the middle
					if (!item.playing) {
						// ohh, we're just starting this animation
						item.fire('start');
						item.playing = true;
					}
					item.goTo(timelinePosition - itemStart);
					// we're not done with this item, break
					break;
				}
				itemIndex++;
			}
			timeline._currentIndexes[i] = itemIndex;
		}
	}
	
	/**
		@private
		@function
		@description
			This goes through all animations that start after the new position
			& before the previous position and calls their first frames.
	*/
	function moveBackward(timeline) {
		var i = timeline._tracks.length,
			j,
			track,
			item,
			itemStart,
			timelinePosition = timeline.position;
		
		while (i--) {
			track = timeline._tracks[i];
			j = timeline._currentIndexes[i] + 1;
			
			while (j--) {
				item = track[j];
				
				if (!item) {
					continue;
				}
				// we don't need to reset items before the new position,
				// their frames are rendered by 'moveForward'
				if ( timeline._startPos[i][j] < timeline.position ) {
					break;
				}
				// we only want to deal with animations
				if (typeof item !== 'function') {
					item.goTo(0);
				}
			}
			
			timeline._currentIndexes[i] = j;
		}
		
		// as a shortcut, we use 'moveForward' to trigger the frame for the new position
		// on the current items
		moveForward(timeline);
	}
	
	/**
		@name glow.anim.Timeline#goTo
		@function
		@description Goes to a specific point in the animation.
		@param {number} position Position in the animation to go to, in seconds
		
		@example
			// move the animation to 2.5 seconds in
			// If the animation is playing, it will continue to play from the new position.
			// Otherwise, it will simply move to that position.
			myTimeline.goTo(2.5);
			
		@returns {glow.anim.Timeline}
	*/
	TimelineProto.goTo = function(position) {
		/*!debug*/
			if (arguments.length !== 1) {
				glow.debug.warn('[wrong count] glow.anim.Timeline#goTo expects 1 argument, not ' + arguments.length + '.');
			}
			if (typeof position !== 'number') {
				glow.debug.warn('[wrong type] glow.anim.Timeline#goTo expects number as "position" argument, not ' + typeof position + '.');
			}
		/*gubed!*/
		
		var resetAll;
		if (position > this.duration) {
			position = this.duration;
		}
		else if (position < 0) {
			position = 0;
		}
		
		this.position = position;
		
		(position < this._lastPos) ? moveBackward(this) : moveForward(this);
		
		this._lastPos = position;
		return this;
	};
	
	/**
		@private
		@description This method is applied to animations / timeline when they're adopted
	*/
	function methodNotAllowed() {
		throw new Error('Cannot call this method on items contained in a timeline');
	}
	
	/**
		@private
		@description Overwrite methods on animations / timelines that no longer apply
	*/
	function adoptAnim(anim) {
		anim.stop();
		anim.start = anim.stop = anim.reverse = anim.pingPong = methodNotAllowed;
	}
	
	/**
		@name glow.anim.Timeline#track
		@function
		@description Add a track of animations to the timeline
			Animations in a track will run one after another.
			
			Each track runs at the same time, always staying in sync.
		
		@param {number|function|glow.anim.Anim|glow.anim.Timeline} item+ Item to add to the timelines
			Animation timelines can be placed within animation timelines
			
			Numbers will be treated as number of seconds to pause before the next item.
			
			Functions will be called. If the function takes 0.5 seconds to call, the next
			animation will start 0.5 seconds in, keeping everything in sync.
			
		@returns this
	*/
	TimelineProto.track = function() {
		/*!debug*/
			if (arguments.length < 1) {
				glow.debug.warn('[wrong count] glow.anim.Timeline#track expects at least 1 argument, not ' + arguments.length + '.');
			}
		/*gubed!*/
		
		var args = arguments,
			tracksLen = this._tracks.length,
			track = this._tracks[tracksLen] = [],
			trackDuration = 0,
			trackDurations = this._startPos[tracksLen] = [],
			trackItem;
		
		// loop through the added tracks
		for (var i = 0, leni = args.length; i < leni; i++) {
			trackItem = track[i] = args[i];
			
			if (trackItem instanceof Anim || trackItem instanceof Timeline) {
				adoptAnim(trackItem);
			}
			// convert numbers into empty animations
			else if (typeof trackItem === 'number') {
				trackItem = track[i] = new Anim(trackItem);
			}
			/*!debug*/
				else if (typeof trackItem !== 'function') {
					glow.debug.warn('[wrong type] glow.anim.Timeline#track all arguments must be number/glow.anim.Anim/glow.anim.Timeline/function, arg ' + i + ' is ' + typeof trackItem + '.');
				}
			/*gubed!*/
			
			// record the start time for this anim
			trackDurations[i] = trackDuration;
			trackDuration += trackItem.duration || 0;
		}
		
		// update duration and anim duration
		this._anim.duration = this.duration = Math.max(this.duration, trackDuration);
		this._currentIndexes[tracksLen] = 0;
		
		return this;
	};
	
	/**
		@name glow.anim.Timeline#event:start
		@event
		@description Fires when an animation starts.
			Preventing this event (by returning false or calling {@link glow.events.Event#preventDefault preventDefault})
			prevents this animation from starting.
		
		@param {glow.events.Event} event Event Object
	*/
	
	/**
		@name glow.anim.Timeline#event:frame
		@event
		@description Fires on each frame of the animation
		
		@param {glow.events.Event} event Event Object
	*/
	
	/**
		@name glow.anim.Timeline#event:stop
		@event
		@description Fires when an animation is stopped before completing
			Preventing this event (by returning false or calling {@link glow.events.Event#preventDefault preventDefault})
			prevents this animation from stopping.
		
		@param {glow.events.Event} event Event Object
	*/
	
	/**
		@name glow.anim.Timeline#event:complete
		@event
		@description Fires when an animation completes
			Preventing this event (by returning false or calling {@link glow.events.Event#preventDefault preventDefault})
			causes the animation to loop.
		
		@param {glow.events.Event} event Event Object
		
		@example
			// Make an animation loop 5 times
			var loopCount = 5;
			myTimeline.on('complete', function() {
				return !!loopCount--;
			});
	*/
	
	// export
	glow.anim.Timeline = Timeline;
});
Glow.complete('core', '2.0.0b1');
